const socketIO = require('socket.io');
const ExpedienteUsuariosConectados = require('./models/ExpedientesUsuariosConectados.js');
const Usuario = require('./models/Usuarios.js');
const Despachos = require('./models/Despachos.js');
const OpenAI = require('openai');

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY
});

const connectedUsers = {}; // Guarda los IDs de los sockets

const Chat = require('./models//ChatV2.js');
const { buildFileUri } = require('./config/s3.js');

const systemMessage = {
  role: 'system',
  content: `Quiero que me respondas siempre con markdown.
  `
};

module.exports = (server) => {
  const io = socketIO(server, {
    cors: {
      origin: '*',
      methods: ['GET', 'POST', 'PUT', 'DELETE']
    }
  });

  io.on('connection', (socket) => {
    // let expedienteUsuarioId; // Para almacenar el ID del documento de ExpedienteUsuariosConectados

    // Cuando un usuario se conecta
    socket.on('ExpedienteUsuariosConectadosSocket', async ({ despacho, usuario, expediente }) => {
      try {
        let findExpedienteUsuario = await ExpedienteUsuariosConectados.findOne({ despacho, expediente }).populate('usuarios', 'nombre foto');

        if (!findExpedienteUsuario) {
          await ExpedienteUsuariosConectados.create({ despacho, expediente, usuarios: [usuario], fecha: new Date() });
          findExpedienteUsuario = await ExpedienteUsuariosConectados.findOne({ despacho, expediente }).populate('usuarios', 'nombre foto');
        } else {
          const { usuarios } = findExpedienteUsuario;
          const existingUsuario = usuarios.find((u) => u._id.toString() === usuario);

          if (!existingUsuario) {
            const findUsuarioNuevo = await Usuario.findById(usuario, 'nombre foto');
            if (findUsuarioNuevo) {
              usuarios.push(findUsuarioNuevo);
              await findExpedienteUsuario.save();
            }
          }
        }

        // Actualizar el registro de usuarios conectados
        if (!connectedUsers[expediente]) {
          connectedUsers[expediente] = new Set();
        }
        connectedUsers[expediente].add(socket.id);

        // Construir las URL de las imágenes
        findExpedienteUsuario.usuarios.forEach(usuario => {
          usuario.foto = usuario.foto ? buildFileUri(`usuarios/${usuario.foto}`) : buildFileUri('default/icono_usuario_100x100_04.jpg');
        });

        // Emitir solo a los usuarios en el mismo expediente
        io.to([...connectedUsers[expediente]]).emit('UpdateExpedienteUsuariosConectadosSocket', { findExpedienteUsuario });
      } catch (error) {
        console.error('Error en ExpedienteUsuariosConectadosSocket:', error);
      }
    });

    socket.on('customDisconnect', async ({ despacho, usuario, expediente }) => {
      try {
        let findExpedienteUsuario = await ExpedienteUsuariosConectados.findOne({ despacho, expediente }).populate('usuarios', 'nombre foto');

        if (findExpedienteUsuario) {
          const { usuarios } = findExpedienteUsuario;
          const index = usuarios.findIndex((u) => u._id.toString() === usuario);
          if (index !== -1) {
            usuarios.splice(index, 1);
            await findExpedienteUsuario.save();

            if (usuarios.length === 0) {
              await ExpedienteUsuariosConectados.findByIdAndDelete(findExpedienteUsuario._id);
              findExpedienteUsuario = null;
            }
          }

          // Actualizar el registro de usuarios conectados
          if (connectedUsers[expediente]) {
            connectedUsers[expediente].delete(socket.id);
            if (connectedUsers[expediente].size === 0 || !connectedUsers[expediente]) {
              delete connectedUsers[expediente];
            }
          }

          // Construir las URL de las imágenes
          findExpedienteUsuario?.usuarios?.forEach(usuario => {
            usuario.foto = usuario.foto ? buildFileUri(`usuarios/${usuario.foto}`) : buildFileUri('default/icono_usuario_100x100_04.jpg');
          });

          // Emitir solo a los usuarios en el mismo expediente
          io.to([...connectedUsers[expediente] || []]).emit('UpdateExpedienteUsuariosConectadosSocket', { findExpedienteUsuario });
        }
      } catch (error) {
        console.error('Error en customDisconnect:', error);
      }
    });

    socket.on('disconnect', async () => {
      try {
        // Encontrar el expediente y usuario asociados al socket
        for (const [expediente, sockets] of Object.entries(connectedUsers)) {
          if (sockets.has(socket.id)) {
            const despacho = socket.handshake.query.despacho;
            const usuario = socket.handshake.query.usuario;

            let findExpedienteUsuario = await ExpedienteUsuariosConectados.findOne({ despacho, expediente }).populate('usuarios', 'nombre foto');

            if (findExpedienteUsuario) {
              const { usuarios } = findExpedienteUsuario;
              const index = usuarios.findIndex((u) => u._id.toString() === usuario);
              if (index !== -1) {
                usuarios.splice(index, 1);
                await findExpedienteUsuario.save();

                if (usuarios.length === 0) {
                  await ExpedienteUsuariosConectados.findByIdAndDelete(findExpedienteUsuario._id);
                  findExpedienteUsuario = null;
                }
              }

              // Actualizar el registro de usuarios conectados
              connectedUsers[expediente].delete(socket.id);
              if (connectedUsers[expediente] && connectedUsers[expediente].size === 0) {
                delete connectedUsers[expediente];
              }

              // Construir las URL de las imágenes
              findExpedienteUsuario?.usuarios.forEach(usuario => {
                usuario.foto = usuario.foto ? buildFileUri(`usuarios/${usuario.foto}`) : buildFileUri('default/icono_usuario_100x100_04.jpg');
              });

              // Emitir solo a los usuarios en el mismo expediente
              io.to([...connectedUsers[expediente] || []]).emit('UpdateExpedienteUsuariosConectadosSocket', { findExpedienteUsuario });
            }
            break;
          }
        }
      } catch (error) {
        console.error('Error en disconnect:', error);
      }
    });

    socket.on('chatMessage', async ({ message, despacho, usuario, tipo }) => {
      sendMessage(socket, message, despacho, usuario, tipo);
    });

    socket.on('joinExpedienteRoom', ({ expediente }) => {
      socket.join(expediente);
      console.log(`Socket ${socket.id} joined expediente ${expediente}`);
    });

    socket.on('notifyExpedienteChange', ({ expediente, modulo, message, usuario }) => {
      // console.log(`Notifying change for expediente ${expediente}`);
      io.to(expediente).emit('expedienteChanged', { expediente, modulo, message, usuario });
    });
  });
};

const sendMessage = async (socket, message, despacho, usuario, tipo) => {
  try {
    const isChatValid = await validarContadorChat({ despacho, tipo });

    if (!isChatValid.valido) {
      socket.emit('chatError', { message: isChatValid.mensaje, enlace: isChatValid.enlace });
      return;
    }

    // buscar los últimos mensajes del chat
    const chats = await Chat.find({ despacho, estatus: 'Activo', usuario }).limit(5).sort({ _id: -1 });

    // Solicitar la respuesta del modelo con el historial de conversación
    const response = await openai.chat.completions.create({
      model: tipo,
      messages: [
        systemMessage,
        ...chats.map((m) => ({
          role: m.role,
          content: m.content,
          usuario: m.usuario
        })),
        {
          role: 'user',
          content: message,
          usuario
        }
      ],
      max_tokens: 1000, // Ajusta a un valor menor
      temperature: 0.5 // Ajusta según necesidad, pero no demasiado alto
    });

    const gptMessage = response.choices[0].message.content.trim().replace('```', '').replace('markdown\n', '');

    const newChats = [
      new Chat({
        role: 'user',
        content: message,
        usuario,
        despacho
      }),
      new Chat({
        role: 'assistant',
        content: gptMessage,
        usuario,
        despacho
      })
    ];

    await Chat.insertMany(newChats);

    // Guardar el historial en la base de datos
    // await chatDoc.save();

    // Enviar la respuesta de vuelta al cliente
    socket.emit('chatResponse', {
      reply: gptMessage,
      tipo,
      despacho,
      contador: isChatValid.contador
    });
  } catch (error) {
    console.error('Error en chatMessage:', error);
    socket.emit('chatError', { message: 'Ocurrió un error al procesar la solicitud. Por favor, intenta de nuevo.' });
  }
};

const validarContadorChat = async ({ despacho, tipo }) => {
  const findDespacho = await Despachos.findById(despacho, 'contadorChatGPT');

  if (!findDespacho) {
    return {
      valido: false,
      mensaje: 'No se encontró el despacho',
      tipo: null
    };
  }

  const diario = findDespacho.contadorChatGPT.diario || null;
  const comprado = findDespacho.contadorChatGPT.comprado || null;

  const validarYActualizarContador = async (contadorObj, tipoMensaje, mensajeExito) => {
    if (contadorObj.contador >= contadorObj.limite) {
      return false;
    } else {
      contadorObj.contador += 1;
      findDespacho.contadorChatGPT[tipoMensaje] = contadorObj;
      await findDespacho.save();
      return {
        valido: true,
        mensaje: mensajeExito,
        tipo: tipoMensaje === 'diario' ? 'gpt-3.5-turbo' : 'gpt-4o-mini',
        contador: {
          gpt3: diario.limite - diario.contador,
          gpt4: comprado.limite - comprado.contador
        }
      };
    }
  };

  // Validación inicial
  let resultado;

  if (tipo === 'gpt-4o-mini') {
    resultado = await validarYActualizarContador(comprado, 'comprado', 'Puede enviar mensajes con GPT-4');
    if (!resultado) {
      // Si gpt-4 ha superado el límite, intenta con gpt-3.5
      resultado = await validarYActualizarContador(diario, 'diario', 'Puede enviar mensajes con GPT-3');
    }
  } else if (tipo === 'gpt-3.5-turbo') {
    resultado = await validarYActualizarContador(diario, 'diario', 'Puede enviar mensajes con GPT-3');
    if (!resultado) {
      // Si gpt-3.5 ha superado el límite, intenta con gpt-4
      resultado = await validarYActualizarContador(comprado, 'comprado', 'Puede enviar mensajes con GPT-4');
    }
  }

  if (!resultado) {
    return {
      valido: false,
      mensaje: `Ha superado el límite de mensajes permitidos para ${tipo === 'gpt-3.5-turbo' ? 'GPT-3' : 'GPT-4'}.`,
      tipo: null,
      enlace: '/shop?tipo=GPT-4',
      contador: {
        gpt3: diario.contador,
        gpt4: comprado.contador
      }
    };
  }

  return resultado;
};
