const { Router } = require('express');
const Juzgados = require('../models/Juzgados.js');
const Legislaciones = require('../models/LegislacionesReglamentos.js');
const Estados = require('../models/Estados.js');

const router = Router();
router.get('/juzgados', async (req, res) => {
  try {
    const findEstados = await Estados.find({});

    const arrayContador = await Promise.all(findEstados.map(async (estado) => {
      // Consulta para filtrar los juzgados con los campos no válidos
      const findJuzgados = await Juzgados.find({
        $and: [
          { direccion: { $eq: '' } },
          { telefonos: { $exists: true, $in: [null, []] } }, // Array vacío o null
          { url: { $in: ['', null] } } // Cadena vacía o null
        ],
        estado: estado._id
      })
        .populate('estado') // Asegúrate de que 'estado' esté referenciado en el esquema
        .exec();

      const totalJuzgadosFaltantes = findJuzgados.length;

      return {
        estado: estado.nombre,
        totalJuzgadosFaltantes
      };
    }));

    // Calcula el total de estados faltantes
    const totalEstadosFaltantes = arrayContador.reduce((acc, item) => acc + item.totalJuzgadosFaltantes, 0);

    res.status(200).json({
      totalFaltanes: totalEstadosFaltantes,
      contador: arrayContador
    });
  } catch (error) {
    console.error('Error al contar los estados faltantes:', error);
    res.status(500).json({ error: 'Error al contar los estados faltantes' });
  }
});

router.get('/legislaciones', async (req, res) => {
  try {
    const findEstados = await Estados.find({});

    const arrayContador = await Promise.all(findEstados.map(async (estado) => {
      // Encuentra las legislaciones por estado
      const findLegislaciones = await Legislaciones.find({
        estado: estado._id
      }).populate('estado').exec();

      // Crea un objeto para contar cuántas veces aparece cada enlace
      const enlacesPorLegislacion = findLegislaciones.reduce((acc, legislacion) => {
        const enlace = legislacion.enlace;
        if (!acc[enlace]) {
          acc[enlace] = [];
        }
        acc[enlace].push(legislacion); // En lugar de contar, guardamos las legislaciones completas
        return acc;
      }, {});

      // Filtra los enlaces que se repiten más de una vez
      const enlacesDuplicados = Object.entries(enlacesPorLegislacion)
        .filter(([enlace, legislaciones]) => legislaciones.length > 1) // Solo incluimos si hay más de una legislación
        .map(([enlace, legislaciones]) => ({
          enlace,
          count: legislaciones.length,
          legislaciones // Incluimos todas las legislaciones repetidas
        }));

      const totalLegislacionesFaltantes = findLegislaciones.length;

      return {
        estado: estado.nombre,
        totalLegislacionesFaltantes,
        enlacesDuplicados // Incluimos los enlaces repetidos
      };
    }));

    // Calcula el total de estados faltantes
    const totalEstadosFaltantes = arrayContador.reduce((acc, item) => acc + item.totalLegislacionesFaltantes, 0);

    res.status(200).json({
      totalFaltanes: totalEstadosFaltantes,
      contador: arrayContador
    });
  } catch (error) {
    console.error('Error al contar los estados faltantes:', error);
    res.status(500).json({ error: 'Error al contar los estados faltantes' });
  }
});

module.exports = router;
