/* eslint-disable array-callback-return */
/* eslint-disable indent */
const { default: mongoose } = require('mongoose');
const { formatDateTime, formatDate, formaTime } = require('../config/functionsGlobal.js');
const { uploadFile, buildFileUri } = require('../config/s3.js');
const ChatRoomModel = require('../models/chatRoom.js');
const MensajeModel = require('../models/mensaje.js');

const createMensaje = async (req, res) => {
  const {
    despacho,
    chatRoom = null,
    enviadoPor,
    usuario = null,
    cliente = null,
    mensaje
  } = req.body;

  if (!despacho) {
    return res.status(400).json({ message: 'Falta el despacho' });
  }

  if (!enviadoPor) {
    return res.status(400).json({ message: 'Falta el enviadoPor' });
  }

  if (!usuario && !cliente) {
    return res.status(400).json({ message: 'Falta el usuario o el cliente' });
  }

  if (!mensaje && !req?.files?.archivos) {
    return res.status(400).json({ message: 'Falta el mensaje' });
  }

  try {
    let archivos = req?.files?.archivos ?? [];

    archivos = Array.isArray(archivos) ? archivos : [archivos];

    const uploadPromises = archivos?.map(file => uploadFile(file, 'mensajes'));
    const fileUploads = await Promise.all(uploadPromises);

    const nuevosArchvios = fileUploads?.map((fileUpload, index) => {
      const file = archivos[index];

      return {
        filename: file.name,
        size: file.size,
        url: fileUpload.fileName
      };
    });
    let returnMessage = null;

    if (!chatRoom) {
      const chat = {
        despacho,
        cliente,
        titulo: mensaje
      };

      const newChatRoom = new ChatRoomModel(chat);

      await newChatRoom.save();

      const mensajeCreate = {
        chatRoom: newChatRoom._id,
        enviadoPor,
        usuario,
        cliente,
        mensaje,
        despacho,
        archivos: nuevosArchvios?.map(archivo => {
          return {
            nombre: archivo.filename,
            archivo: archivo.url
          };
        }),
        vistoPor: {
          cliente: enviadoPor === 'Cliente',
          despacho: enviadoPor === 'Despacho'
        }
      };
      const newMensaje = new MensajeModel(mensajeCreate);

      await newMensaje.save();

      returnMessage = newMensaje;

      returnMessage.archivos = newMensaje.archivos.map(archivo => {
        return {
          nombre: archivo.nombre,
          archivo: buildFileUri(`mensajes/${archivo.archivo}`)
        };
      });
    } else {
      const mensajeCreate = {
        chatRoom,
        enviadoPor,
        usuario,
        cliente,
        mensaje,
        despacho,
        archivos: nuevosArchvios?.map(archivo => {
          return {
            nombre: archivo.filename,
            archivo: archivo.url
          };
        }),
        vistoPor: {
          cliente: enviadoPor === 'Cliente',
          despacho: enviadoPor === 'Despacho'
        }
      };
      const newMensaje = new MensajeModel(mensajeCreate);
      await newMensaje.save();

      returnMessage = newMensaje;

      returnMessage.archivos = newMensaje.archivos.map(archivo => {
        return {
          nombre: archivo.nombre,
          archivo: buildFileUri(`mensajes/${archivo.archivo}`)
        };
      });
    }
    res.status(201).json({ message: 'Mensaje creado', mensaje: returnMessage });
  } catch (error) {
    console.log('Error:', error);
    res.status(500).json({ message: error.message });
  }
};

const obtenerMensajesForCLiente = async (req, res) => {
  const { cliente } = req.params;
  const { estatus, page = 1, fechaInicio, fechaFinal } = req.query;

  const options = {
    page,
    limit: 10,
    sort: {
      fechaCreacion: -1,
      estatus: 1
    },
    lean: true
  };

  const query = {
    cliente
  };

  if (fechaInicio && fechaFinal) {
    query.fechaCreacion = {
      $gte: new Date(fechaInicio).setHours(0, 0, 0),
      $lte: new Date(fechaFinal).setHours(23, 59, 59)
    };
  }

  if (estatus) {
    query.estatus = estatus;
  }

  console.log('query', query);
  if (!cliente) {
    return res.status(400).json({ message: 'Falta el cliente' });
  }

  try {
    const [chats, totalNoLeicos] = await Promise.all([
      ChatRoomModel.paginate(query, options),
      MensajeModel.find({
        cliente,
        'vistoPor.cliente': false
      })
    ]);

    const chatsNews = chats.docs.map(chat => {
      return {
        ...chat,
        fecha: formatDateTime(chat.fechaCreacion),
        totalNoLeidos: totalNoLeicos.filter(mensaje => mensaje.chatRoom.toString() === chat._id.toString()).length
      };
    });

    chats.docs = chatsNews;
    res.status(200).json({
      mensajes: chats,
      totalNoLeicos: totalNoLeicos.length
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};
const obtenerMensajesForDespacho = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, fechaInicio, fechaFinal } = req.query;

  const options = {
    page,
    limit: 10,
    sort: {
      fechaCreacion: -1,
      estatus: 1
    },
    populate: [
      {
        path: 'cliente',
        select: 'nombre foto'
      }
    ],
    lean: true
  };

  const query = {
    despacho
  };

  if (fechaInicio && fechaFinal) {
    query.fechaCreacion = {
      $gte: new Date(fechaInicio).setHours(0, 0, 0),
      $lte: new Date(fechaFinal).setHours(23, 59, 59)
    };
  }

  if (estatus) {
    query.estatus = estatus;
  }

  if (!despacho) {
    return res.status(400).json({ message: 'Falta el despacho' });
  }

  try {
    const [chats, totalNoLeicos] = await Promise.all([
      ChatRoomModel.paginate(query, options),
      MensajeModel.find({
        despacho,
        'vistoPor.despacho': false
      })
    ]);

    const chatsNews = chats.docs.map(chat => {
      return {
        ...chat,
        fecha: formatDateTime(chat.fechaCreacion),
        totalNoLeidos: totalNoLeicos.filter(mensaje => mensaje.chatRoom.toString() === chat._id.toString()).length,
        cliente: {
          _id: chat.cliente._id,
          nombre: chat.cliente.nombre,
          foto: chat.cliente.foto
            ? buildFileUri(`clientes/${chat.cliente.foto}`)
            : buildFileUri('default/icono_usuario_100x100_04.jpg')
        }
      };
    });

    chats.docs = chatsNews;
    res.status(200).json({
      mensajes: chats,
      totalNoLeicos: totalNoLeicos.length
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const queryContador = (tipo, id) => {
  let query = {};

  if (tipo === 'Cliente') {
    query = {
      cliente: mongoose.Types.ObjectId(id),
      'vistoPor.cliente': false
    };
  } else {
    query = {
      despacho: mongoose.Types.ObjectId(id),
      'vistoPor.despacho': false
    };
  }

  return query;
};

const contadorTotalMensajes = async (req, res) => {
  const { tipo = '', id } = req.query;

  if (!id) {
    return res.status(400).json({ message: 'Falta el id' });
  }

  try {
    const query = queryContador(tipo, id);
    console.log('query', query);

    const total = await MensajeModel.countDocuments(query);

    res.status(200).json({ total });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const obtenerHistorialClienteMensajes = async (req, res) => {
  const { chatRoom } = req.params;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 20,
    sort: {
      fechaCreacion: -1
    },
    lean: true,
    populate: [
      {
        path: 'usuario',
        select: 'nombre foto apellidoPaterno apellidoMaterno'
      }
    ]
  };

  const query = {
    chatRoom
  };

  if (!chatRoom) {
    return res.status(400).json({ message: 'Falta el chatRoom' });
  }

  try {
    const mensajes = await MensajeModel.paginate(query, options);

    const historial = mensajes.docs.map(mensaje => {
      return {
        _id: mensaje._id,
        position: mensaje.enviadoPor === 'Cliente' ? 'right' : 'left',
        text: mensaje.mensaje,
        timestamp: formatDate(mensaje.fechaCreacion),
        fechaCreacion: mensaje.fechaCreacion,
        vistoPor: mensaje.vistoPor,
        time: formaTime(mensaje.fechaCreacion),
        archivos: mensaje.archivos.map(archivo => {
          return {
            nombre: archivo.nombre,
            archivo: buildFileUri(`mensajes/${archivo.archivo}`)
          };
        }),
        usuario: mensaje.enviadoPor === 'Cliente'
          ? {
            nombre: mensaje.cliente.nombre,
            foto: mensaje.cliente.foto
              ? buildFileUri(`clientes/${mensaje.cliente.foto}`)
              : buildFileUri('default/icono_usuario_100x100_04.jpg')
          }
          : {
            nombre: `${mensaje?.usuario?.nombre || ''} ${mensaje?.usuario?.apellidoPaterno || ''} ${mensaje?.usuario?.apellidoMaterno || ''}`,
            foto: mensaje.usuario.foto
              ? buildFileUri(`usuarios/${mensaje.usuario.foto}`)
              : buildFileUri('default/icono_usuario_100x100_04.jpg')
          }
      };
    });

    mensajes.docs = historial;

    // Actualizar mensajes a vistos por el cliente
    const queryUpdate = {
      chatRoom: mongoose.Types.ObjectId(chatRoom),
      'vistoPor.cliente': false
    };

    await MensajeModel.updateMany(queryUpdate,
      {
        $set: {
          'vistoPor.cliente': true
        }
      }
    );

    const queryContadores = queryContador('Cliente', chatRoom);

    const totalNoLeidos = await MensajeModel.countDocuments(queryContadores);

    const chat = await ChatRoomModel.findById(chatRoom);

    res.status(200).json({
      mensajes,
      totalNoLeidos,
      chat
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const obtenerHistorialDespachoMensajes = async (req, res) => {
  const { chatRoom } = req.params;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 20,
    sort: {
      fechaCreacion: -1
    },
    lean: true,
    populate: [
      {
        path: 'cliente',
        select: 'nombre foto'
      }
    ]
  };

  const query = {
    chatRoom
  };

  if (!chatRoom) {
    return res.status(400).json({ message: 'Falta el chatRoom' });
  }

  try {
    const mensajes = await MensajeModel.paginate(query, options);

    const historial = mensajes.docs.map(mensaje => {
      return {
        _id: mensaje._id,
        position: mensaje.enviadoPor === 'Despacho' ? 'right' : 'left',
        text: mensaje.mensaje,
        timestamp: formatDate(mensaje.fechaCreacion),
        fechaCreacion: mensaje.fechaCreacion,
        vistoPor: mensaje.vistoPor,
        time: formaTime(mensaje.fechaCreacion),
        archivos: mensaje.archivos.map(archivo => {
          return {
            nombre: archivo.nombre,
            archivo: buildFileUri(`mensajes/${archivo.archivo}`)
          };
        }),
        usuario: mensaje.enviadoPor === 'Cliente'
          ? {
            nombre: mensaje.cliente.nombre,
            foto: mensaje.cliente.foto
              ? buildFileUri(`clientes/${mensaje.cliente.foto}`)
              : buildFileUri('default/icono_usuario_100x100_04.jpg')
          }
          : {
            nombre: `${mensaje?.usuario?.nombre || ''} ${mensaje?.usuario?.apellidoPaterno || ''} ${mensaje?.usuario?.apellidoMaterno || ''}`,
            foto: mensaje.usuario.foto
              ? buildFileUri(`usuarios/${mensaje.usuario.foto}`)
              : buildFileUri('default/icono_usuario_100x100_04.jpg')
          }
      };
    });

    mensajes.docs = historial;

    // Actualizar mensajes a vistos por el cliente
    const queryUpdate = {
      chatRoom: mongoose.Types.ObjectId(chatRoom),
      'vistoPor.despacho': false
    };

    await MensajeModel.updateMany(queryUpdate,
      {
        $set: {
          'vistoPor.despacho': true
        }
      }
    );

    const queryContadores = queryContador('Despacho', chatRoom);

    const totalNoLeidos = await MensajeModel.countDocuments(queryContadores);

    const chat = await ChatRoomModel.findById(chatRoom);

    res.status(200).json({
      mensajes,
      totalNoLeidos,
      chat
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const cerrarChat = async (req, res) => {
  const { _id } = req.params;

  try {
    const findChat = await ChatRoomModel.findById(_id);

    if (!findChat) {
      return res.status(404).json({ error: 'Chat no encontrado' });
    }

    findChat.estatus = 'Cerrado';
    await findChat.save();
  } catch (error) {
    console.error('Error deleting messages:', error);
    res.status(500).json({ message: error });
  }
};

module.exports = {
  createMensaje,
  obtenerMensajesForCLiente,
  contadorTotalMensajes,
  obtenerHistorialClienteMensajes,
  cerrarChat,
  obtenerMensajesForDespacho,
  obtenerHistorialDespachoMensajes
};
