const { buildFileUri } = require("../config/s3.js");
const { formatDateTime, formatMoney } = require("../config/functionsGlobal.js");
const moment = require("moment-timezone");
const VentaModel = require("../models/Ventas.js");

const getVentas = async (req, res) => {
  const { page = 1, fechaInicio, fechaFinal } = req.query;

  const limit = 10;

  const query = {};

  if (fechaInicio && fechaFinal) {
    const fechaInicioUTC = moment(fechaInicio).startOf("day").toDate();
    const fechaFinalUTC = moment(fechaFinal).endOf("day").toDate();

    query.fechaVenta = {
      $gte: fechaInicioUTC,
      $lte: fechaFinalUTC,
    };
  }

  const options = {
    page,
    limit,
    sort: { fechaVenta: -1 },
    populate: [
      {
        path: "producto",
        select: "nombre imagen tipo_producto",
      },
      {
        path: "usuario",
        select: "nombre apellidoPaterno apellidoMaterno foto",
      },
      {
        path: "despacho",
        select: "nombre logo",
      },
    ],
    lean: true,
  };

  try {
    const [findVentas] = await Promise.all([
      VentaModel.paginate(query, options),
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const tipo = venta.despacho?._id ? "Despacho" : "Distribuidor";
      const { fechaVenta, producto } = venta;
      let imagen = "";

      if (tipo === "Despacho") {
        imagen = venta?.despacho?.logo
          ? buildFileUri(
              `despachos/${venta?.despacho?._id}/logo/${venta.despacho.logo}`
            )
          : buildFileUri("default/icono_usuario_100x100_04.jpg");
      } else {
        imagen = venta?.usuario?.foto
          ? buildFileUri(`usuarios/${venta?.usuario?.foto}`)
          : buildFileUri("default/icono_usuario_100x100_04.jpg");
      }

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = "";

      if (tipoReferencia === "Paypal") {
        tipoReferenciaLogo =
          "https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png";
      } else if (tipoReferencia === "Stripe") {
        tipoReferenciaLogo =
          "https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png";
      }

      return {
        ...venta,
        fechaVenta: formatDateTime(fechaVenta),
        tipoReferenciaLogo,
        producto: {
          ...producto,
          imagen: producto.imagen
            ? buildFileUri(`productos/${producto?.imagen}`)
            : "",
        },
        importe: formatMoney(venta?.importe),
        usuario: {
          ...venta.usuario,
          tipo,
          nombreCompleto: `${venta?.usuario?.nombre || ""} ${
            venta?.usuario?.apellidoPaterno || ""
          } ${venta?.usuario?.apellidoMaterno || ""}`,
          imagen,
        },
      };
    });

    return res.status(200).json({
      ventas: findVentas,
    });
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al obtener las ventas", error });
  }
};

module.exports = {
  getVentas,
};
