const { formatDate, sendWhatsappRecovery, desencriptar, sendWhatsappRecovery2 } = require('../config/functionsGlobal');
const { sendMail } = require('../config/mail');
const { RegistroUsuarioHTML } = require('../Mail/RegistroUsuarioHTML');
const Despachos = require('../models/Despachos');
const TipoUsuarios = require('../models/TipoUsuarios');
const Usuarios = require('../models/Usuarios');
const Ventas = require('../models/Ventas');

const urlDespacho = process.env.DISTRIBUIDOR_APP;

const obtenerVendedores = async (req, res) => {
  const { estatus, page = 1, search } = req.query;
  const { identificador } = req.params;

  const options = {
    page: parseInt(page, 10),
    limit: 20,
    sort: { _id: -1, nombre: 1 },
    populate: {
      path: 'tipoUsuarios.vendedor'
      //   match: { tipo: "promotor" },
      //   select: "nombre",
    }
  };

  const query = {
    'licencia.referidoPor': identificador
    // "tipoUsuarios.promotor.tipo": "promotor",
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
  }

  try {
    const data = await Usuarios.paginate(query, options);

    data.docs = data.docs.map((user) => {
      user.fechaCreacion = formatDate(user.fechaCreacion);
      return user;
    });

    data.docs = data.docs.filter(
      (doc) =>
        doc.tipoUsuarios.vendedor &&
        doc.tipoUsuarios.vendedor.tipo === 'vendedor'
    );

    return res.status(200).json(data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener los vendedores',
      error
    });
  }
};

const crearVendedor = async (req, res) => {
  const { nombre, apellidoP, correo, celular, referidoPor, comision } =
    req.body;

  if (!nombre || !apellidoP || !correo || !celular) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }
  try {
    const existe = await Usuarios.findOne({ email: correo });

    if (existe) {
      return res.status(400).json({ message: 'El correo ya está en uso' });
    }

    const dataUsuario = await Usuarios.findById(referidoPor);

    if (comision > dataUsuario.licencia.comision) {
      return res.status(400).json({
        message: `La comisión no puede ser mayor que el ${dataUsuario.licencia.comision}%`
      });
    }
    const tipoUsuario = await TipoUsuarios.findOne({ tipo: 'vendedor' });

    const usuario = Usuarios({
      nombre,
      apellidoPaterno: apellidoP,
      email: correo,
      telefono: celular,
      tipoUsuarios: {
        vendedor: tipoUsuario._id
      },
      estatus: 'Activo',
      licencia: {
        referidoPor: dataUsuario._id,
        producto: dataUsuario.licencia.producto || undefined,
        comision,
        vigencia: dataUsuario.licencia.vigencia || undefined
      },
      fechaCreacion: new Date()
    });

    const result = await usuario.save();

    sendWhatsappRecovery2({
      idDespacho: null,
      nombreUsuario: nombre,
      to: celular,
      correo,
      url: urlDespacho + `/password/${result?._id || ''}`
    });

    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      correo,
      '',
      `${urlDespacho}/password/${result?._id || ''}`
    );

    sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correo);

    if (!result) {
      return res.status(400).json({ message: 'Error al crear el vendedor' });
    }

    return res.status(200).json({ message: 'Vendedor creado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al crear al vendedor',
      error
    });
  }
};

const obtenerVendedorPorId = async (req, res) => {
  const { id } = req.params;

  try {
    const vendedor = await Usuarios.findById(id).populate({
      path: 'tipoUsuarios.vendedor'
    });

    if (!vendedor) {
      return res.status(404).json({ mensaje: 'Vendedor no encontrado' });
    }

    return res.status(200).json(vendedor);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener el vendedor',
      error
    });
  }
};

const enviarCredenciales = async (req, res) => {
  const { tipo, id } = req.body;

  try {
    const usuario = await Usuarios.findById(id);

    const password = desencriptar(usuario.password);
    if (tipo === 'correo') {
      const htmlRegistro = RegistroUsuarioHTML(
        `${usuario.nombre} ${usuario.apellidoPaterno}`,
        usuario.email,
        password,
        urlDespacho
      );
      sendMail(htmlRegistro, 'NILDDA: Envió Accesos', usuario.email);
    }

    if (tipo === 'whatsApp') {
      sendWhatsappRecovery({
        to: usuario.telefono,
        correo: usuario.email,
        password,
        url: urlDespacho
      });
    }

    return res.status(200).json({ message: 'Accesos enviados' });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener el vendedor',
      error
    });
  }
};

const actualizarVendedor = async (req, res) => {
  const { id } = req.params;
  const {
    nombre,
    apellidoP,
    correo,
    celular,
    distribuidor,
    comision,
    estatus
  } = req.body;

  if (!nombre || !apellidoP || !correo || !celular) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }

  try {
    const promotor = await Usuarios.findById(id);
    if (!promotor) {
      return res.status(404).json({ message: 'Promotor no encontrado' });
    }

    const existe = await Usuarios.findOne({ email: correo, _id: { $ne: id } });
    if (existe) {
      return res
        .status(400)
        .json({ message: 'El correo ya está en uso por otro usuario' });
    }

    const dataDistribuidor = await Usuarios.findById(distribuidor);
    if (!dataDistribuidor) {
      return res.status(400).json({ message: 'Distribuidor no encontrado' });
    }

    if (comision > dataDistribuidor.licencia.comision) {
      return res.status(400).json({
        message: `La comisión no puede ser mayor que el ${dataDistribuidor.licencia.comision}%`
      });
    }

    promotor.nombre = nombre;
    promotor.apellidoPaterno = apellidoP;
    promotor.email = correo;
    promotor.telefono = celular;
    promotor.estatus = estatus;
    promotor.licencia.referidoPor = dataDistribuidor._id;
    promotor.licencia.comision = comision;

    const result = await promotor.save();

    if (!result) {
      return res
        .status(400)
        .json({ message: 'Error al actualizar el promotor' });
    }

    return res.status(200).json({ message: 'Promotor actualizado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al actualizar el promotor',
      error
    });
  }
};

const eliminarUsuario = async (req, res) => {
  const { id } = req.params;

  try {
    const usuario = await Usuarios.findById(id);
    if (!usuario) {
      return res.status(404).json({ message: 'Usuario no encontrado' });
    }

    const ventas = await Ventas.countDocuments({
      who: id
    });
    if (ventas > 0) {
      return res.status(400).json({
        message:
          'No se puede eliminar al usuario porque tiene ventas asignadas'
      });
    }

    const despachos = await Despachos.countDocuments({
      distribuidor: id
    });

    if (despachos > 0) {
      return res.status(400).json({
        message:
          'No se puede eliminar al usuario porque tiene despachos asignados'
      });
    }

    const vendedores = await Usuarios.countDocuments({
      'licencia.referidoPor': id
    });

    if (vendedores > 0) {
      return res.status(400).json({
        message:
          'No se puede eliminar al usuario porque tiene vendedores asignados'
      });
    }

    await Usuarios.findByIdAndDelete(id);

    return res.status(200).json({ message: 'Usuario eliminado correctamente' });
  } catch (error) {
    console.error('Error:', error);
    return res.status(500).json({
      message: 'Error al eliminar el usuario',
      error
    });
  }
};

module.exports = {
  obtenerVendedores,
  crearVendedor,
  obtenerVendedorPorId,
  enviarCredenciales,
  actualizarVendedor,
  eliminarUsuario
};
