const UsuariosModel = require("../models/Usuarios.js");
const moment = require("moment-timezone");
const { uploadFile, deleteFileS3 } = require("../config/s3.js");
const {
  generatePassword,
  encriptar,
  desencriptar,
  sendWhatsappRecovery,
} = require("../config/functionsGlobal.js");
const { sendMail } = require("../config/mail.js");
const { RegistroUsuarioHTML } = require("../Mail/RegistroUsuarioHTML.js");
const ADMIN_APP = process.env.ADMIN_APP;
const { buildFileUri } = require("../config/s3.js");

const obtenerUsuarios = async (req, res) => {
  try {
    const { page = 1, search = "", estatus = "" } = req.query;

    const limit = 10;

    const options = {
      page,
      limit,
      sort: {
        estatus: 1,
        nombre: 1,
      },
      populate: "tipoUsuarios.admin",
    };

    const query = {
      "tipoUsuarios.admin": { $exists: true },
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.$or = [
        { nombre: { $regex: search, $options: "i" } },
        { email: { $regex: search, $options: "i" } },
        { telefono: { $regex: search, $options: "i" } },
        { apellidoPaterno: { $regex: search, $options: "i" } },
        { apellidoMaterno: { $regex: search, $options: "i" } },
      ];
    }

    const usuarios = await UsuariosModel.paginate(query, options);

    usuarios.docs = usuarios.docs.map((user) => {
      if (user.foto) {
        user.foto = buildFileUri(`usuarios/${user.foto}`);
      } else {
        user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
      }
      return user;
    });

    res.status(200).json({ usuarios });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const obtenerUsuariosSinPaginar = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: "Falta el despacho" });
  }

  const usuarios = await UsuariosModel.find({
    despacho,
    "tipoUsuarios.despacho": { $exists: true },
    estatus: "Activo",
  }).select("nombre apellidoPaterno apellidoMaterno foto");

  usuarios.forEach((user) => {
    if (user.foto) {
      user.foto = buildFileUri(`usuarios/${user.foto}`);
    } else {
      user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }
  });

  return res.status(200).json({ usuarios });
};

const obtenerUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const user = await UsuariosModel.findById(id);
    if (!user) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }
    if (user.foto) {
      user.foto = buildFileUri(
        `despachos/${user?.despacho?._id}/usuarios/${user.foto}`
      );
    } else {
      user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (user.password) user.password = desencriptar(user.password);

    res.status(200).json({ message: "Usuario encontrado", data: user });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const createUsuario = async (req, res) => {
  try {
    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuario,
      fechaNacimiento,
    } = req.body;

    if (!nombre) {
      return res.status(400).json({ message: "Falta el nombre" });
    }

    if (!email) {
      return res.status(400).json({ message: "Falta el correo" });
    }

    if (!tipoUsuario) {
      return res.status(400).json({ message: "Falta el tipo de usuario" });
    }

    const fechaNacimientoMexico = fechaNacimiento
      ? moment(fechaNacimiento).tz("America/Mexico_City")
      : null;

    const objUser = {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuarios: {
        admin: tipoUsuario,
      },
      estatus: "Activo",
      fechaNacimiento: fechaNacimientoMexico,
    };

    const findEmailExist = await UsuariosModel.findOne({ email });

    if (findEmailExist) {
      return res.status(400).json({ message: "El correo ya existe" });
    }

    const passwordSin = generatePassword();
    const passwordEncrypt = encriptar(passwordSin);
    objUser.password = passwordEncrypt;
    const userSave = await UsuariosModel.create(objUser);

    // enviar correo con la contraseña
    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      email,
      passwordSin,
      `${ADMIN_APP}/login`
    );
    sendMail(htmlRegistro, "NILDDA: Envió Accesos", email);

    // enviar whatsapp con la contraseña
    if (telefono) {
      sendWhatsappRecovery({
        to: telefono,
        nombre,
        correo: email,
        password: passwordSin,
        url: ADMIN_APP,
      });
    }

    res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const actualizarUsuario = async (req, res) => {
  try {
    const { id } = req.params;

    // const filename = req?.file?.filename ?? null;

    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      password,
    } = req.body;

    const passwordEncrypt = encriptar(password);

    // Actualizar el usuario sin devolver el objeto actualizado
    const objAntes = await UsuariosModel.findById(id);

    if (!objAntes) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    if (req.files && req.files.foto) {
      deleteFileS3(`usuarios/${objAntes.foto}`);
    }

    objAntes.nombre = nombre ?? objAntes.nombre;
    objAntes.apellidoPaterno = apellidoPaterno ?? objAntes.apellidoPaterno;
    objAntes.apellidoMaterno = apellidoMaterno ?? objAntes.apellidoMaterno;
    objAntes.telefono = telefono ?? objAntes.telefono;
    objAntes.email = email ?? objAntes.email;
    objAntes.password = passwordEncrypt ?? objAntes.password;

    if (req.files && req.files.foto) {
      const foto = await uploadFile(req.files.foto, "usuarios");
      objAntes.foto = foto.fileName;
    }

    await objAntes.save();

    // Buscar y devolver el objeto actualizado
    const userUpdate = await UsuariosModel.findById(id).populate("despacho");

    if (userUpdate.foto) {
      userUpdate.foto = buildFileUri(`usuarios/${userUpdate.foto}`);
    } else {
      userUpdate.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (userUpdate.despacho && userUpdate.despacho.logo) {
      userUpdate.despacho.logo = buildFileUri(
        `despachos/${userUpdate.despacho._id}/logo/${userUpdate.despacho.logo}`
      );
    }

    userUpdate.password = desencriptar(userUpdate.password);

    res.status(200).json({ message: "Usuario actualizado", data: userUpdate });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const updateUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuario,
      password,
      estatus,
    } = req.body;

    let foto = null;

    if (req?.files?.foto) {
      foto = await uploadFile(req.files.foto, "usuarios");
    }

    const objUser = {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      "tipoUsuarios.admin": tipoUsuario,
      estatus,
      password: encriptar(password),
    };

    if (foto) {
      const user = await UsuariosModel.findById(id).select("foto");

      if (user.foto) {
        deleteFileS3(`usuarios/${user.foto}`);
      }

      objUser.foto = foto.fileName;
    }

    const objAntes = await UsuariosModel.findByIdAndUpdate(id, objUser);

    if (!objAntes) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    res.status(200).json({ message: "Usuario actualizado" });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const deleteUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const user = await UsuariosModel.findById(id);

    if (!user) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    user.estatus = "Inactivo";
    await user.save();

    // await UsuariosModel.findByIdAndDelete(id);
    res.status(200).json({ message: "Se modifico el estatus del usuario" });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

module.exports = {
  obtenerUsuarios,
  obtenerUsuariosSinPaginar,
  createUsuario,
  obtenerUsuario,
  actualizarUsuario,
  updateUsuario,
  deleteUsuario,
};
