/* eslint-disable indent */
const UsuariosModel = require("../models/Usuarios.js");
const TipoUsuarioModel = require("../models/TipoUsuarios.js");
const DespachoModel = require("../models/Despachos.js");
const productoModel = require("../models/Productos.js");
const SuscripcionesModel = require("../models/Suscripciones.js");
const VentasSchema = require("../models/Ventas.js");
const ModulosSchema = require("../models/Modulos.js");
const TipoUsuario = require("../models/TipoUsuarios.js");
const ClienteModel = require("../models/Clientes.js");
const moment = require("moment-timezone");
const fs = require("fs");
const path = require("path");
const APP_URL = process.env.APP_URL;
const { uploadFile, deleteFileS3 } = require("../config/s3.js");
const {
  validarPassword,
  generatePassword,
  encriptar,
  desencriptar,
  sendWhatsappRecovery,
  sendWhatsappRecovery2,
} = require("../config/functionsGlobal.js");
const { sendMail } = require("../config/mail.js");
const { RegistroUsuarioHTML } = require("../Mail/RegistroUsuarioHTML.js");
const CuentasBancariasModel = require("../models/CuentaBancaria.js");
const { buildFileUri } = require("../config/s3.js");
const Ventas = require("../models/Ventas.js");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcryptjs");
const { OAuth2Client } = require("google-auth-library");
const ContadorProspecto = require("../models/ContadorProspecto.js");
const Expedientes = require("../models/Expedientes.js");
const ExpedientesFolios = require("../models/ExpedientesFolios.js");
const Gastos = require("../models/Gastos.js");
const DESPACHO_APP = process.env.DESPACHO_APP;
const CLIENTE_APP = process.env.CLIENTE_APP;
const DISTRIBUIDOR_APP = process.env.DISTRIBUIDOR_APP;
const ADMIN_APP = process.env.ADMIN_APP;
const client = new OAuth2Client(process.env.GOOGLE_CLIENT_ID);

const login = async (req, res) => {
  try {
    let { correo, password, tipo } = req.body;

    correo = correo?.trim();
    password = password?.trim();

    console.log(req.body);
    if (!correo || !password || !tipo) {
      return res
        .status(400)
        .json({ message: "Correo y contraseña son requeridos" });
    }

    const userFind = await UsuariosModel.findOne({
      email: correo,
      estatus: "Activo",
    }).populate("despacho");

    if (!userFind) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    if (tipo === "admin" && !userFind.tipoUsuarios?.admin) {
      return res
        .status(404)
        .json({ message: "No tienes acceso como administrador" });
    }

    const passwordOk = validarPassword(password, userFind.password);

    if (!passwordOk) {
      return res.status(401).json({ message: "Contraseña incorrecta" });
    }

    // Procesar foto de usuario
    if (userFind.foto) {
      userFind.foto = buildFileUri(`usuarios/${userFind.foto}`);
    } else {
      userFind.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    // Procesar logo del despacho
    if (userFind.despacho && userFind.despacho.logo) {
      userFind.despacho.logo = buildFileUri(
        `despachos/${userFind.despacho._id}/logo/${userFind.despacho.logo}`
      );
    }

    return res.status(200).json({ message: "Usuario logeado", data: userFind });
  } catch (error) {
    console.error("❌ Error en login:", error);
    return res
      .status(500)
      .json({ message: "Error del servidor", error: error.message });
  }
};

const linkGoogle = async (req, res) => {
  const { userId, googleId, token } = req.body;
  try {
    console.log(req.body);
    // Verifica el token (si es necesario, usa una biblioteca para verificar tokens de Google)
    const decoded = jwt.decode(token);

    if (decoded.sub !== googleId) {
      return res.status(401).json({ message: "Token de Google no válido" });
    }

    // Busca el usuario y actualiza su googleId
    const user = await UsuariosModel.findById(userId);
    if (!user) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    user.googleId = googleId;
    await user.save();

    res.json({ message: "Cuenta de Google vinculada exitosamente" });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Error interno del servidor" });
  }
};

const loginGoogle = async (req, res) => {
  const { token } = req.body;

  try {
    const ticket = await client.verifyIdToken({
      idToken: token,
      audience: process.env.GOOGLE_CLIENT_ID,
    });

    const payload = ticket.getPayload(); // ✅ Token verificado
    const googleId = payload.sub;

    let user = await UsuariosModel.findOne({ googleId }).populate("despacho");

    if (!user) {
      // Opcional: buscar por correo y vincular
      user = await UsuariosModel.findOne({ email: payload.email });
      if (user) {
        user.googleId = googleId;
        await user.save();
      }
    }

    if (!user) {
      return res
        .status(404)
        .json({ message: "No se encontró una cuenta vinculada a Google." });
    }

    // Procesar foto y despacho si existe
    if (user.foto) {
      user.foto = buildFileUri(`usuarios/${user.foto}`);
    } else {
      user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (user.despacho && user.despacho.logo) {
      user.despacho.logo = buildFileUri(
        `despachos/${user.despacho._id}/logo/${user.despacho.logo}`
      );
    }

    res.json({ message: "Inicio de sesión exitoso", data: user });
  } catch (error) {
    console.error("❌ loginGoogle error:", error);
    res.status(500).json({ message: "Error interno al autenticar con Google" });
  }
};

const loginCliente = async (req, res) => {
  try {
    const { correo, password } = req.body;

    if (!correo || !password) {
      return res.status(400).json({ message: "Faltan datos" });
    }

    const userFind = await ClienteModel.findOne({ correo, estatus: "Activo" });

    if (!userFind) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    const passwordValid = validarPassword(password, userFind.password);

    if (!passwordValid) {
      return res.status(404).json({ message: "Contraseña incorrecta" });
    }

    userFind.password = desencriptar(userFind.password);

    console.log(path.join("src/uploads/clientes", userFind.foto));
    if (
      userFind.foto !== "" &&
      fs.existsSync(path.join("src/uploads/clientes", userFind.foto))
    ) {
      userFind.foto = `${APP_URL}/uploads/clientes/${userFind.foto}`;
    } else {
      userFind.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    res.status(200).json({ message: "Usuario logeado", data: userFind });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const register = async (req, res) => {
  try {
    const fecha = new Date();
    const {
      nombre,
      apellidoPaterno = "",
      apellidoMaterno = "",
      telefono = "",
      email,
      tipoUsuario = "",
      producto,
      formaPago,
      estado,
      nombreDespacho,
      googleId,
    } = req.body;

    const cleanEmail = Array.isArray(email) ? email[0].trim() : email.trim();

    const existe = await UsuariosModel.findOne({ email: correoElectronico });
    if (existe) {
      return res.status(400).json({ message: "El correo ya está registrado." });
    }

    if (!nombre || !cleanEmail || !tipoUsuario || !nombreDespacho) {
      return res.status(400).json({ message: "Faltan datos obligatorios" });
    }

    if (tipoUsuario === "despacho" && (!producto || !formaPago)) {
      return res
        .status(400)
        .json({ message: "Falta el producto o forma de pago" });
    }

    const findEmailExist = await UsuariosModel.findOne({ email: cleanEmail });
    if (findEmailExist) {
      return res.status(400).json({ message: "El correo ya existe" });
    }

    let objDespacho = {};
    let precioProducto = 0;

    const objUser = {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email: cleanEmail,
      tipoUsuarios: {},
    };

    if (tipoUsuario === "admin") {
      const tipoAdmin = await TipoUsuarioModel.findOne({ tipo: "admin" });
      if (!tipoAdmin) {
        return res
          .status(400)
          .json({ message: "No existe el tipo de usuario admin" });
      }
      objUser.tipoUsuarios.admin = tipoAdmin._id;
    }

    if (tipoUsuario === "despacho") {
      const tipoDespacho = await TipoUsuarioModel.findOne({ tipo: "despacho" });
      if (!tipoDespacho) {
        return res
          .status(400)
          .json({ message: "No existe el tipo de usuario despacho" });
      }

      const findProducto = await productoModel.findById(producto);
      if (!findProducto) {
        return res.status(400).json({ message: "No existe el producto" });
      }

      const vigencia = fecha.setMonth(fecha.getMonth() + 1);
      objDespacho = {
        nombre: nombreDespacho,
        contadorExp: {
          contador: 0,
          limite: findProducto.cantidad,
          vigencia,
        },
        correo: cleanEmail,
        telefono,
        estado,
      };

      objUser.tipoUsuarios.despacho = tipoDespacho._id;
      precioProducto = findProducto.precio;
    }

    const passwordGenerate = generatePassword();
    const passwordEncrypted = encriptar(passwordGenerate);
    objUser.password = passwordEncrypted;

    const userSave = await UsuariosModel.create(objUser);

    if (tipoUsuario === "despacho") {
      objDespacho.creadoPor = userSave._id;
      const despachoCreate = await DespachoModel.create(objDespacho);

      await UsuariosModel.findByIdAndUpdate(userSave._id, {
        despacho: despachoCreate._id,
      });

      await SuscripcionesModel.create({
        despacho: despachoCreate._id,
        fechaInicio: fecha,
        fechaFin: fecha.setMonth(fecha.getMonth() + 1),
        estatus: "Vigente",
        creadoPor: userSave._id,
        producto,
        precio: precioProducto,
        observaciones: "Se adquirió el producto por primera vez",
      });

      await VentasSchema.create({
        despacho: despachoCreate._id,
        fechaVenta: fecha,
        producto,
        importe: precioProducto,
        fecha_pago: fecha,
        formaPago,
        estatus: "Pagado",
      });

      const modulos = await ModulosSchema.find({
        estatus: "Activo",
        tipo: "despacho",
      });

      await TipoUsuario.create({
        nombre: "Administrador",
        tipo: "despacho",
        despacho: despachoCreate._id,
        modulos: modulos.map((modulo) => ({
          modulo: modulo._id,
          permisos: {
            create: true,
            read: true,
            update: true,
            delete: true,
            download: true,
          },
        })),
      });
    }

    const nuevoUsuario = new UsuariosModel({
      nombre: nombreCompleto,
      correo: correoElectronico,
      telefono,
      nombreDespacho,
      googleId: googleId || null, // guarda el googleId si viene
    });

    await nuevoUsuario.save();

    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      cleanEmail,
      passwordGenerate
    );

    await sendMail(htmlRegistro, "NILDDA: Envió Accesos", cleanEmail);

    return res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    return res
      .status(500)
      .json({ message: error.message, line_error: error.stack });
  }
};

const rememberPasswordCliente = async ({ correo, res, tipo }) => {
  const findCliente = await ClienteModel.findOne({
    correo,
    estatus: "Activo",
  });

  if (!findCliente) {
    return res.status(404).json({ message: "Cliente no encontrado" });
  }

  const nombreCompleto = `${findCliente.nombre}`;
  const password = desencriptar(findCliente?.password || "");

  const htmlRegistro = RegistroUsuarioHTML(
    nombreCompleto,
    correo,
    password,
    CLIENTE_APP
  );

  if (findCliente.correo && tipo === "email") {
    sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);
  }

  if (tipo === "todos" || tipo === "whatsapp") {
    if (findCliente.telefono) {
      sendWhatsappRecovery2({
        idDespacho: findCliente.despacho,
        to: findCliente.telefono,
        nombreUsuario: nombreCompleto,
        correo,
        url: CLIENTE_APP + `/password/${findCliente._id}`,
      });
    }

    // if (findCliente.correo) {
    //   sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);
    // }
  }

  return res.status(200).json({ message: "Accesos enviados", tipo: "cliente" });
};

const rememberPasswordUsuario = async ({
  correo,
  res,
  tipo,
  tipoUsuario,
  wp,
}) => {
  const query = {
    email: correo,
    estatus: "Activo",
  };

  if (tipoUsuario === "admin") {
    query["tipoUsuarios.admin"] = { $exists: true };
  }

  if (tipoUsuario === "despacho") {
    query.despacho = { $exists: true };
  }

  if (tipoUsuario === "distribuidor") {
    query["tipoUsuarios.distribuidor"] = { $exists: true };
  }

  const findUser = await UsuariosModel.findOne(query);

  if (!findUser) {
    if (!findUser) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }
  }

  let plataforma = null;

  if (tipoUsuario === "admin") plataforma = ADMIN_APP;

  if (tipoUsuario === "despacho") plataforma = DESPACHO_APP;

  if (tipoUsuario === "distribuidor") plataforma = DISTRIBUIDOR_APP;

  const nombreCompleto = `${findUser.nombre} ${findUser.apellidoPaterno} ${findUser.apellidoMaterno}`;

  if (!findUser.password) {
    const htmlRegistro = RegistroUsuarioHTML(
      nombreCompleto,
      correo,
      "",
      `${DESPACHO_APP}/password/${findUser._id}`,
      true
    );
    sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);
    return res
      .status(200)
      .json({ message: "Accesos enviados", tipo: "despacho" });
  }

  const password = desencriptar(findUser.password);

  const htmlRegistro = RegistroUsuarioHTML(
    nombreCompleto,
    correo,
    password,
    plataforma || DESPACHO_APP
  );

  if (findUser.telefono && wp === "Si") {
    sendWhatsappRecovery2({
      idDespacho: findUser?.despacho,
      to: findUser.telefono,
      nombreUsuario: nombreCompleto,
      correo,
      password,
      url: plataforma,
    });
  }

  if (findUser.email && tipo === "email") {
    sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);
  }

  if (tipo === "todos") {
    if (findUser.telefono) {
      sendWhatsappRecovery2({
        idDespacho: findUser?._id,
        to: findUser.telefono,
        nombreUsuario: nombreCompleto,
        correo,
        password,
        url: plataforma,
      });
    }

    if (findUser.email) {
      sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);
    }
  }

  findUser.password = password;

  res.status(200).json({ message: "Accesos enviados", tipo: "despacho" });
};

const rememberPassword = async (req, res) => {
  try {
    const {
      correo,
      tipo,
      acceso = "despacho",
      tipoUsuario,
      wp = "",
    } = req.body;

    if (!correo) {
      return res.status(400).json({ message: "Falta el correo" });
    }

    if (acceso === "cliente") {
      rememberPasswordCliente({ correo, res, tipo });
    } else {
      rememberPasswordUsuario({ correo, res, tipo, tipoUsuario, wp });
    }
  } catch (error) {
    console.log(error);
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const actualizarUsuario = async (req, res) => {
  try {
    const { id } = req.params;

    // const filename = req?.file?.filename ?? null;

    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      password,
    } = req.body;

    const passwordEncrypt = encriptar(password);

    // Actualizar el usuario sin devolver el objeto actualizado
    const objAntes = await UsuariosModel.findById(id);

    if (!objAntes) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    if (req.files && req.files.foto) {
      deleteFileS3(`usuarios/${objAntes.foto}`);
    }

    objAntes.nombre = nombre ?? objAntes.nombre;
    objAntes.apellidoPaterno = apellidoPaterno ?? objAntes.apellidoPaterno;
    objAntes.apellidoMaterno = apellidoMaterno ?? objAntes.apellidoMaterno;
    objAntes.telefono = telefono ?? objAntes.telefono;
    objAntes.email = email ?? objAntes.email;
    objAntes.password = passwordEncrypt ?? objAntes.password;

    if (req.files && req.files.foto) {
      const foto = await uploadFile(req.files.foto, "usuarios");
      objAntes.foto = foto.fileName;
    }

    await objAntes.save();

    // Buscar y devolver el objeto actualizado
    const userUpdate = await UsuariosModel.findById(id).populate("despacho");

    if (userUpdate.foto) {
      userUpdate.foto = buildFileUri(`usuarios/${userUpdate.foto}`);
    } else {
      userUpdate.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (userUpdate.despacho && userUpdate.despacho.logo) {
      userUpdate.despacho.logo = buildFileUri(
        `despachos/${userUpdate.despacho._id}/logo/${userUpdate.despacho.logo}`
      );
    }

    userUpdate.password = desencriptar(userUpdate.password);

    res.status(200).json({ message: "Usuario actualizado", data: userUpdate });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const obtenerUsuarios = async (req, res) => {
  try {
    const { page = 1, search = "", estatus = "" } = req.query;
    const { despacho } = req.params;

    if (!despacho) {
      return res.status(400).json({ message: "Falta el despacho" });
    }

    const limit = 10;

    const options = {
      page,
      limit,
      sort: {
        estatus: 1,
        nombre: 1,
      },
      populate: {
        path: "tipoUsuarios.despacho",
        select: "nombre",
      },
    };

    const query = {
      despacho,
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.$or = [
        { nombre: { $regex: search, $options: "i" } },
        { email: { $regex: search, $options: "i" } },
        { telefono: { $regex: search, $options: "i" } },
        { apellidoPaterno: { $regex: search, $options: "i" } },
        { apellidoMaterno: { $regex: search, $options: "i" } },
      ];
    }

    const usuarios = await UsuariosModel.paginate(query, options);

    usuarios.docs = usuarios.docs.map((user) => {
      if (user.foto) {
        user.foto = buildFileUri(`usuarios/${user.foto}`);
      } else {
        user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
      }

      // user.password = desencriptar(user.password);
      return user;
    });

    res.status(200).json({ usuarios });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const obtenerUsuariosSinPaginar = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: "Falta el despacho" });
  }

  const usuarios = await UsuariosModel.find({
    despacho,
    "tipoUsuarios.despacho": { $exists: true },
    estatus: "Activo",
  }).select("nombre apellidoPaterno apellidoMaterno foto email");

  usuarios.forEach((user) => {
    if (user.foto) {
      user.foto = buildFileUri(`usuarios/${user.foto}`);
    } else {
      user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }
  });

  return res.status(200).json({ usuarios });
};

const createUsuarioDespacho = async (req, res) => {
  try {
    const { despacho } = req.params;
    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuario,
      fechaNacimiento,
    } = req.body;

    if (!despacho) {
      return res.status(400).json({ message: "Falta el despacho" });
    }

    if (!nombre) {
      return res.status(400).json({ message: "Falta el nombre" });
    }

    if (!email) {
      return res.status(400).json({ message: "Falta el correo" });
    }

    if (!tipoUsuario) {
      return res.status(400).json({ message: "Falta el tipo de usuario" });
    }

    const fechaNacimientoMexico = fechaNacimiento
      ? moment(fechaNacimiento).tz("America/Mexico_City")
      : null;

    // const password = generatePassword();

    const objUser = {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuarios: {
        despacho: tipoUsuario,
      },
      despacho,
      // password: encriptar(password),
      estatus: "Activo",
      fechaNacimiento: fechaNacimientoMexico,
    };

    const findEmailExist = await UsuariosModel.findOne({
      email,
    });

    if (findEmailExist) {
      return res.status(400).json({ message: "El correo ya existe" });
    }

    const userSave = await UsuariosModel.create(objUser);

    // enviar correo con la contraseña
    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      email,
      "",
      `${DESPACHO_APP}/password/${userSave._id}`,
      true
    );
    sendMail(htmlRegistro, "NILDDA: Envió Accesos", email);

    if (telefono) {
      sendWhatsappRecovery2({
        idDespacho: despacho,
        to: telefono,
        nombre,
        correo: email,
        url: DESPACHO_APP + `/password/${userSave._id}`,
      });
    }

    res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const obtenerUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const user = await UsuariosModel.findById(id);
    if (!user) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }
    if (user.foto) {
      user.foto = buildFileUri(
        `despachos/${user?.despacho?._id}/usuarios/${user.foto}`
      );
    } else {
      user.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (user.password) {
      user.password = desencriptar(user.password);
    }

    res.status(200).json({ message: "Usuario encontrado", data: user });
  } catch (error) {
    console.log(error);
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const updateUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuario,
      password,
      estatus,
    } = req.body;

    let foto = null;

    if (req?.files?.foto) {
      foto = await uploadFile(req.files.foto, "usuarios");
    }

    const objUser = {
      nombre,
      apellidoPaterno,
      apellidoMaterno,
      telefono,
      email,
      tipoUsuarios: {
        despacho: tipoUsuario,
      },
      estatus,
      password: encriptar(password),
    };

    if (foto) {
      const user = await UsuariosModel.findById(id).select("foto");

      if (user.foto) {
        deleteFileS3(`usuarios/${user.foto}`);
      }

      objUser.foto = foto.fileName;
    }

    const objAntes = await UsuariosModel.findByIdAndUpdate(id, objUser);

    if (!objAntes) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    res.status(200).json({ message: "Usuario actualizado" });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const deleteUsuario = async (req, res) => {
  try {
    const { id } = req.params;
    const user = await UsuariosModel.findById(id);

    if (!user) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    console.log(id);
    const [
      buscarenProspecto,
      buscarenExpediente,
      buscarenExpedienteFolio,
      buscarenUsuarios,
      buscarenGastos,
      buscarenVentas,
    ] = await Promise.all([
      ContadorProspecto.exists({
        $or: [{ prospecto: id }, { distribuidor: id }],
      }),
      Expedientes.exists({ creadoPor: id }),
      ExpedientesFolios.exists({ creadoPor: id }),
      UsuariosModel.exists({ "licencia.referidoPor": id }),
      Gastos.exists({ $or: [{ creadoPor: id }, { usuarioAsignado: id }] }),
      Ventas.exists({
        $or: [
          { usuario: id },
          { who: id },
          { "distribuidor.distribuidor": id },
          { "vendedor.vendedor": id },
          { "promotor.promotor": id },
        ],
      }),
    ]);

    const hayRelacion =
      buscarenProspecto ||
      buscarenExpediente ||
      buscarenExpedienteFolio ||
      buscarenUsuarios ||
      buscarenGastos ||
      buscarenVentas;

    if (hayRelacion) {
      return res.status(400).json({
        message:
          "No se puede eliminar el usuario porque tiene registros relacionados.",
        relacionados: {
          prospecto: !!buscarenProspecto,
          expediente: !!buscarenExpediente,
          expedienteFolio: !!buscarenExpedienteFolio,
          referido: !!buscarenUsuarios,
          gastos: !!buscarenGastos,
          ventas: !!buscarenVentas,
        },
      });
    }
    await UsuariosModel.findByIdAndDelete(id);
    res.status(200).json({ message: "Usuario eliminado" });
  } catch (error) {
    res.status(404).json({ message: error.message, line_error: error.stack });
  }
};

const cargaUsuario = async (req, res) => {
  try {
    const { usuario } = req.params;

    if (!usuario) {
      return res.status(400).json({ message: "Falta el usuario" });
    }

    const userFind = await UsuariosModel.findOne({ _id: usuario }).populate(
      "despacho"
    );

    const fehcaMexico = moment().tz("America/Mexico_City").format();
    userFind.ultimoAcceso = fehcaMexico;
    await userFind.save();

    if (!userFind) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    userFind.password = userFind.password
      ? desencriptar(userFind.password)
      : "";

    if (userFind.foto) {
      userFind.foto = buildFileUri(`usuarios/${userFind.foto}`);
    } else {
      userFind.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
    }

    if (userFind.despacho && userFind.despacho.logo) {
      userFind.despacho.logo = buildFileUri(
        `despachos/${userFind?.despacho?._id}/logo/${userFind.despacho.logo}`
      );
    }

    res.status(200).json({ message: "Usuario encontrado", data: userFind });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const obtenerInformacioDistribuidor = async (req, res) => {
  const { id } = req.params;
  try {
    const [usuario, findCuenta] = await Promise.all([
      UsuariosModel.findById(id).select(
        "nombre apellidoPaterno apellidoMaterno email telefono foto comprobantes despacho rfc"
      ),
      CuentasBancariasModel.findOne({ usuario: id }),
    ]);

    if (!usuario) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    if (usuario?.comprobantes?.constancia) {
      usuario.comprobantes.constancia = buildFileUri(
        `distribuidores/${usuario?._id}/comprobantes/${usuario.comprobantes.constancia}`
      );
    }

    if (usuario?.comprobantes?.ine?.frontal) {
      usuario.comprobantes.ine.frontal = buildFileUri(
        `distribuidores/${usuario?._id}/comprobantes/${usuario.comprobantes.ine.frontal}`
      );
    }

    if (usuario?.comprobantes?.ine?.trasera) {
      usuario.comprobantes.ine.trasera = buildFileUri(
        `distribuidores/${usuario?._id}/comprobantes/${usuario.comprobantes.ine.trasera}`
      );
    }

    if (usuario?.comprobantes?.domicilio) {
      usuario.comprobantes.domicilio = buildFileUri(
        `distribuidores/${usuario?._id}/comprobantes/${usuario.comprobantes.domicilio}`
      );
    }

    res.status(200).json({
      usuario,
      cuenta: findCuenta,
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const actualizarRetirarFondos = async (req, res) => {
  const { id } = req.params;
  const constancia = req?.files?.constancia || null;
  const ineFrontal = req?.files?.ineFrontal || null;
  const ineTrasera = req?.files?.ineTrasera || null;
  const domicilio = req?.files?.domicilio || null;
  const { rfc, tipoTransferencia, numero, tarjeta, banco, titular } = req.body;

  try {
    const [usuario, findCuenta] = await Promise.all([
      UsuariosModel.findById(id),
      CuentasBancariasModel.findOne({ usuario: id }),
    ]);

    if (!usuario) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    if (constancia) {
      if (usuario.comprobantes.constancia) {
        deleteFileS3(
          `distribuidores/${usuario._id}/comprobantes/${usuario.comprobantes.constancia}`
        );
      }
      const constanciaFile = await uploadFile(
        constancia,
        `distribuidores/${usuario._id}/comprobantes`
      );
      usuario.comprobantes.constancia = constanciaFile.fileName;
    }

    if (ineFrontal) {
      if (usuario.comprobantes.ine.frontal) {
        deleteFileS3(
          `distribuidores/${usuario._id}/comprobantes/${usuario.comprobantes.ine.frontal}`
        );
      }
      const ineFrontalFile = await uploadFile(
        ineFrontal,
        `distribuidores/${usuario._id}/comprobantes`
      );
      usuario.comprobantes.ine.frontal = ineFrontalFile.fileName;
    }

    if (ineTrasera) {
      if (usuario.comprobantes.ine.trasera) {
        deleteFileS3(
          `distribuidores/${usuario._id}/comprobantes/${usuario.comprobantes.ine.trasera}`
        );
      }
      const ineTraseraFile = await uploadFile(
        ineTrasera,
        `distribuidores/${usuario._id}/comprobantes`
      );
      console.log(ineTraseraFile);
      usuario.comprobantes.ine.trasera = ineTraseraFile.fileName;
    }

    if (domicilio) {
      if (usuario.comprobantes.domicilio) {
        deleteFileS3(
          `distribuidores/${usuario._id}/comprobantes/${usuario.comprobantes.domicilio}`
        );
      }
      const domicilioFile = await uploadFile(
        domicilio,
        `distribuidores/${usuario._id}/comprobantes`
      );
      usuario.comprobantes.domicilio = domicilioFile.fileName;
    }
    usuario.rfc = rfc;

    usuario.save();

    if (findCuenta) {
      findCuenta.tipo = tipoTransferencia;
      findCuenta.banco = banco;
      findCuenta.titular = titular;

      if (tipoTransferencia === "Débito") {
        findCuenta.banco = banco;
        findCuenta.numero = tarjeta;
      }

      if (tipoTransferencia === "Clabe") {
        findCuenta.numero = numero;
        findCuenta.banco = null;
      }

      await findCuenta.save();
    } else {
      const cuentaNueva = {
        usuario: id,
        tipo: tipoTransferencia,
        titular,
      };

      if (tipoTransferencia === "Débito") {
        cuentaNueva.banco = banco;
        cuentaNueva.numero = tarjeta;
      }

      if (tipoTransferencia === "Clabe") {
        cuentaNueva.numero = numero;
      }

      await CuentasBancariasModel.create(cuentaNueva);
    }

    res.status(200).json({ message: "Comprobantes actualizados" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const findDistribuidorForClave = async (req, res) => {
  try {
    const { clave } = req.params;

    if (!clave) {
      return res.status(400).json({ message: "Falta la clave" });
    }

    const findDistribuidor = await UsuariosModel.findOne({
      clave,
      // 'tipoUsuarios.distribuidor': { $ne: null },
      // producto si es diferente null
      "licencia.producto": { $ne: null },
      estatus: "Activo",
    }).select(
      "nombre apellidoPaterno apellidoMaterno email telefono foto licencia clave"
    );

    if (!findDistribuidor) {
      return res.status(404).json({ message: "Distribuidor no encontrado" });
    }

    // licencia.vigencias que aun esta activa
    if (findDistribuidor?.licencia) {
      const fechaMexico = moment().tz("America/Mexico_City");
      if (
        findDistribuidor &&
        findDistribuidor.licencia.vigencia > fechaMexico
      ) {
        if (findDistribuidor.foto) {
          findDistribuidor.foto = buildFileUri(
            `usuarios/${findDistribuidor.foto}`
          );
        } else {
          findDistribuidor.foto = buildFileUri(
            "default/icono_usuario_100x100_04.jpg"
          );
        }
        return res
          .status(200)
          .json({ message: "Distribuidor encontrado", data: findDistribuidor });
      } else {
        return res.status(404).json({ message: "La licencia ha expirado" });
      }
    } else {
      return res.status(404).json({ message: "Distribuidor no encontrado" });
    }
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

const findDistribuidorForClaveProspecto = async (req, res) => {
  try {
    const { clave } = req.params;

    if (!clave) {
      return res.status(400).json({ message: "Falta la clave" });
    }

    const findDistribuidor = await UsuariosModel.findOne({
      clave,
      "tipoUsuarios.distribuidor": { $ne: null },
      estatus: "Prospecto",
    }).select(
      "nombre apellidoPaterno apellidoMaterno email telefono foto licencia"
    );

    if (!findDistribuidor) {
      return res.status(404).json({ message: "Distribuidor no encontrado" });
    }

    return res
      .status(200)
      .json({ message: "Distribuidor encontrado", data: findDistribuidor });
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

const generateDistribuidorRegistroProspecto = async (req, res) => {
  const { nombreCompleto, correo, telefono, referidoPor = null } = req.body;

  if (!nombreCompleto) {
    return res.status(400).json({ message: "Falta el nombre" });
  }

  if (!correo) {
    return res.status(400).json({ message: "Falta el correo" });
  }

  if (!telefono) {
    return res.status(400).json({ message: "Falta el telefono" });
  }

  try {
    // buscar si no existe el correo
    const findEmailExist = await UsuariosModel.findOne({
      email: correo,
    });

    if (findEmailExist) {
      return res.status(400).json({ message: "El correo ya existe" });
    }

    const passwordSin = generatePassword();
    const passwordEncrypt = encriptar(passwordSin);

    const findTipoUsuarioDistribuidor = await TipoUsuarioModel.findOne({
      tipo: "distribuidor",
    }).select("_id");

    let referido = null;

    if (referidoPor) {
      const findDistribuidor = await UsuariosModel.findOne({
        clave: referidoPor,
      }).select("_id");

      if (findDistribuidor) {
        referido = findDistribuidor._id;
      }
    }

    const objUser = {
      nombre: nombreCompleto,
      email: correo,
      telefono,
      password: passwordEncrypt,
      tipoUsuarios: {
        distribuidor: findTipoUsuarioDistribuidor._id,
      },
      estatus: "Prospecto",
      licencia: {
        referidoPor: referido,
        comision: 0,
      },
    };

    const userSave = await UsuariosModel.create(objUser);

    // sendWhatsappRecovery({ to: '2213425514', correo, password: passwordSin, url: DISTRIBUIDOR_APP });

    res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};
const ConvertirProspectoDistribuidor = async (req, res) => {
  const { _id, producto, tipoReferencia, referencia } = req.body;

  if (!_id) {
    return res.status(400).json({ message: "Falta el id del usuario" });
  }

  if (!producto) {
    return res.status(400).json({ message: "Falta el producto" });
  }
  if (!tipoReferencia) {
    return res.status(400).json({ message: "Falta el producto" });
  }
  if (!referencia) {
    return res.status(400).json({ message: "Falta la referencia" });
  }

  try {
    const findUsuario = await UsuariosModel.findById(_id).populate(
      "licencia.referidoPor"
    );

    if (!findUsuario) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    const findProducto = await productoModel.findById(producto);

    if (!findProducto) {
      return res.status(404).json({ message: "Producto no encontrado" });
    }

    const fechaMexico = moment().tz("America/Mexico_City");

    findUsuario.licencia = {
      ...findUsuario.licencia, // Mantiene las propiedades existentes en licencia
      producto, // Sobrescribe el producto
      vigencia: fechaMexico.add(findProducto.cantidad, "years"), // Actualiza la vigencia
      comision: findProducto.utilidad, // Actualiza la comisión
    };

    findUsuario.estatus = "Activo";

    await findUsuario.save();

    // falta mandar su contrato

    const urlAjunto = buildFileUri(`productos/${findProducto.adjunto}`);

    const passwordSin = desencriptar(findUsuario.password);

    const htmlRegistro = RegistroUsuarioHTML(
      findUsuario.nombre,
      findUsuario.email,
      passwordSin,
      `${DISTRIBUIDOR_APP}/login`
    );

    const files = [
      {
        filename: "Contrato.pdf",
        path: urlAjunto,
      },
    ];

    sendMail(htmlRegistro, "NILDDA: Envió Accesos", findUsuario.email);
    sendMail(htmlRegistro, "Contrato Distribuidor", findUsuario.email, files);

    sendWhatsappRecovery({
      to: findUsuario.telefono,
      correo: findUsuario.email,
      password: passwordSin,
      url: DISTRIBUIDOR_APP,
    });

    // venta de producto

    const findDistribuidor = await UsuariosModel.findById(
      findUsuario.licencia.referidoPor
    ).select("licencia");

    let licencia;
    if (findDistribuidor) {
      licencia = findDistribuidor.licencia;
    }

    const newVenta = new Ventas({
      despacho: null,
      usuario: findUsuario._id,
      producto: findProducto._id,
      importe: findProducto.precio,
      tipoReferencia,
      referencia,
      estatus: "Pagado",
      distribuidor: findDistribuidor
        ? {
            comision: licencia.comision,
            distribuidor: findDistribuidor._id,
            importe: (findProducto.precio * licencia.comision) / 100,
          }
        : {
            comision: 0,
            distribuidor: null,
            importe: 0,
          },
    });

    await VentasSchema.create(newVenta);

    res.status(201).json({ message: "Usuario creado", data: findUsuario });
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

const generateDistribuidorRegistroPruebas = async (req, res) => {
  const { nombreCompleto, correo, telefono } = req.body;

  if (!nombreCompleto) {
    return res.status(400).json({ message: "Falta el nombre" });
  }

  if (!correo) {
    return res.status(400).json({ message: "Falta el correo" });
  }

  if (!telefono) {
    return res.status(400).json({ message: "Falta el telefono" });
  }

  try {
    // buscar si no existe el correo
    // const findEmailExist = await UsuariosModel.findOne({
    //   email: correo
    // });

    // if (findEmailExist) {
    //   return res.status(400).json({ message: 'El correo ya existe' });
    // }

    const passwordSin = generatePassword();

    const passwordEncrypt = encriptar(passwordSin);

    const findTipoUsuarioDistribuidor = await TipoUsuarioModel.findOne({
      tipo: "distribuidor",
    }).select("_id");

    // const producto20 = '66c7b0deb8586f9830adb34c';
    const producto30 = "66c7c834b8586f9830adb34d";
    // const producto40 = '66c7c8f8b8586f9830adb34e';

    // const vigencia1 = moment().tz('America/Mexico_City').add(1, 'years');
    const vigencia2 = moment().tz("America/Mexico_City").add(2, "years");
    // const vigencia3 = moment().tz('America/Mexico_City').add(3, 'years');

    const objUser = {
      nombre: nombreCompleto,
      email: correo,
      telefono,
      password: passwordEncrypt,
      estatus: "Activo",
      tipoUsuarios: {
        distribuidor: findTipoUsuarioDistribuidor._id,
      },
      licencia: {
        producto: producto30,
        vigencia: vigencia2,
        comision: 30,
      },
    };

    const userSave = await UsuariosModel.create(objUser);

    // const htmlRegistro = RegistroUsuarioHTML(nombreCompleto, correo, passwordSin, `${DISTRIBUIDOR_APP}/login`);

    // sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correo);

    sendWhatsappRecovery({
      to: "2213425514",
      correo,
      password: passwordSin,
      url: DISTRIBUIDOR_APP,
    });

    res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

const generateDistribuidorRegistro = async (req, res) => {
  const { nombreCompleto, correo, telefono, producto } = req.body;

  if (!nombreCompleto) {
    return res.status(400).json({ message: "Falta el nombre" });
  }

  if (!correo) {
    return res.status(400).json({ message: "Falta el correo" });
  }

  if (!telefono) {
    return res.status(400).json({ message: "Falta el telefono" });
  }

  if (!producto) {
    return res.status(400).json({ message: "Falta el producto" });
  }

  try {
    // buscar si no existe el correo
    const findEmailExist = await UsuariosModel.findOne({
      email: correo,
    });

    if (findEmailExist) {
      return res.status(400).json({ message: "El correo ya existe" });
    }

    const passwordSin = generatePassword();

    const passwordEncrypt = encriptar(passwordSin);

    const findTipoUsuarioDistribuidor = await TipoUsuarioModel.findOne({
      tipo: "distribuidor",
    }).select("_id");

    const findProducto = await productoModel.findById(producto);

    if (!findProducto) {
      return res.status(404).json({ message: "Producto no encontrado" });
    }

    const fechaMexico = moment().tz("America/Mexico_City");
    const objUser = {
      nombre: nombreCompleto,
      email: correo,
      telefono,
      password: passwordEncrypt,
      estatus: "Prospecto",
      tipoUsuarios: {
        distribuidor: findTipoUsuarioDistribuidor._id,
      },
      licencia: {
        producto,
        vigencia: fechaMexico.add(findProducto.cantidad, "years"),
        comision: findProducto.utilidad,
      },
    };

    const userSave = await UsuariosModel.create(objUser);

    sendWhatsappRecovery({
      to: telefono,
      correo,
      password: passwordSin,
      url: DISTRIBUIDOR_APP,
    });

    const htmlRegistro = RegistroUsuarioHTML(
      nombreCompleto,
      correo,
      passwordSin,
      `${DISTRIBUIDOR_APP}/login`
    );

    sendMail(htmlRegistro, "NILDDA: Envió Accesos", correo);

    res.status(201).json({ message: "Usuario creado", data: userSave });
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

module.exports = {
  login,
  register,
  rememberPassword,
  actualizarUsuario,
  obtenerUsuarios,
  createUsuarioDespacho,
  obtenerUsuario,
  updateUsuario,
  deleteUsuario,
  cargaUsuario,
  loginCliente,
  obtenerUsuariosSinPaginar,
  obtenerInformacioDistribuidor,
  actualizarRetirarFondos,
  findDistribuidorForClave,
  generateDistribuidorRegistroProspecto,
  findDistribuidorForClaveProspecto,
  ConvertirProspectoDistribuidor,
  generateDistribuidorRegistro,
  generateDistribuidorRegistroPruebas,
  linkGoogle,
  loginGoogle,
};
