/* eslint-disable indent */
const axios = require('axios');
const moment = require('moment');

const obtenerTesis = async (req, res) => {
  const { page = 0, size = 20 } = req.query;

  const { expression, fields = [], epocas = [], instancias = [], tipoTesis = [] } = req.body;

  if (fields.length === 0) {
    return res.status(400).json({
      message: 'Es obligatorio un campo seleccionado de campos de búsqueda'
    });
  }

  if (epocas.length === 0) {
    return res.status(400).json({
      message: 'Es obligatorio un campo de época seleccionado'
    });
  }

  if (instancias.length === 0) {
    return res.status(400).json({
      message: 'Es obligatorio un campo de instancia seleccionado'
    });
  }

  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis?page=${page}&size=${size}`;

    const fieldsUser = fields
      .map((field) => {
        switch (field) {
          case 'localizacionBusqueda':
            return 'Localización:';
          case 'rubro':
            return 'Rubro (título y subtítulo):';
          case 'texto':
            return 'Texto:';
          case 'precedentes':
            return 'Precedentes:';
          default:
            return field;
        }
      })
      .join(' \n');

    const fieldsText = fields
      .map((field) => {
        switch (field) {
          case 'localizacionBusqueda':
            return 'Localización';
          case 'rubro':
            return 'Rubro (título y subtítulo)';
          case 'texto':
            return 'Texto';
          case 'precedentes':
            return 'Precedentes';
          default:
            return field;
        }
      })
      .join(', ');

    const body = {
      filterExpression: '',
      classifiers: [
        {
          name: 'idEpoca',
          value: epocas, // 200-> 11a.Época , 100 ->10a.Época, 5 -> 9a.Época, 4 ->8a.Época ,3 ->7a.Época,2 ->6a.Época, 1 ->5a.Época
          allSelected: false,
          visible: false,
          isMatrix: false
        },
        {
          name: 'numInstancia',
          value: instancias, // 6 - > Pleno, 1 -> 1a.Sala, 2 -> 2a.Sala, 3 -> 3a.Sala, 4 -> 4a.Sala, 5 -> S.Auxiliar, 60 -> Plenos Reg, 50 -> Plenos CTO, 7 ->TCC
          allSelected: false,
          visible: false,
          isMatrix: false
        },
        {
          name: 'tipoDocumento',
          value: ['1'],
          allSelected: false,
          visible: false,
          isMatrix: false
        },
        tipoTesis[0] !== '2'
          ? {
            name: 'idTipoTesis',
            value: tipoTesis, // 1 -> Jurisprudencia,  0 ->Aislada        Solo recibir un valor, group radio buttons
            allSelected: false,
            visible: false,
            isMatrix: false
          }
          : undefined
      ].filter(Boolean),
      searchTerms: [
        {
          expression,
          fields,
          fieldsUser,
          fieldsText,
          operator: 0,
          operatorUser: 'Y',
          operatorText: 'Y',
          lsFields: [],
          esInicial: true,
          esNRD: false
        }
      ],
      bFacet: true,
      ius: [],
      idApp: 'SJFAPP2020'
      //   lbSearch: [
      //     "Todo",
      //     // "11a. Época - Todas las Instancias",
      //     // "10a. Época - Todas las Instancias",
      //     // "9a. Época - Todas las Instancias",
      //   ],
      //   lbSearch: lbSearch,
    };

    const response = await axios.post(uri, body);

    response?.data?.documents?.forEach((doc) => {
      doc.fechaPublicacion = doc?.fechaPublicacion ? moment(doc?.fechaPublicacion)?.format('DD/MM/YYYY HH:mm A') : '';
    });

    return res.status(200).json(response.data);
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener las tesis',
      error: error.message
    });
  }
};

const obtenerTesisByID = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id ' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/${id}?isSemanal=true&hostName=https://sjf2.scjn.gob.mx`;
    const response = await axios.get(uri);
    return res.status(200).json(response.data);
  } catch (error) {
    try {
      const uri = `https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/${id}?hostName=https://sjf2.scjn.gob.mx`;
      const response = await axios.get(uri);
      return res.status(200).json(response.data);
    } catch (error) {
      return res.status(500).json({
        message: 'Tesis no encontrada',
        error: error.message
      });
    }
  }
};

const descargarTesisById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/reporte/${id}?nameDocto=Tesis&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`;
    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    res.setHeader('Content-Disposition', `attachment; filename="tesis_${id}.pdf"`);
    res.setHeader('Content-Type', 'application/pdf');
    res.status(200).send(response.data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener las tesis',
      error: error.message
    });
  }
};

const imprimirTesisById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/reporte/${id}?nameDocto=Tesis&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`;
    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    res.setHeader('Content-Disposition', `inline; filename="tesis_${id}.pdf"`);
    res.setHeader('Content-Type', 'application/pdf');
    res.status(200).send(response.data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener las tesis',
      error: error.message
    });
  }
};

const obtenerVotosByID = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id ' });
  }
  try {
    console.log('hola');
    const uri = `https://sjf2.scjn.gob.mx/services/sjfvotosmicroservice/api/public/votos/${id}?hostName=https://sjf2.scjn.gob.mx`;
    const response = await axios.get(uri);
    return res.status(200).json(response.data);
  } catch (error) {
    return res.status(500).json({
      message: 'Ocurrió un error al obtener las tesis',
      error: error.message
    });
  }
};

const descargarVotosById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjfvotosmicroservice/api/public/votos/reporte/${id}?nameDocto=Voto&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`;
    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    res.setHeader('Content-Disposition', `attachment; filename="votos_${id}.pdf"`);
    res.setHeader('Content-Type', 'application/pdf');
    res.status(200).send(response.data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener los votos',
      error: error.message
    });
  }
};

const imprimirVotosById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjfvotosmicroservice/api/public/votos/reporte/${id}?nameDocto=Voto&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`;
    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    res.setHeader('Content-Disposition', `inline; filename="votos_${id}.pdf"`);
    res.setHeader('Content-Type', 'application/pdf');
    res.status(200).send(response.data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener los votos',
      error: error.message
    });
  }
};

const obtenerPrecedenteBydId = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://sjf2.scjn.gob.mx/services/sjfejecutoriamicroservice/api/public/ejecutorias/${id}?hostName=https://sjf2.scjn.gob.mx`;
    const response = await axios.get(uri);
    return res.status(200).json(response.data);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      message: 'Ocurrió un error al obtener los precedentes',
      error: error.message
    });
  }
};

const descargarPrecedenteById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }

  try {
    const response = await axios.post(
      `https://sjf2.scjn.gob.mx/services/sjfejecutoriamicroservice/api/public/ejecutorias/reportedetalle/${id}?nameDocto=Precedente%20(Sentencia)&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`,
      [],
      {
        responseType: 'arraybuffer', // Configura el tipo de respuesta como arraybuffer
        referrer: 'https://sjf2.scjn.gob.mx/detalle/ejecutoria/32426',
        mode: 'cors'
      }
    );

    // Configura el encabezado de respuesta adecuado para archivos binarios
    res.set('Content-Type', 'application/pdf'); // Ajusta el tipo MIME según el tipo de archivo esperado
    res.set('Content-Disposition', `attachment; filename="precedente-${id}.pdf"`); // Nombre del archivo descargado
    res.send(response.data); // Envía el archivo binario al cliente
  } catch (error) {
    console.error(error);
    res.status(500).json({
      message: 'Ocurrió un error al obtener los precedentes',
      error: error.message
    });
  }
};

const imprimirPrecedenteById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }

  try {
    const response = await axios.post(
      `https://sjf2.scjn.gob.mx/services/sjfejecutoriamicroservice/api/public/ejecutorias/reportedetalle/${id}?nameDocto=Precedente%20(Sentencia)&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020`,
      [],
      {
        responseType: 'arraybuffer', // Configura el tipo de respuesta como arraybuffer
        referrer: 'https://sjf2.scjn.gob.mx/detalle/ejecutoria/32426',
        mode: 'cors'
      }
    );

    // Configura el encabezado de respuesta adecuado para archivos binarios
    res.set('Content-Type', 'application/pdf'); // Ajusta el tipo MIME según el tipo de archivo esperado
    res.set('Content-Disposition', `inline; filename="precedente-${id}.pdf"`); // Nombre del archivo descargado
    res.send(response.data); // Envía el archivo binario al cliente
  } catch (error) {
    console.error(error);
    res.status(500).json({
      message: 'Ocurrió un error al obtener los precedentes',
      error: error.message
    });
  }
};

module.exports = {
  obtenerTesisByID,
  obtenerTesis,
  descargarTesisById,
  imprimirTesisById,
  obtenerVotosByID,
  descargarVotosById,
  imprimirVotosById,
  obtenerPrecedenteBydId,
  descargarPrecedenteById,
  imprimirPrecedenteById
};
