const DespachoModel = require('../models/Despachos.js');
const UsuariosModel = require('../models/Usuarios.js');
const { buildFileUri } = require('../config/s3.js');
const moment = require('moment-timezone');

const getReferidos = async (req, res) => {
  const { distribuidor } = req.params;
  const { estatus, page = 1, search, plan } = req.query;

  if (plan) {
    if (plan !== 'Con' && plan !== 'Sin' && plan !== 'Todos') {
      return res.status(400).json({ message: 'El parámetro plan debe ser "Con", "Sin" o "Todos"' });
    }
  }

  const options = {
    page,
    limit: 20,
    sort: {
      'licencia.paquete': 1,
      'contadorExp.vigencia': 1
    },
    populate: [
      {
        path: 'contadorExp.paquete',
        select: 'nombre imagen'
      }
    ]
  };

  const optionsReferidos = {
    page,
    limit: 20,
    sort: {
      'licencia.producto': 1,
      'licencia.vigencia': 1
    },
    populate: [
      {
        path: 'licencia.producto',
        select: 'nombre imagen'
      }
    ]
  };

  const query = {
    distribuidor
  };

  const queryReferidos = {
    'licencia.referidoPor': distribuidor
  };

  if (estatus) {
    query.estatus = estatus;
    queryReferidos.estatus = estatus;
  }

  if (search) {
    query.$or = [
      { nombre: { $regex: search, $options: 'i' } }
    ];

    queryReferidos.$or = [
      { nombre: { $regex: search, $options: 'i' } }
    ];
  }

  if (plan) {
    if (plan === 'Con') {
      query.$and = [
        { 'contadorExp.paquete': { $exists: true } },
        { 'contadorExp.vigencia': { $gte: new Date() } }
      ];

      queryReferidos.$and = [
        { 'licencia.producto': { $exists: true } },
        { 'licencia.vigencia': { $gte: new Date() } }
      ];
    } else if (plan === 'Sin') {
      query.$or = [
        { 'contadorExp.paquete': null },
        { 'contadorExp.vigencia': { $lt: new Date() } }
      ];

      queryReferidos.$or = [
        { 'licencia.producto': null },
        { 'licencia.vigencia': { $lt: new Date() } }
      ];
    }
  }

  try {
    // Realizar la búsqueda paginada de despachos y usuarios referidos
    const [findDespachos, findUsuariosReferidos] = await Promise.all([
      DespachoModel.paginate(query, options),
      UsuariosModel.paginate(queryReferidos, optionsReferidos)
    ]);

    const resuls = [];

    // Procesar los despachos
    for (const despacho of findDespachos.docs) {
      const { contadorExp } = despacho;
      const { vigencia } = contadorExp;
      const fechaMexico = moment().tz('America/Mexico_City').format('YYYY-MM-DD HH:mm:ss');

      const diasRestantes = moment(vigencia).diff(fechaMexico, 'days');

      resuls.push({
        _id: despacho._id,
        nombre: despacho.nombre,
        direccion: despacho.direccion,
        estado: despacho.estado,
        estatus: despacho.estatus,
        logo: despacho.logo ? buildFileUri(`despachos/${despacho._id}/logo/${despacho.logo}`) : buildFileUri('default/icono_usuario_100x100_04.jpg'),
        diasRestantes,
        correo: despacho.correo,
        telefono: despacho.telefono,
        // producto: contadorExp?.paquete?.nombre || '',
        producto: {
          nombre: contadorExp?.paquete?.nombre || '',
          imagen: contadorExp?.paquete?.imagen ? buildFileUri(`productos/${contadorExp.paquete.imagen}`) : buildFileUri('default/icono_usuario_100x100_04.jpg')
        },
        tipo: 'Despacho'
      });
    }

    // Procesar los usuarios referidos
    for (const usuario of findUsuariosReferidos.docs) {
      const { licencia } = usuario;
      const { vigencia } = licencia;
      const fechaMexico = moment().tz('America/Mexico_City').format('YYYY-MM-DD HH:mm:ss');

      const diasRestantes = moment(vigencia).diff(fechaMexico, 'days');

      resuls.push({
        _id: usuario._id,
        nombre: usuario.nombre,
        direccion: usuario.direccion,
        estado: usuario.estado,
        estatus: usuario.estatus,
        diasRestantes,
        correo: usuario.correo,
        telefono: usuario.telefono,
        logo: usuario.logo ? buildFileUri(`usuarios/${usuario._id}/logo/${usuario.logo}`) : buildFileUri('default/icono_usuario_100x100_04.jpg'),
        producto: {
          nombre: licencia?.producto?.nombre ? `Licencia ${licencia?.producto?.nombre}` : '',
          imagen: licencia?.producto?.imagen ? buildFileUri(`productos/${licencia.producto.imagen}`) : buildFileUri('default/icono_usuario_100x100_04.jpg')
        },
        tipo: 'Distribuidor'
      });
    }

    // Enviar la respuesta
    return res.status(200).json({
      docs: resuls,
      totalDocs: findDespachos.totalDocs + findUsuariosReferidos.totalDocs,
      limit: findDespachos.limit + findUsuariosReferidos.limit,
      totalPages: findDespachos.totalPages + findUsuariosReferidos.totalPages,
      page: findDespachos.page + findUsuariosReferidos.page,
      pagingCounter: findDespachos.pagingCounter + findUsuariosReferidos.pagingCounter,
      hasPrevPage: findDespachos.hasPrevPage || findUsuariosReferidos.hasPrevPage,
      hasNextPage: findDespachos.hasNextPage || findUsuariosReferidos.hasNextPage,
      prevPage: findDespachos.prevPage || findUsuariosReferidos.prevPage,
      nextPage: findDespachos.nextPage || findUsuariosReferidos.nextPage
    });
  } catch (error) {
    console.error('Error:', error);
    return res.status(500).json({ message: 'Error al obtener los despachos y referidos', error });
  }
};

module.exports = {
  getReferidos
};
