const RecursosIncidenciasModel = require('../models/RecursosIncidencias.js');

const getRecursosIncidenciasMateria = async (req, res) => {
  const { materia } = req.params;
  if (!materia) {
    return res.status(400).json({ message: 'No se ha enviado la materia' });
  }

  try {
    const query = {
      materia,
      estatus: 'Activo'
    };

    const recursosIncidencias = await RecursosIncidenciasModel.find(query);

    res.status(201).json({
      message: 'Recursos incidencias encontrados',
      data: recursosIncidencias
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getRecursosOrIncidentes = async (req, res) => {
  const { estatus, tipo, page = 1, search, materia } = req.query;
  const options = {
    page,
    limit: 20,
    sort: { materia: 1, nombre: 1 }, // Ordena por 'materia.nombre' en orden ascendente
    populate: {
      path: 'materia',
      select: 'nombre'
    }
  };

  try {
    const query = {
      tipo
    };

    if (materia) {
      query.materia = materia;
    }

    if (search) {
      query.nombre = { $regex: search, $options: 'i' };
    }

    if (estatus) {
      query.estatus = estatus;
    }

    const recursos = await RecursosIncidenciasModel.paginate(query, options);

    res.status(200).json(recursos);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const getRecursoOrIncidenciaById = async (req, res) => {
  const { id } = req.params;

  if (!id) {
    return res.status(400).json({ message: 'El id es obligatorio' });
  }

  try {
    const recurso = await RecursosIncidenciasModel.findById(id);

    console.log(recurso);
    res.status(200).json(recurso);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const updateRecursoOrIncidencia = async (req, res) => {
  const { nombre, estatus, materia } = req.body;
  const { id } = req.params;

  if (!id) {
    return res.status(404).json({ message: 'El id es requerido' });
  }

  if (!nombre) {
    return res.status(404).json({ message: 'El nombre es requerido' });
  }

  try {
    const recurso = await RecursosIncidenciasModel.findById(id);

    if (!recurso) {
      return res.status(404).json({ message: 'Recurso no encontrado' });
    }

    recurso.nombre = nombre;
    recurso.materia = materia || recurso.materia;
    recurso.estatus = estatus || 'Activo';

    const updateRecurso = await recurso.save();
    res.status(200).json(updateRecurso);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const newRecursoIncidencia = async (req, res) => {
  const { nombre, estatus, tipo, materia } = req.body;

  if (!nombre) {
    return res.status(404).json({ message: 'El nombre es requerido' });
  }

  try {
    const data = {
      nombre,
      materia,
      tipo,
      estatus: estatus || 'Activo'
    };

    const newRecurso = RecursosIncidenciasModel.create(data);
    res.status(201).json(newRecurso);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const deleteRecursoOrIncidencia = async (req, res) => {
  const { id } = req.params;

  try {
    await RecursosIncidenciasModel.findByIdAndRemove(id);
    res.status(200).json({ message: 'Eliminado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getRecursosIncidenciasMateria,
  getRecursosOrIncidentes,
  getRecursoOrIncidenciaById,
  updateRecursoOrIncidencia,
  newRecursoIncidencia,
  deleteRecursoOrIncidencia
};
