const RecursosModel = require('../models/RecursosDespacho.js');

const getRecursos = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search, materia } = req.query;

  const options = {
    page,
    limit: 15,
    sort: { materia: 1, nombre: 1 },
    populate: 'materia'
  };

  const query = {
    despacho
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.nombre = { $regex: search, $options: 'i' };
  }

  if (materia) {
    query.materia = materia;
  }

  try {
    const recursos = await RecursosModel.paginate(query, options);
    res.status(200).json({ recursos });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getRecursosSinPaginar = async (req, res) => {
  const { despacho } = req.params;
  const { materia } = req.query;

  try {
    const query = {
      despacho
    };

    if (materia) {
      query.materia = materia;
    }

    query.estatus = 'Activo';

    const materiasFind = await RecursosModel.find(query).sort({ nombre: 1 });

    const recursos = materiasFind.map(materia => {
      return {
        value: materia._id,
        label: materia.nombre,
        key: materia._id
      };
    });

    res.status(200).json({ recursos });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createRecurso = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, estatus, materia } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!materia) {
    return res.status(400).json({ message: 'La materia es requerida' });
  }

  const obj = {
    despacho,
    nombre,
    materia,
    estatus: estatus || 'Activo'
  };

  try {
    const recurso = await RecursosModel.create(obj);
    res.status(201).json({ recurso, message: 'Recurso creada correctamente' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getRecursoById = async (req, res) => {
  const { despacho, id } = req.params;

  try {
    const recurso = await RecursosModel.findOne({ _id: id, despacho });

    if (!recurso) {
      return res.status(404).json({ message: 'No se encontró el recurso' });
    }

    res.status(200).json(recurso);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateRecurso = async (req, res) => {
  const { id, despacho } = req.params;
  const { nombre, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  try {
    const recurso = await RecursosModel.findOne({ _id: id, despacho });

    if (!recurso) {
      return res.status(404).json({ message: 'No se encontró el recurso' });
    }

    const update = {
      nombre,
      estatus: estatus || 'Activo'
    };

    await RecursosModel.findOneAndUpdate({ _id: id, despacho }, update, { new: true });

    res.status(200).json({ message: 'Recurso actualizado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteRecurso = async (req, res) => {
  const { id, despacho } = req.params;

  try {
    const recurso = await RecursosModel.findOne({ _id: id, despacho });

    if (!recurso) {
      return res.status(404).json({ message: 'No se encontró el recurso' });
    }

    await RecursosModel.findOneAndDelete({ _id: id, despacho });

    return res.status(200).json({ message: 'Recurso eliminado' });
  } catch (error) {
    res.status(404).json({ message: 'Ocurrió un error inténtelo de nuevo.' });
  }
};

module.exports = {
  getRecursos,
  getRecursosSinPaginar,
  createRecurso,
  getRecursoById,
  updateRecurso,
  deleteRecurso
};
