const ProspectoModel = require('../models/ProspectosDespacho.js');

const getProspectos = async (req, res) => {
  const { despacho } = req.params;
  const { page = 1, search } = req.query;

  const options = {
    page,
    limit: 10,
    sort: { nombre: 1 }
  };

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  const query = {
    despacho
  };

  if (search) {
    query.$or = [
      { nombre: { $regex: search, $options: 'i' } },
      { correo: { $regex: search, $options: 'i' } },
      { telefono: { $regex: search, $options: 'i' } }
    ];
  }

  try {
    const findProspecto = await ProspectoModel.paginate(query, options);

    res.status(200).json({ prospectos: findProspecto });
  } catch (error) {
    res.status(404).json({ message: 'No se encontraron prospectos', error: error.message });
  }
};

const createProspecto = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, correo, telefono } = req.body;

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del prospecto es requerido' });
  }

  try {
    const objProspecto = {
      despacho,
      nombre,
      correo,
      telefono
    };

    const newProspecto = await ProspectoModel.create(objProspecto);

    res.status(201).json({
      message: 'Prospecto creado correctamente',
      prospecto: newProspecto
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getProspecto = async (req, res) => {
  const { id } = req.params;

  try {
    const prospecto = await ProspectoModel.findById(id);

    if (!prospecto) {
      return res.status(404).json({ message: 'Prospecto no encontrado' });
    }

    res.status(200).json({ prospecto });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateProspecto = async (req, res) => {
  const { id } = req.params;
  const { nombre, correo, telefono } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del prospecto es requerido' });
  }

  try {
    const objProspecto = {
      nombre,
      correo,
      telefono
    };

    const updatedProspecto = await ProspectoModel.findByIdAndUpdate(id, objProspecto, { new: true });

    if (!updatedProspecto) {
      return res.status(404).json({ message: 'Prospecto no encontrado' });
    }

    res.status(200).json({
      message: 'Prospecto actualizado correctamente',
      prospecto: updatedProspecto
    });
  } catch (error) {
    console.error(error);

    res.status(409).json({ message: error.message });
  }
};

const deleteProspecto = async (req, res) => {
  const { id } = req.params;

  try {
    await ProspectoModel.findByIdAndRemove(id);
    res.status(200).json({ message: 'Prospecto eliminado correctamente' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getProspectoSinPaginar = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  try {
    const findProspecto = await ProspectoModel.find({ despacho }).sort({ nombre: 1 });

    const prospectos = findProspecto.map((prospecto) => {
      return {
        value: prospecto._id,
        label: prospecto.nombre,
        correo: prospecto.correo
      };
    });

    res.status(200).json({ prospectos });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getProspectos,
  createProspecto,
  getProspecto,
  updateProspecto,
  deleteProspecto,
  getProspectoSinPaginar

};
