const {
  formatDateTime,
  generatePassword,
  encriptar,
  formatDate,
} = require("../config/functionsGlobal.js");

const { buildFileUri } = require("../config/s3.js");
const ContadorProspecto = require("../models/ContadorProspecto.js");
const UsuariosModel = require("../models/Usuarios.js");
const moment = require("moment-timezone");
const { formatDistanceToNow } = require("date-fns");
const { es } = require("date-fns/locale");
const TipoUsuarios = require("../models/TipoUsuarios.js");

const getProspectos = async (req, res) => {
  const { estatus, page = 1, search } = req.query;
  const { distribuidor } = req.params;

  const options = {
    page,
    limit: 20,
    sort: {
      _id: -1,
      nombre: 1,
    },
    populate: [
      {
        path: "licencia.producto",
        select: "nombre",
      },
      // {
      //   path: "tipoUsuarios.despacho",
      //   select: "nombre",
      // },
    ],
  };

  const query = {
    "licencia.referidoPor": distribuidor,
    tipo: { $in: ["distribuidor", "despacho"] },
    estatus: "Prospecto",
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: "i" } }];
  }
  try {
    const data = await UsuariosModel.paginate(query, options);
    const locale = es;
    data.docs = await Promise.all(
      data.docs.map(async (element) => {
        element.tipo = capitalizeFirstLetter(element.tipo);
        const fechaCreated = formatDate(element?.fechaCreacion);

        const whatsAppCount = await ContadorProspecto.findOne({
          prospecto: element._id,
          distribuidor: distribuidor,
          tipo: "WhatsApp",
        }).select("contador ultimoContacto");

        const correoCount = await ContadorProspecto.findOne({
          prospecto: element._id,
          distribuidor: distribuidor,
          tipo: "Correo",
        }).select("contador ultimoContacto");

        return {
          ...element._doc,
          fechaCreacion: fechaCreated,
          whatsApp: {
            count: whatsAppCount?.contador || 0,
            ultimoContacto: whatsAppCount?.ultimoContacto
              ? formatDistanceToNow(whatsAppCount?.ultimoContacto, {
                  locale: locale,
                })
              : null,
          },
          correo: {
            count: correoCount?.contador || 0,
            ultimoContacto: correoCount?.ultimoContacto
              ? formatDistanceToNow(correoCount?.ultimoContacto, {
                  locale: locale,
                })
              : null,
          },
        };
      })
    );

    return res.status(200).json(data);
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al obtener los prospectos", error });
  }
};

const createProspecto = async (req, res) => {
  const { distribuidor } = req.params;
  const { nombre, correo, telefono, tipo } = req.body;

  try {
    const existe = await UsuariosModel.findOne({ email: correo });
    if (existe) {
      return res.status(400).json({ message: "El correo ya está en uso" });
    }

    const password = generatePassword();
    const passwordEncriptada = encriptar(password);

    // const findTipoUsuario = await TipoUsuarios.findOne({ tipo: tipo });

    const usuario = UsuariosModel({
      nombre: nombre,
      email: correo,
      telefono: telefono,
      password: passwordEncriptada,
      tipo: tipo,
      // tipoUsuarios: { [tipo]: findTipoUsuario._id },
      estatus: "Prospecto",
      licencia: {
        referidoPor: distribuidor,
      },
      fechaCreacion: new Date(),
    });

    const result = await usuario.save();

    if (!result) {
      return res.status(400).json({ message: "Prospecto no guardado" });
    }
    return res.status(200).json({ message: "Prospecto guardado" });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al crear el prospecto", error });
  }
};

const updateProspecto = async (req, res) => {
  const { distribuidor } = req.params;
  const { id, nombre, correo, telefono, tipo } = req.body;

  try {
    const prospecto = await UsuariosModel.findById(id);

    if (!prospecto) {
      return res.status(404).json({ message: "Prospecto no encontrado" });
    }

    if (nombre) prospecto.nombre = nombre;
    if (correo) prospecto.email = correo;
    if (telefono) prospecto.telefono = telefono;
    if (tipo) prospecto.tipo = tipo;

    await prospecto.save();

    return res.status(200).json({ message: "Prospecto actualizado" });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al actualizar el prospecto", error });
  }
};

const getProspectoById = async (req, res) => {
  const { id } = req.params;

  try {
    const prospecto = await UsuariosModel.findById(id);

    if (!prospecto) {
      return res.status(404).json({ message: "Prospecto no encontrado" });
    }

    return res.status(200).json(prospecto);
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al encontrar el prospecto", error });
  }
};

const deleteProspecto = async (req, res) => {
  const { id } = req.params;

  try {
    const prospecto = await UsuariosModel.findByIdAndDelete(id);

    if (!prospecto) {
      return res.status(404).json({ message: "Prospecto no encontrado" });
    }

    const resultado = await ContadorProspecto.find({
      prospecto: id,
    }).select("_id ");

    if (resultado.length > 0) {
      resultado.forEach((e) => {
        e.remove();
      });
    }

    return res
      .status(200)
      .json({ message: "Prospecto eliminado correctamente" });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al eliminar el prospecto", error });
  }
};

const addCountProspecto = async (req, res) => {
  const { distribuidor } = req.params;
  const { tipo, usuario } = req.query;

  try {
    let contadorProspecto = await ContadorProspecto.findOne({
      prospecto: usuario,
      distribuidor: distribuidor,
      tipo: tipo,
    });

    if (contadorProspecto) {
      contadorProspecto.contador += 1;
      contadorProspecto.ultimoContacto = new Date();
    } else {
      contadorProspecto = new ContadorProspecto({
        prospecto: usuario,
        distribuidor: distribuidor,
        tipo: tipo,
        contador: 1,
        ultimoContacto: new Date(),
      });
    }

    await contadorProspecto.save();

    return res.status(200).json({ message: "Contador actualizado" });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: "Error al agregar el prospecto", error });
  }
};

const capitalizeFirstLetter = (string) => {
  if (!string) return ""; // Si la cadena está vacía o indefinida
  return string.charAt(0).toUpperCase() + string.slice(1);
};

module.exports = {
  getProspectos,
  addCountProspecto,
  createProspecto,
  updateProspecto,
  getProspectoById,
  deleteProspecto,
};
