const { sendWhatsappRecovery, desencriptar, sendWhatsappRecovery2 } = require('../config/functionsGlobal');
const { sendMail } = require('../config/mail');
const { RegistroUsuarioHTML } = require('../Mail/RegistroUsuarioHTML');
const TipoUsuarios = require('../models/TipoUsuarios');
const Usuarios = require('../models/Usuarios');

const urlAPP = process.env.DISTRIBUIDOR_APP;

const obtenerPromotores = async (req, res) => {
  const { estatus, page = 1, search } = req.query;
  const { distribuidor } = req.params;

  const options = {
    page: parseInt(page, 10),
    limit: 20,
    sort: { _id: -1, nombre: 1 },
    populate: {
      path: 'tipoUsuarios.promotor'
    }
  };

  const query = {
    'licencia.referidoPor': distribuidor
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
  }

  try {
    const data = await Usuarios.paginate(query, options);

    const filteredDocs = data.docs.filter(
      (doc) =>
        doc.tipoUsuarios.promotor &&
        doc.tipoUsuarios.promotor.tipo === 'promotor'
    );

    return res.status(200).json({ ...data, docs: filteredDocs });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener los promotores',
      error
    });
  }
};

const crearPromotor = async (req, res) => {
  const { nombre, apellidoP, correo, celular, distribuidor, comision } =
    req.body;

  if (!nombre || !apellidoP || !correo || !celular) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }
  try {
    const existe = await Usuarios.findOne({ email: correo });

    if (existe) {
      return res.status(400).json({ message: 'El correo ya está en uso' });
    }

    const dataDistribuidor = await Usuarios.findById(distribuidor);

    if (comision > dataDistribuidor.licencia.comision) {
      return res.status(400).json({
        message: `La comisión no puede ser mayor que el ${dataDistribuidor.licencia.comision}%`
      });
    }
    // const password = generatePassword();
    // const passwordEncriptada = encriptar(password);

    const tipoUsuario = await TipoUsuarios.findOne({ tipo: 'promotor' });

    const usuario = Usuarios({
      nombre,
      apellidoPaterno: apellidoP,
      email: correo,
      telefono: celular,
      // password: passwordEncriptada,
      tipoUsuarios: {
        promotor: tipoUsuario._id
      },
      estatus: 'Activo',
      licencia: {
        referidoPor: dataDistribuidor._id,
        producto: dataDistribuidor.licencia.producto || undefined,
        comision,
        vigencia: dataDistribuidor.licencia.vigencia || undefined
      },
      fechaCreacion: new Date()
    });

    const result = await usuario.save();

    sendWhatsappRecovery2({
      idDespacho: null,
      nombreUsuario: nombre,
      to: celular,
      correo,
      url: urlAPP + `/password/${result?._id || ''}`
    });

    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      correo,
      '',
      `${urlAPP}/password/${result?._id || ''}`
    );

    sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correo);

    if (!result) {
      return res.status(400).json({ message: 'Error al crear el promotor' });
    }

    return res.status(200).json({ message: 'Promotor creado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al crear al promotor',
      error
    });
  }
};

const obtenerPromotorPorId = async (req, res) => {
  const { id } = req.params;

  try {
    const promotor = await Usuarios.findById(id).populate({
      path: 'tipoUsuarios.promotor'
    });

    if (!promotor) {
      return res.status(404).json({ mensaje: 'Promotor no encontrado' });
    }

    return res.status(200).json(promotor);
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener el promotor',
      error
    });
  }
};

const actualizarPromotor = async (req, res) => {
  const { id } = req.params;
  const { nombre, apellidoP, correo, celular, distribuidor, comision, estatus } =
    req.body;

  if (!nombre || !apellidoP || !correo || !celular) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }

  try {
    const promotor = await Usuarios.findById(id);
    if (!promotor) {
      return res.status(404).json({ message: 'Promotor no encontrado' });
    }

    const existe = await Usuarios.findOne({ email: correo, _id: { $ne: id } });
    if (existe) {
      return res
        .status(400)
        .json({ message: 'El correo ya está en uso por otro usuario' });
    }

    const dataDistribuidor = await Usuarios.findById(distribuidor);
    if (!dataDistribuidor) {
      return res.status(400).json({ message: 'Distribuidor no encontrado' });
    }

    if (comision > dataDistribuidor.licencia.comision) {
      return res.status(400).json({
        message: `La comisión no puede ser mayor que el ${dataDistribuidor.licencia.comision}%`
      });
    }

    promotor.nombre = nombre;
    promotor.apellidoPaterno = apellidoP;
    promotor.email = correo;
    promotor.telefono = celular;
    promotor.licencia.referidoPor = dataDistribuidor._id;
    promotor.licencia.comision = comision;
    promotor.estatus = estatus || 'Activo';

    const result = await promotor.save();

    if (!result) {
      return res
        .status(400)
        .json({ message: 'Error al actualizar el promotor' });
    }

    return res.status(200).json({ message: 'Promotor actualizado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al actualizar el promotor',
      error
    });
  }
};

const enviarCredenciales = async (req, res) => {
  const { tipo, id } = req.body;

  try {
    const usuario = await Usuarios.findById(id);

    const password = desencriptar(usuario.password);
    if (tipo === 'correo') {
      const htmlRegistro = RegistroUsuarioHTML(
        `${usuario.nombre} ${usuario.apellidoPaterno}`,
        usuario.email,
        password,
        urlAPP
      );
      sendMail(htmlRegistro, 'NILDDA: Envió Accesos', usuario.email);
    }

    if (tipo === 'whatsApp') {
      sendWhatsappRecovery({
        to: usuario.telefono,
        correo: usuario.email,
        password,
        url: urlAPP
      });
    }

    return res.status(200).json({ message: 'Accesos enviados' });
  } catch (error) {
    console.error(error);
    return res.status(500).json({
      mensaje: 'Error al obtener el vendedor',
      error
    });
  }
};

module.exports = {
  crearPromotor,
  obtenerPromotores,
  obtenerPromotorPorId,
  enviarCredenciales,
  actualizarPromotor
};
