/* eslint-disable new-cap */
const paqueteModel = require("../models/Productos.js");
const ventaModel = require("../models/Ventas.js");
const { formatMoney } = require("../config/functionsGlobal.js");
const { buildFileUri, uploadFile, deleteFileS3 } = require("../config/s3.js");

const getPaquete = async (req, res) => {
  try {
    const { estatus, page = 1, search } = req.query;

    const options = {
      page,
      limit: 10,
      sort: {
        cantidad: 1,
        nombre: 1,
      },
    };

    const query = {
      tipo_producto: "Licencia",
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.nombre = { $regex: search, $options: "i" };
    }

    const paquetes = await paqueteModel.paginate(query, options);

    paquetes.docs = paquetes.docs.map((paquete) => {
      return {
        ...paquete._doc,
        precio: formatMoney(paquete.precio),
        imagen: buildFileUri(`productos/${paquete.imagen}`),
        adjunto: buildFileUri(`productos/${paquete.adjunto}`),
      };
    });

    res.status(200).json({ paquetes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getPaqueteSinPaginar = async (req, res) => {
  try {
    const results = await paqueteModel.find({
      estatus: "Activo",
      tipo_producto: "Licencia",
    });

    const recursos = results.map((e) => {
      return {
        value: e._id,
        label: e.nombre,
        key: e._id,
      };
    });

    res.status(200).json(recursos);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createPaquete = async (req, res) => {
  const {
    nombre,
    precio,
    cantidad,
    recomendado,
    beneficios = [],
    utilidad,
    param,
  } = req.body;

  const imagen = req.files?.imagen || null;
  const imagenPrincipal = req.files?.imagenPrincipal || null;
  const adjunto = req.files?.adjunto || null;

  if (!nombre) {
    return res
      .status(400)
      .json({ message: "El nombre de la licencia es requerido" });
  }

  if (!precio) {
    return res
      .status(400)
      .json({ message: "El precio de la licencia es requerido" });
  }

  if (!imagen) {
    return res
      .status(400)
      .json({ message: "La imagen de la licencia es requerida" });
  }

  if (!cantidad) {
    return res
      .status(400)
      .json({ message: "La cantidad de la licencia es requerida" });
  }

  if (!utilidad) {
    return res
      .status(400)
      .json({ message: "La utilidad de la licencia es requerida" });
  }

  if (beneficios.length === 0) {
    return res
      .status(400)
      .json({ message: "Los beneficios de la licencia son requeridos" });
  }

  if (!adjunto) {
    return res
      .status(400)
      .json({ message: "La adjunto de la licencia es requerida" });
  }

  try {
    const upload = await uploadFile(imagen, "productos");

    const uploadPrincipal = await uploadFile(imagenPrincipal, "productos");

    const uploadAdjunto = await uploadFile(adjunto, "productos");

    const newPaquete = new paqueteModel({
      nombre,
      tiempo: 0,
      imagen: upload.fileName,
      precio,
      cantidad: cantidad || 0,
      recomendado: recomendado || false,
      beneficios,
      tipo_producto: "Licencia",
      utilidad,
      param,
      adjunto: uploadAdjunto.fileName,
      imagenPrincipal: uploadPrincipal.fileName,
    });

    await newPaquete.save();

    res.status(201).json({ paquete: newPaquete, message: "Licencia creada" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deletePaquete = async (req, res) => {
  const { id } = req.params;

  try {
    const ventas = await ventaModel.find({ producto: id });

    if (ventas.length > 0) {
      const paquete = await paqueteModel.findById(id);
      paquete.estatus = "Inactivo";
      await paquete.save();

      return res.status(409).json({
        message:
          "No se puede eliminar el paquete porque tiene ventas asociadas, se cambio a estatus inactivo",
      });
    }

    await paqueteModel.findByIdAndRemove(id);

    res.status(200).json({ message: "Licencia eliminado" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};
const getPaqueteById = async (req, res) => {
  const { id } = req.params;

  try {
    const paquete = await paqueteModel.findById(id);

    if (!paquete) {
      return res.status(404).json({ message: "Licencia no encontrada" });
    }

    paquete.imagen = buildFileUri(`productos/${paquete.imagen}`);
    paquete.adjunto = buildFileUri(`productos/${paquete.adjunto}`);
    paquete.imagenPrincipal = buildFileUri(
      `productos/${paquete.imagenPrincipal}`
    );

    res.status(200).json(paquete);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updatePaquete = async (req, res) => {
  const { id } = req.params;
  const {
    nombre,
    precio,
    cantidad,
    recomendado,
    beneficios = [],
    estatus,
    param,
    utilidad,
  } = req.body;

  const imagen = req.files?.imagen || null;
  const imagenPrincipal = req.files?.imagenPrincipal || null;
  const adjunto = req.files?.adjunto || null;

  if (!nombre) {
    return res
      .status(400)
      .json({ message: "El nombre de la licencia es requerido" });
  }

  if (!precio) {
    return res
      .status(400)
      .json({ message: "El precio de la licencia es requerido" });
  }

  if (!cantidad) {
    return res
      .status(400)
      .json({ message: "La cantidad de la licencia es requerida" });
  }

  if (!utilidad) {
    return res
      .status(400)
      .json({ message: "La utilidad de la licencia es requerida" });
  }

  try {
    let upload = null;

    if (imagen) {
      upload = await uploadFile(imagen, "productos");
    }

    const paquete = await paqueteModel.findById(id);

    if (!paquete) {
      return res.status(404).json({ message: "Licencia no encontrado" });
    }

    paquete.nombre = nombre;
    paquete.precio = precio;
    paquete.cantidad = cantidad;
    paquete.recomendado = recomendado || false;
    paquete.beneficios = beneficios;
    paquete.estatus = estatus || "Activo";
    paquete.param = param || "";
    paquete.utilidad = utilidad;

    if (imagen) {
      if (paquete.imagen) {
        await deleteFileS3(`productos/${paquete.imagen}`, paquete.imagen);

        paquete.imagen = upload.fileName;
      }
    }

    if (imagenPrincipal) {
      const uploadPrincipal = await uploadFile(imagenPrincipal, "productos");
      if (paquete.imagenPrincipal) {
        await deleteFileS3(`productos/${paquete.imagenPrincipal}`);

        paquete.imagenPrincipal = uploadPrincipal.fileName;
      }
    }

    if (adjunto) {
      const uploadAdjunto = await uploadFile(adjunto, "productos");
      if (paquete.adjunto) {
        await deleteFileS3(`productos/${paquete.adjunto}`);
      }
      paquete.adjunto = uploadAdjunto.fileName;
    }

    await paquete.save();
    res.status(200).json({ paquete, message: "Paquete actualizado" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

module.exports = {
  getPaquete,
  createPaquete,
  deletePaquete,
  getPaqueteById,
  updatePaquete,
  getPaqueteSinPaginar,
};
