/* eslint-disable new-cap */
const paqueteModel = require('../models/Productos.js');
const ventaModel = require('../models/Ventas.js');
const { formatMoney } = require('../config/functionsGlobal.js');
const { buildFileUri, uploadFile, deleteFileS3 } = require('../config/s3.js');

const getPaquete = async (req, res) => {
  try {
    const { estatus, page = 1, search } = req.query;

    const options = {
      page,
      limit: 10,
      sort: {
        cantidad: 1,
        nombre: 1
      }
    };

    const query = {
      tipo_producto: 'GPT-4'
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.nombre = { $regex: search, $options: 'i' };
    }

    const paquetes = await paqueteModel.paginate(query, options);

    paquetes.docs = paquetes.docs.map(paquete => {
      return {
        ...paquete._doc,
        precio: formatMoney(paquete.precio),
        imagen: buildFileUri(`productos/${paquete.imagen}`)
      };
    });

    res.status(200).json({ paquetes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

// const getPaqueteSinPaginar = async (req, res) => {
//   const { estado, municipio } = req.query;

//   const query = {
//     estatus: 'Activo'
//   };

//   if (estado) {
//     query.estado = estado;
//   }

//   if (municipio) {
//     query.municipio = municipio;
//   }

//   try {
//     const juzgados = await paqueteModel.find(query);

//     const juzgadosSinPaginar = juzgados.map(juzgado => {
//       return {
//         value: juzgado._id,
//         nombre: juzgado.nombre,
//         label: juzgado.nombre
//       };
//     });

//     res.status(200).json(juzgadosSinPaginar);
//   } catch (error) {
//     res.status(404).json({ message: error.message });
//   }
// };

const createPaquete = async (req, res) => {
  const { nombre, precio, cantidad, recomendado, beneficios = [] } = req.body;

  const imagen = req.files?.imagen || null;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del juzgado es requerido' });
  }

  if (!precio) {
    return res.status(400).json({ message: 'El precio del juzgado es requerido' });
  }

  if (!imagen) {
    return res.status(400).json({ message: 'La imagen del juzgado es requerida' });
  }

  if (!cantidad) {
    return res.status(400).json({ message: 'La cantidad del juzgado es requerida' });
  }

  try {
    const upload = await uploadFile(imagen, 'productos');

    const newPaquete = new paqueteModel({
      nombre,
      tiempo: 0,
      imagen: upload.fileName,
      precio,
      cantidad,
      recomendado: recomendado || false,
      beneficios,
      tipo_producto: 'GPT-4'

    });

    await newPaquete.save();

    res.status(201).json({ paquete: newPaquete, message: 'Paquete creado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deletePaquete = async (req, res) => {
  const { id } = req.params;

  try {
    const ventas = await ventaModel.find({ producto: id });

    if (ventas.length > 0) {
      const paquete = await paqueteModel.findById(id);
      paquete.estatus = 'Inactivo';
      await paquete.save();

      return res.status(409).json({ message: 'No se puede eliminar el paquete porque tiene ventas asociadas, se cambio a estatus inactivo' });
    }

    await paqueteModel.findByIdAndRemove(id);

    res.status(200).json({ message: 'Paquete eliminado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};
const getPaqueteById = async (req, res) => {
  const { id } = req.params;

  try {
    const paquete = await paqueteModel.findById(id);

    if (!paquete) {
      return res.status(404).json({ message: 'Paquete no encontrado' });
    }

    paquete.imagen = buildFileUri(`productos/${paquete.imagen}`);

    res.status(200).json(paquete);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updatePaquete = async (req, res) => {
  const { id } = req.params;
  const { nombre, precio, cantidad, recomendado, beneficios = [], estatus } = req.body;

  const imagen = req.files?.imagen || null;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del juzgado es requerido' });
  }

  if (!precio) {
    return res.status(400).json({ message: 'El precio del juzgado es requerido' });
  }

  if (!cantidad) {
    return res.status(400).json({ message: 'La cantidad del juzgado es requerida' });
  }

  try {
    let upload = null;

    if (imagen) {
      upload = await uploadFile(imagen, 'productos');
    }

    const paquete = await paqueteModel.findById(id);

    if (!paquete) {
      return res.status(404).json({ message: 'Paquete no encontrado' });
    }

    paquete.nombre = nombre;
    paquete.precio = precio;
    paquete.cantidad = cantidad;
    paquete.recomendado = recomendado || false;
    paquete.beneficios = beneficios;
    paquete.estatus = estatus || 'Activo';

    if (imagen) {
      if (paquete.imagen) {
        await deleteFileS3('productos', paquete.imagen);

        paquete.imagen = upload.fileName;
      }
    }

    await paquete.save();
    res.status(200).json({ paquete, message: 'Paquete actualizado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

module.exports = {
  getPaquete,
  // getPaqueteSinPaginar,
  createPaquete,
  deletePaquete,
  getPaqueteById,
  updatePaquete
};
