/* eslint-disable array-callback-return */
const { default: mongoose } = require('mongoose');
const CorteModel = require('../models/CortesDistribuidores.js');
const VentaModel = require('../models/Ventas.js');
const moment = require('moment-timezone');
const { formatMoney, formatDateTime, formatDate } = require('../config/functionsGlobal.js');
const { buildFileUri, uploadFile } = require('../config/s3.js');

const getCortesDistribuidores = async (req, res) => {
  const { page = 1, fechaInicio, fechaFinal, estatus } = req.query;

  const options = {
    page,
    limit: 10,
    sort: { 'fecha.corte': -1 },
    populate: [
      {
        path: 'distribuidor'
      }
    ]
  };

  const query = {
  };

  if (fechaInicio && fechaFinal) {
    query['fecha.corte'] = {
      $gte: moment(fechaInicio).startOf('day').toDate(),
      $lte: moment(fechaFinal).endOf('day').toDate()
    };
  }

  if (estatus) {
    query.estatus = estatus;
  }

  try {
    const [findCortes] = await Promise.all([
      CorteModel.paginate(query, options)
    ]);

    findCortes.docs = findCortes.docs.map(corte => {
      return {
        ...corte._doc,
        observaciones: corte.observaciones || '',
        distribuidor: {
          nombre: `${corte?.distribuidor?.nombre || ''} ${corte?.distribuidor?.apellidoPaterno || ''} ${corte?.distribuidor?.apellidoMaterno || ''}`,
          foto: corte?.distribuidor?.foto?.includes('http')
            ? buildFileUri(corte.distribuidor.foto)
            : buildFileUri('default/icono_usuario_100x100_04.jpg'),
          comision: (corte?.distribuidor?.licencia?.comision || '0') + '%'
        },
        fecha: {
          corte: formatDateTime(corte.fecha.corte),
          inicio: formatDate(corte.fecha.inicio),
          fin: formatDate(corte.fecha.fin)
        },
        importePago: formatMoney(corte.importePago),
        importeSinComision: formatMoney(corte.importeSinComision),

        comprobante: {
          xml: corte.comprobante.xml ? buildFileUri(`cortes/${corte.comprobante.xml}`) : null,
          factura: corte.comprobante.factura ? buildFileUri(`cortes/${corte.comprobante.factura}`) : null,
          pago: corte.comprobante.pago ? buildFileUri(`cortes/${corte.comprobante.pago}`) : null,
          fechaPago: corte.comprobante.fechaPago ? formatDateTime(corte.comprobante.fechaPago) : null
        },
        ventas: corte.ventas.map(venta => {
          return {
            ...venta._doc,
            fechaVenta: formatDateTime(venta.fechaVenta),
            fecha_pago: venta.fecha_pago ? formatDateTime(venta.fecha_pago) : null,
            importe: formatMoney(venta.importe)

          };
        })
      };
    });

    res.status(200).json({ cortes: findCortes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

// const previewDelCorte = async (req, res) => {
//   const { distribuidor } = req.params;

//   if (!distribuidor) {
//     return res.status(400).json({ message: 'El distribuidor es requerido' });
//   }

//   try {
//     const ventas = await getVentasXFechas({ distribuidor });

//     if (!ventas) {
//       return res.status(404).json({ message: 'No se encontraron ventas' });
//     }

//     let { ventasTotales, totalImporte, findVentas, desde, hasta } = ventas;
//     if (!findVentas) {
//       return res.status(404).json({ message: 'No se encontraron ventas' });
//     }
//     findVentas = findVentas.map(venta => {
//       return {
//         ...venta,
//         fechaVenta: formatDateTime(venta.fechaVenta),
//         distribuidor: {
//           comision: (venta?.distribuidor?.comision || '0') + '%',
//           importe: formatMoney(venta.distribuidor.importe)
//         }

//       };
//     });

//     res.status(200).json({ ventasTotales, totalImporte: formatMoney(totalImporte), ventas: findVentas, desde: formatDate(desde), hasta: formatDate(hasta) });
//   } catch (error) {
//     res.status(404).json({ message: error.message });
//   }
// };

// const getVentasXFechas = async ({ distribuidor }) => {
//   const query = {
//     'distribuidor.corte': null,
//     'distribuidor.distribuidor': mongoose.Types.ObjectId(distribuidor),
//     estatus: 'Pagado'
//   };

//   try {
//     let total = 0;
//     let findVentas = await VentaModel.find(query)
//       .populate({ path: 'producto', select: 'nombre imagen' })
//       .sort({ fechaVenta: -1 }).lean();

//     if (!findVentas?.length || findVentas?.length === 0) {
//       return null;
//     }

//     findVentas = findVentas.map(venta => {
//       total += venta.importe;
//       return {
//         ...venta,
//         fechaVenta: venta.fechaVenta,
//         // fechaVenta: formatDateTime(venta.fechaVenta),
//         producto: {
//           ...venta.producto,
//           imagen: venta?.producto?.imagen?.includes('http')
//             ? venta.producto.imagen
//             : buildFileUri(`productos/${venta.producto.imagen}`)
//           // precio: formatMoney(venta?.producto?.precio || 0)
//         },
//         importe: formatMoney(venta.importe)
//       };
//     });

//     const ventasTotales = findVentas?.length || 0;
//     const totalImporte = findVentas?.reduce((acc, venta) => acc + venta?.distribuidor?.importe, 0) || 0;
//     const desde = findVentas[findVentas.length - 1].fechaVenta || null;
//     const hasta = findVentas[0].fechaVenta || null;

//     return {
//       ventasTotales,
//       totalImporte,
//       findVentas,
//       desde,
//       hasta,
//       importeSinComision: total
//     };
//   } catch (error) {
//     console.error(error); // Log the error for debugging purposes
//     return null;
//   }
// };

const getCorteById = async (req, res) => {
  const { id } = req.params;

  if (!id) {
    return res.status(400).json({ message: 'El ID es requerido' });
  }

  try {
    const findCorte = await CorteModel.findById(id).populate([{
      path: 'ventas',
      select: 'producto importe distribuidor.comision  distribuidor.importe fechaVenta tipoReferencia estatus',
      populate: [
        {
          path: 'producto',
          select: 'nombre imagen'
        },
        {
          path: 'usuario',
          select: 'nombre apellidoPaterno apellidoMaterno foto'
        },
        {
          path: 'despacho',
          select: 'nombre logo'
        }
      ]
    }
    ]).lean();

    if (!findCorte) {
      return res.status(404).json({ message: 'Corte no encontrado' });
    }

    findCorte.ventas = findCorte.ventas.map(venta => {
      let tipoUsuario = null;

      if (venta?.despacho?._id) {
        tipoUsuario = 'Despacho';
      } else {
        tipoUsuario = 'Distribuidor';
      }

      return {
        ...venta,
        producto: {
          ...venta.producto,
          imagen: venta?.producto?.imagen?.includes('http')
            ? venta.producto.imagen
            : buildFileUri(`productos/${venta.producto.imagen}`)
        },
        importe: formatMoney(venta.importe),
        distribuidor: {
          comision: (venta?.distribuidor?.comision || '0') + '%',
          importe: formatMoney(venta.distribuidor.importe)
        },
        usuario: {
          nombre: tipoUsuario === 'Despacho' ? venta.despacho.nombre : `${venta?.usuario?.nombre || ''} ${venta?.usuario?.apellidoPaterno || ''} ${venta?.usuario?.apellidoMaterno || ''}`,
          foto: tipoUsuario === 'Despacho'
            ? buildFileUri(`despachos/${venta.despacho._id}/logo/${venta.despacho.logo}`)
            : venta?.usuario?.foto?.includes('http')
              ? buildFileUri(venta.usuario.foto)
              : buildFileUri('default/icono_usuario_100x100_04.jpg'),
          tipo: tipoUsuario

        },
        fechaVenta: formatDateTime(venta.fechaVenta)
      };
    });

    findCorte.fecha = {
      corte: formatDateTime(findCorte.fecha.corte),
      inicio: formatDate(findCorte.fecha.inicio),
      fin: formatDate(findCorte.fecha.fin)
    };

    findCorte.importePago = formatMoney(findCorte.importePago);

    findCorte.importeSinComision = formatMoney(findCorte.importeSinComision);

    findCorte.comprobante = {
      xml: findCorte.comprobante.xml ? buildFileUri(`cortes/${findCorte.comprobante.xml}`) : null,
      factura: findCorte.comprobante.factura ? buildFileUri(`cortes/${findCorte.comprobante.factura}`) : null,
      pago: findCorte.comprobante.pago ? buildFileUri(`cortes/${findCorte.comprobante.pago}`) : null,
      fechaPago: findCorte.comprobante.fechaPago ? formatDateTime(findCorte.comprobante.fechaPago) : null
    };

    res.status(200).json({ corte: findCorte });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const rechazarCorte = async (req, res) => {
  const { id } = req.params;
  const { observaciones } = req.body;

  if (!id) {
    return res.status(400).json({ message: 'El ID es requerido' });
  }

  try {
    const findCorte = await CorteModel.findById(id);

    if (!findCorte) {
      return res.status(404).json({ message: 'Corte no encontrado' });
    }

    findCorte.estatus = 'Rechazado';
    findCorte.observaciones = observaciones || '';

    await findCorte.save();

    res.status(200).json({ message: 'Corte rechazado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const subirComprobantePago = async (req, res) => {
  const { id } = req.params;
  const comprobante = req.files?.comprobante || null;

  if (!id) {
    return res.status(400).json({ message: 'El ID es requerido' });
  }

  if (!comprobante) {
    return res.status(400).json({ message: 'El comprobante es requerido' });
  }

  try {
    const findCorte = await CorteModel.findById(id);

    if (!findCorte) {
      return res.status(404).json({ message: 'Corte no encontrado' });
    }

    const comprobantePago = await uploadFile(comprobante, 'cortes');

    findCorte.comprobante.pago = comprobantePago.fileName;
    findCorte.comprobante.fechaPago = new Date();
    findCorte.estatus = 'Pagado';
    findCorte.observaciones = null;
    await findCorte.save();

    res.status(200).json({ message: 'Comprobante subido' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getCortesDistribuidores,
  // previewDelCorte,
  getCorteById,
  rechazarCorte,
  subirComprobantePago
  // getVentasXFechas
};
