const MateriasModel = require("../models/MateriasDespacho.js");
const ExpedientesModel = require("../models/Expedientes.js");
const EtapasProcesalesModel = require("../models/EtapasProcesalesDespachos.js");
const FoliosModel = require("../models/ExpedientesFolios.js");

const getMaterias = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search } = req.query;

  const options = {
    page,
    limit: 15,
    sort: { nombre: 1 },
  };

  try {
    const query = {
      despacho,
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.nombre = { $regex: search, $options: "i" };
    }

    const materias = await MateriasModel.paginate(query, options);
    res.status(200).json({ materias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getMateriasSinPaginar = async (req, res) => {
  const { despacho } = req.params;

  try {
    const query = {
      despacho,
    };

    query.estatus = "Activo";

    const materiasFind = await MateriasModel.find(query).sort({ nombre: 1 });

    const materias = materiasFind.map((materia) => {
      return {
        value: materia._id,
        label: materia.nombre,
        key: materia._id,
      };
    });

    res.status(200).json({ materias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createMateria = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, estatus, clave = "", creadoPor } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }

  // validar el nombre de la materia no repetido
  const findMateria = await MateriasModel.findOne({ nombre, despacho });

  if (findMateria) {
    return res.status(409).json({ message: "La materia ya existe" });
  }

  // const findFolio = await FoliosModel.findOne({ clave, despacho }).populate(
  //   "materia"
  // );

  // if (findFolio) {
  //   return res.status(409).json({
  //     message: "La clave ya existe en la materia " + findFolio?.materia?.nombre,
  //   });
  // }

  const materia = {
    despacho,
    nombre,
    estatus: estatus || "Activo",
  };

  try {
    const newMateria = await MateriasModel.create(materia);

    const id = newMateria._id;

    const newFolio = {
      despacho,
      materia: id,
      clave,
      folio: 1,
      creadoPor,
    };

    await FoliosModel.create(newFolio);

    res
      .status(201)
      .json({ materia: newMateria, message: "Materia creada correctamente" });
  } catch (error) {
    console.log(error);
    res.status(409).json({ message: error.message });
  }
};

const getMateriaById = async (req, res) => {
  const { despacho, id } = req.params;

  try {
    const materia = await MateriasModel.findOne({ _id: id, despacho });

    if (!materia) {
      return res
        .status(404)
        .json({ message: "No se encontró el tipo de trabajo o servicio" });
    }

    res.status(200).json(materia);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateMateria = async (req, res) => {
  const { id, despacho } = req.params;
  const { nombre, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }

  const materia = {
    nombre,
    estatus: estatus || "Activo",
    despacho,
  };

  try {
    const updatedMateria = await MateriasModel.findByIdAndUpdate(id, materia, {
      new: true,
    });

    // actualizar expedientes materia->nombre
    await ExpedientesModel.updateMany(
      { materia: id, despacho },
      { $set: { materia: nombre } }
    );
    res.status(200).json(updatedMateria);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteMateria = async (req, res) => {
  const { id, despacho } = req.params;

  try {
    const materia = await MateriasModel.findOne({ _id: id, despacho });

    if (!materia) {
      return res
        .status(404)
        .json({ message: "No se encontró el tipo de trabajo o servicio" });
    }

    const expedientes = await ExpedientesModel.countDocuments({
      materia: id,
      despacho,
    });

    const etapasProcesales = await EtapasProcesalesModel.countDocuments({
      materia: id,
      despacho,
    });

    if (etapasProcesales > 0) {
      return res.status(400).json({
        message:
          "No se puede eliminar la materia porque tiene etapas asociadas",
      });
    }

    if (expedientes > 0) {
      return res.status(400).json({
        message:
          "No se puede eliminar la materia porque tiene expedientes asociados",
      });
    }

    await MateriasModel.findOneAndDelete({ _id: id, despacho });
    // eliminar folio
    await FoliosModel.findOneAndDelete({ materia: id, despacho });

    res.status(200).json({ message: "Materia eliminada" });
  } catch (error) {
    res.status(404).json({ message: "Ocurrió un error inténtelo de nuevo." });
  }
};

module.exports = {
  getMaterias,
  getMateriasSinPaginar,
  createMateria,
  getMateriaById,
  updateMateria,
  deleteMateria,
};
