const EtapasProcesales = require('../models/EtapasProcesales.js');
const MateriasModel = require('../models/Materias.js');
const RecursosIncidencias = require('../models/RecursosIncidencias.js');

const getMaterias = async (req, res) => {
  const { estatus, page = 1, search } = req.query;

  const options = {
    page,
    limit: 50,
    sort: { nombre: 1 }
  };

  try {
    const query = {};

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.nombre = { $regex: search, $options: 'i' };
    }

    const materias = await MateriasModel.paginate(query, options);
    res.status(200).json({ materias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getMateriasSinPaginar = async (req, res) => {
  const { estatus } = req.query;

  try {
    const query = {};

    if (estatus) {
      query.estatus = estatus;
    }

    const materiasFind = await MateriasModel.find(query).sort({ nombre: 1 });

    const materias = materiasFind.map(materia => {
      return {
        value: materia._id,
        text: materia.nombre,
        label: materia.nombre,
        key: materia._id
      };
    });

    res.status(200).json({ materias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createMateria = async (req, res) => {
  const { nombre, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  const materia = {
    nombre,
    estatus: estatus || 'Activo'
  };

  try {
    const newMateria = MateriasModel.create(materia);
    res.status(201).json(newMateria);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getMateriaById = async (req, res) => {
  const { id } = req.params;

  try {
    const materia = await MateriasModel.findById(id);
    if (!materia) {
      return res.status(404).json({ message: 'No se encontró la materia' });
    }

    res.status(200).json(materia);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateMateria = async (req, res) => {
  const { id } = req.params;
  const { nombre, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  const materia = {
    nombre,
    estatus: estatus || 'Activo'
  };

  try {
    const updatedMateria = await MateriasModel.findByIdAndUpdate(id, materia, { new: true });
    res.status(200).json(updatedMateria);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteMateria = async (req, res) => {
  const { id } = req.params;

  console.log(id);
  try {
    const isRelatedInCollection1 = await RecursosIncidencias.exists({ materia: id });
    const isRelatedInCollection2 = await EtapasProcesales.exists({ materia: id });

    if (isRelatedInCollection1 || isRelatedInCollection2) {
      return res.status(400).json({ message: 'No se puede eliminar, el tipo de trabajo o servicio está relacionada con otros documentos.' });
    }

    await MateriasModel.findByIdAndRemove(id);
    res.status(200).json({ message: 'Materia eliminada' });
  } catch (error) {
    console.log(error);
    res.status(500).json({ message: error.message });
  }
};

module.exports = {
  getMaterias,
  getMateriasSinPaginar,
  createMateria,
  getMateriaById,
  updateMateria,
  deleteMateria
};
