const MachotesModel = require('../models/machotes.js');
const iconv = require('iconv-lite');
const { buildFileUri, uploadFile, deleteFileS3 } = require('../config/s3.js');

const getMachotes = async (req, res) => {
  const { despacho, usuario } = req.params;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 10,
    sort: { fechaCreacion: -1 },
    populate: [
      {
        path: 'creador',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      }
    ]

  };

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  if (!usuario) {
    return res.status(400).json({ message: 'El usuario es requerido' });
  }

  const query = {
    despacho,
    $or: [
      { creador: usuario },
      { usuariosCompartidos: usuario }
    ]
  };

  try {
    const machotes = await MachotesModel.paginate(query, options);

    // Eliminar documentos duplicados
    const uniqueDocs = Array.from(new Set(machotes.docs.map(doc => doc._id.toString())))
      .map(id => machotes.docs.find(doc => doc._id.toString() === id));

    // Actualizar la respuesta con los documentos únicos
    machotes.docs = uniqueDocs;

    machotes.docs = machotes.docs.map(machote => {
      return {
        _id: machote._id,
        nombre: machote.nombre,
        descripcion: machote.descripcion,
        creador: {
          _id: machote.creador._id,
          nombre: machote.creador.nombre,
          apellidoPaterno: machote.creador.apellidoPaterno,
          apellidoMaterno: machote.creador.apellidoMaterno,
          foto: machote.creador.foto
            ? buildFileUri(`usuarios/${machote.creador.foto}`)
            : buildFileUri('default/icono_usuario_100x100_04.jpg')

        },
        archivos: machote.archivos.map(archivo => {
          return {
            _id: archivo._id,
            nombre: archivo.nombre,
            ruta: buildFileUri(`despachos/${despacho}/machotes/${archivo.ruta}`)
          };
        })
      };
    });

    res.status(200).json({ machotes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createMachotes = async (req, res) => {
  const { despacho } = req.params;
  const { usuario, usuariosCompartidos, nombre, descripcion } = req.body;
  try {
    let { archivos } = req?.files || [];

    if (!despacho) {
      return res.status(400).json({ message: 'El despacho es requerido' });
    }

    if (!usuario) {
      return res.status(400).json({ message: 'El usuario es requerido' });
    }

    if (!usuariosCompartidos) {
      return res.status(400).json({ message: 'Los usuarios compartidos son requeridos' });
    }

    if (!nombre) {
      return res.status(400).json({ message: 'El nombre es requerido' });
    }

    if (archivos.length === 0) {
      return res.status(400).json({ message: 'Los archivos son requeridos' });
    }

    archivos = Array.isArray(archivos) ? archivos : [archivos];

    const machotesInsert = new MachotesModel({
      despacho,
      creador: usuario,
      nombre,
      descripcion,
      usuariosCompartidos,
      archivos: await Promise.all(archivos.map(async (archivo) => ({
        nombre: iconv.decode(Buffer.from(archivo.name, 'binary'), 'utf-8'),
        ruta: await uploadFile(archivo, `despachos/${despacho}/machotes`).then(data => data.fileName)
      })))
    });

    await machotesInsert.save();

    res.status(200).json({ message: 'Documentos creados' });
  } catch (error) {
    console.log(error);
    return res.status(400).json({ message: error.message });
  }
};

const deleteDocumento = async (req, res) => {
  const { id } = req.params;
  try {
    const documento = await MachotesModel.findByIdAndDelete(id);

    if (!documento) {
      return res.status(404).json({ message: 'Documento no encontrado' });
    }

    documento.archivos.forEach(archivo => {
      // fs.unlinkSync(path.join('src/uploads/machotes', archivo.ruta));
      deleteFileS3(`despachos/${documento.despacho}/machotes/${archivo.ruta}`);
    });

    res.status(200).json({ message: 'Documento eliminado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const geMachoteById = async (req, res) => {
  const { _id, despacho } = req.params;
  try {
    const machote = await MachotesModel.findOne({ _id, despacho }).populate({
      path: 'usuariosCompartidos',
      select: 'nombre apellidoPaterno apellidoMaterno foto'
    });

    if (!machote) {
      return res.status(404).json({ message: 'Formato no encontrado' });
    }

    machote.archivos = machote.archivos.map(archivo => {
      return {
        _id: archivo._id,
        nombre: archivo.nombre,
        ruta: buildFileUri(`despachos/${despacho}/machotes/${archivo.ruta}`)
      };
    });

    machote.usuariosCompartidos.forEach(usuario => {
      usuario.foto = usuario.foto
        ? buildFileUri(`usuarios/${usuario.foto}`)
        : buildFileUri('default/icono_usuario_100x100_04.jpg');
    });

    res.status(200).json({ machote });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateNameDocument = async (req, res) => {
  const { _id } = req.params;
  const { usuariosCompartidos, nombre, descripcion } = req.body;

  if (!usuariosCompartidos) {
    return res.status(400).json({ message: 'Los usuarios compartidos son requeridos' });
  }

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  try {
    const machote = await MachotesModel.findById(_id);

    if (!machote) {
      return res.status(404).json({ message: 'Formato no encontrado' });
    }

    machote.usuariosCompartidos = usuariosCompartidos;
    machote.nombre = nombre;
    machote.descripcion = descripcion;

    await machote.save();
    res.status(200).json({ message: 'Machote actualizado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteFileMachote = async (req, res) => {
  const { _id, idArchivo } = req.params;
  try {
    // Lógica para obtener la ruta del archivo basado en idArchivo o _id
    // Supongamos que tienes la ruta almacenada en la base de datos
    const machote = await MachotesModel.findOne({
      _id
    });
    if (!machote) {
      return res.status(404).json({ message: 'Formato no encontrado' });
    }

    // Supongamos que archivos es un array y queremos eliminar el archivo específico
    const archivoAEliminar = machote.archivos.find(archivo => archivo._id.toString() === idArchivo);
    console.log(archivoAEliminar);
    if (!archivoAEliminar) {
      return res.status(404).json({ message: 'Archivo no encontrado' });
    }

    await deleteFileS3(`despachos/${machote.despacho}/machotes/${archivoAEliminar.ruta}`);

    // Lógica para eliminar el archivo de la base de datos
    machote.archivos.pull({ _id: idArchivo });
    await machote.save();

    res.status(200).json({ message: 'Archivo eliminado' });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const addFileMachote = async (req, res) => {
  const { _id, despacho } = req.params;
  let { archivos } = req?.files ?? [];

  if (archivos.length === 0) {
    return res.status(400).json({ message: 'Los archivos son requeridos' });
  }

  try {
    const machote = await MachotesModel.findById(_id);

    if (!machote) {
      return res.status(404).json({ message: 'Formato no encontrado' });
    }

    if (archivos?.length > 1) {
      return res.status(400).json({ message: 'Solo puedes subir un archivo' });
    }

    console.log(archivos);
    archivos = Array.isArray(archivos) ? archivos : [archivos];

    machote.archivos.push(...await Promise.all(archivos.map(async (archivo) => ({
      nombre: archivo.name,
      ruta: await uploadFile(archivo, `despachos/${despacho}/machotes`).then(data => data.fileName)
    }))));

    await machote.save();
    res.status(200).json({ message: 'Archivos agregados' });
  } catch (error) {
    console.log(error);
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getMachotes,
  createMachotes,
  deleteDocumento,
  updateNameDocument,
  geMachoteById,
  deleteFileMachote,
  addFileMachote
};
