const { uploadFile, deleteFileS3, buildFileUri } = require('../config/s3.js');
const LegislacionesModel = require('../models/LegislacionesReglamentos.js');

const getLegislaciones = async (req, res) => {
  const { page = 1, search, estado, estatus } = req.query;
  const options = {
    page,
    limit: 20,
    sort: { estado: 1, nombre: 1 },
    populate: 'estado'
  };

  const query = {};

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
  }

  if (estado) {
    query.estado = estado;
  }

  if (estatus) {
    query.estatus = estatus;
  }

  try {
    const findLegislaciones = await LegislacionesModel.paginate(query, options);

    findLegislaciones.docs.map((e) => {
      if (e.adjunto) {
        e.adjunto = buildFileUri(`legislaciones-reglamentos/${e.adjunto}`);
      }
    });
    res.status(200).json({ legislaciones: findLegislaciones });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createLegislacion = async (req, res) => {
  const { nombre, estado, enlace } = req.body;
  const adjunto = req.files?.adjunto || null;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }
  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }
  if (!enlace) {
    return res.status(400).json({ message: 'El enlace es requerido' });
  }

  try {
    let fileName = null;

    if (adjunto) {
      const uploadAdjunto = await uploadFile(
        adjunto,
        'legislaciones-reglamentos'
      );
      fileName = uploadAdjunto.fileName;
    }

    const newLegislacion = new LegislacionesModel({
      nombre,
      estado,
      enlace,
      adjunto: fileName
    });

    await newLegislacion.save();

    res
      .status(201)
      .json({ message: 'Legislación creada', legislacion: newLegislacion });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateLegislacion = async (req, res) => {
  const { id } = req.params;
  const { nombre, estado, enlace, estatus } = req.body;
  const adjunto = req.files?.adjunto || null;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }

  if (!enlace) {
    return res.status(400).json({ message: 'El enlace es requerido' });
  }

  try {
    const legislacion = await LegislacionesModel.findById(id);
    if (!legislacion) {
      return res.status(404).json({ message: 'Legislación no encontrada' });
    }

    if (adjunto) {
      const uploadAdjunto = await uploadFile(
        adjunto,
        'legislaciones-reglamentos'
      );

      if (legislacion.adjunto) {
        await deleteFileS3(`legislaciones-reglamentos/${legislacion.adjunto}`);
      }
      legislacion.adjunto = uploadAdjunto.fileName;
    }

    legislacion.nombre = nombre;
    legislacion.estado = estado;
    legislacion.enlace = enlace;
    legislacion.estatus = estatus || 'Activo';

    const updatedLegislacion = await legislacion.save();

    res.status(200).json({
      message: 'Legislación actualizada',
      legislacion: updatedLegislacion
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getLegislacionById = async (req, res) => {
  const { id } = req.params;
  try {
    const legislacion = await LegislacionesModel.findById(id);

    legislacion.adjunto = buildFileUri(
      `legislaciones-reglamentos/${legislacion.adjunto}`
    );
    return res.status(200).json(legislacion);
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

const deleteLegislacion = async (req, res) => {
  const { id } = req.params;

  try {
    await LegislacionesModel.findByIdAndDelete(id);
    res.status(200).json({ message: 'Legislación eliminada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const generateExcel = async (req, res) => {
  const XLSX = require('xlsx');

  try {
    const findLegislaciones = await LegislacionesModel.find().populate('estado').sort({ estado: 1, nombre: 1 }).lean();

    const formattedLegislaciones = findLegislaciones.map(legislacion => ({
      _id: legislacion._id.toString(),
      Nombre: legislacion.nombre,
      Estado: legislacion.estado.nombre,
      Enlace: legislacion.enlace,
      Estatus: legislacion.estatus
    }));

    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(formattedLegislaciones);

    XLSX.utils.book_append_sheet(wb, ws, 'Legislaciones');

    const excelFileName = `legislaciones-${Date.now()}.xlsx`;
    const excelFilePath = `./${excelFileName}`;

    XLSX.writeFile(wb, excelFilePath);

    res.status(200).json({
      excelFileName,
      excelFilePath
    });

    return res.status(200).json(findLegislaciones);
  } catch (error) {
    console.log(error);
    return res.status(404).json({ message: error.message });
  }
};

// const generateExcel = async (req, res) => {
//   const XLSX = require('xlsx');

//   try {
//     // Obtén los juzgados y popula el campo 'estado'
//     const findJuzgados = await Juzgados.find().populate('estado').populate('municipio').sort({ estado: 1, municipio: 1, nombre: 1 }).lean();

//     // Define las columnas que deseas incluir en el archivo Excel
//     const formattedJuzgados = findJuzgados.map(juzgado => ({
//       _id: juzgado._id.toString(),
//       Nombre: juzgado.nombre,
//       Tipo: juzgado.tipo,
//       Direccion: juzgado.direccion,
//       Estado: juzgado.estado.nombre,
//       Municipio: juzgado.municipio.nombre,
//       Telefonos: juzgado.telefonos.join(', '),
//       URL: juzgado.url,
//       Estatus: juzgado.estatus
//     }));

//     // Crea un nuevo libro y hoja de trabajo
//     const wb = XLSX.utils.book_new();
//     const ws = XLSX.utils.json_to_sheet(formattedJuzgados);

//     // Agrega la hoja de trabajo al libro
//     XLSX.utils.book_append_sheet(wb, ws, 'Juzgados');

//     // Define el nombre y la ruta del archivo Excel
//     const excelFileName = `juzgados-${Date.now()}.xlsx`;
//     const excelFilePath = `./${excelFileName}`;

//     // Escribe el archivo Excel
//     XLSX.writeFile(wb, excelFilePath);

//     // Responde con el nombre y la ruta del archivo Excel
//     res.status(200).json({
//       excelFileName,
//       excelFilePath
//     });
//   } catch (error) {
//     // En caso de error, responde con el mensaje de error
//     return res.status(404).json({ message: error.message });
//   }
// };
module.exports = {
  getLegislaciones,
  createLegislacion,
  updateLegislacion,
  getLegislacionById,
  deleteLegislacion,
  generateExcel
};
