const Juzgados = require('../models/Juzgados.js');
const Estados = require('../models/Estados.js');
const Municipios = require('../models/Municipios.js');

const getJuzgados = async (req, res) => {
  try {
    const { estatus, estado, municipio, page = 1, search } = req.query;

    const options = {
      page,
      limit: 10,
      sort: {
        estado: 1, // Ordenar por estado ascendente
        municipio: 1, // Ordenar por municipio ascendente
        nombre: 1 // Ordenar por nombre ascendente
      },
      populate: [
        {
          path: 'estado',
          select: 'nombre'
        },
        {
          path: 'municipio',
          select: 'nombre'
        }
      ]
    };

    const query = {};

    if (estatus) {
      query.estatus = estatus;
    }

    if (estado) {
      query.estado = estado;
    }

    if (municipio) {
      query.municipio = municipio;
    }

    if (search) {
      query.$or = [
        { nombre: { $regex: search, $options: 'i' } },
        { tipo: { $regex: search, $options: 'i' } },
        { direccion: { $regex: search, $options: 'i' } },
        { telefonos: { $regex: search, $options: 'i' } },
        { url: { $regex: search, $options: 'i' } }
      ];
    }
    const juzgados = await Juzgados.paginate(query, options);

    res.status(200).json({ juzgados });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getJuzgadosSinPaginar = async (req, res) => {
  const { estado, municipio } = req.query;

  const query = {
    estatus: 'Activo'
  };

  if (estado) {
    query.estado = estado;
  }

  if (municipio) {
    query.municipio = municipio;
  }

  try {
    const juzgados = await Juzgados.find(query);

    const juzgadosSinPaginar = juzgados.map(juzgado => {
      return {
        value: juzgado._id,
        nombre: juzgado.nombre,
        label: juzgado.nombre
      };
    });

    res.status(200).json(juzgadosSinPaginar);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createJuzgado = async (req, res) => {
  const { nombre, tipo, direccion, telefonos, estado, municipio, paginaWeb } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del juzgado es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado del juzgado es requerido' });
  }

  if (!municipio) {
    return res.status(400).json({ message: 'El municipio del juzgado es requerido' });
  }

  try {
    const newTelefonos = telefonos.length > 0
      ? telefonos.split(',').map(tel => tel.trim())
      : [];

    const newJuzgado = new Juzgados({
      nombre,
      tipo: tipo || '',
      direccion: direccion || '',
      telefonos: newTelefonos,
      estado,
      municipio,
      url: paginaWeb || '',
      estatus: 'Activo'
    });

    await newJuzgado.save();

    res.status(201).json({ juagado: newJuzgado, message: 'Juzgado creado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteJuzgado = async (req, res) => {
  const { id } = req.params;

  try {
    await Juzgados.findByIdAndRemove(id);

    res.status(200).json({ message: 'Juzgado eliminado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};
const getJuzgadosById = async (req, res) => {
  const { id } = req.params;

  try {
    const juzgado = await Juzgados.findById(id);

    res.status(200).json(juzgado);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateJuzgado = async (req, res) => {
  const { id } = req.params;
  const { nombre, tipo, direccion, telefonos, estado, municipio, paginaWeb, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre del juzgado es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado del juzgado es requerido' });
  }

  if (!municipio) {
    return res.status(400).json({ message: 'El municipio del juzgado es requerido' });
  }

  try {
    const juzgado = await Juzgados.findById(id);
    const newTelefonos = telefonos.length > 0
      ? telefonos.split(',').map(tel => tel.trim())
      : [];
    juzgado.nombre = nombre;
    juzgado.tipo = tipo || '';
    juzgado.direccion = direccion || '';
    juzgado.telefonos = newTelefonos;
    juzgado.estado = estado;
    juzgado.municipio = municipio;
    juzgado.url = paginaWeb || '';
    juzgado.estatus = estatus || 'Activo';

    await juzgado.save();

    res.status(200).json({ juzgado, message: 'Juzgado actualizado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const generateExcel = async (req, res) => {
  const XLSX = require('xlsx');

  try {
    // Obtén los juzgados y popula el campo 'estado'
    const findJuzgados = await Juzgados.find().populate('estado').populate('municipio').sort({ estado: 1, municipio: 1, nombre: 1 }).lean();

    // Define las columnas que deseas incluir en el archivo Excel
    const formattedJuzgados = findJuzgados.map(juzgado => ({
      _id: juzgado._id.toString(),
      Nombre: juzgado.nombre,
      Tipo: juzgado.tipo,
      Direccion: juzgado.direccion,
      Estado: juzgado.estado.nombre,
      Municipio: juzgado.municipio.nombre,
      Telefonos: juzgado.telefonos.join(', '),
      URL: juzgado.url,
      Estatus: juzgado.estatus
    }));

    // Crea un nuevo libro y hoja de trabajo
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(formattedJuzgados);

    // Agrega la hoja de trabajo al libro
    XLSX.utils.book_append_sheet(wb, ws, 'Juzgados');

    // Define el nombre y la ruta del archivo Excel
    const excelFileName = `juzgados-${Date.now()}.xlsx`;
    const excelFilePath = `./${excelFileName}`;

    // Escribe el archivo Excel
    XLSX.writeFile(wb, excelFilePath);

    // Responde con el nombre y la ruta del archivo Excel
    res.status(200).json({
      excelFileName,
      excelFilePath
    });
  } catch (error) {
    // En caso de error, responde con el mensaje de error
    return res.status(404).json({ message: error.message });
  }
};

const generateExcelEstadosMunicipios = async (req, res) => {
  try {
    const XLSX = require('xlsx');
    // 1. Obtener todos los estados ordenados por nombre
    const estados = await Estados.find().sort({ nombre: 1 }).lean();

    // Crear un nuevo workbook de Excel
    const workbook = XLSX.utils.book_new();

    // 2. Recorrer cada estado
    for (const estado of estados) {
      // 3. Obtener los municipios de cada estado
      const municipios = await Municipios.find({ estado: estado._id }).sort({ nombre: 1 }).lean();

      // 4. Preparar los datos para la hoja de Excel
      const sheetData = [
        [estado.nombre], // Encabezados
        [estado._id.toString()], // Primera fila con ID y nombre del estado
        [],
        ['Clave', 'Municipio'], // Encabezados
        ...municipios.map(municipio => [
          municipio._id.toString(), // Clave del municipio
          municipio.nombre // Nombre del municipio
        ]) // Filas de municipios
      ];

      // 5. Crear una hoja para el estado
      const worksheet = XLSX.utils.aoa_to_sheet(sheetData);

      // 6. Añadir la hoja al workbook
      XLSX.utils.book_append_sheet(workbook, worksheet, estado.nombre);
    }

    // 7. Generar el archivo Excel en formato de buffer
    const excelBuffer = XLSX.write(workbook, { type: 'buffer', bookType: 'xlsx' });

    // 8. Enviar el archivo Excel como respuesta
    res.setHeader('Content-Disposition', 'attachment; filename="EstadosMunicipios.xlsx"');
    res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    return res.send(excelBuffer);
  } catch (error) {
    // En caso de error, responde con el mensaje de error
    return res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getJuzgados,
  getJuzgadosSinPaginar,
  createJuzgado,
  deleteJuzgado,
  getJuzgadosById,
  updateJuzgado,
  generateExcel,
  generateExcelEstadosMunicipios
};
