const axios = require('axios');
const JurisprudenciaOrdenar = require('../models/JurisprudenciaOrdenar');

const getOrdenamiento = async (req, res) => {
  try {
    const results = await JurisprudenciaOrdenar.find();

    const sorts = results.map((e) => {
      return {
        value: e.value,
        text: e.nombre,
        label: e.label,
        key: e._id
      };
    });

    return res.status(200).json(sorts);
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

const getJurisprudencia = async (req, res) => {
  // const url = `/jurisprudencia-idh?page=${page}&search=${search}&categorias=${categorias.join(',')}pais=${pais.join(',')}documentos=${documentos.join(',')}ordenamiento=${ordenamiento}`;
  const {
    search,
    limit = 10,
    page = 1,
    sort = 'citations',
    categorias = '',
    pais = '',
    documentos = '',
    fechaInicio,
    fechaFinal
  } = req.query;

  /// Ordenamiento de los resultados
  // Mas consultados - > popularity
  // Mas recientes - > date
  // Mas antiguos - > date_asc
  // Relevancia - > score
  // Mas citados - > citations
  try {
    // Construcción de la URI con los parámetros de la consulta
    const uri = `https://jurisprudencia.corteidh.or.cr/search.json?product_id=WW&jurisdiction=EA&textolibre=${search}&bypass_rabl=true&include=parent,abstract,snippet,properties_with_ids,citation_counts&per_page=${limit}&page=${page}&sort=${sort}&include_local_exclusive=true&cbm=6.0|361.0|5.0|9.0|4.0|2.0=0.01|400.0|1.0|0.001|1.5|0.2&type=document&locale=es&hide_ct6=true&t=1722929840&estado=${pais || ''
      }&categoriaCorte=${categorias || ''}&tipoDeDocumento=${documentos || ''
      }&date=${fechaInicio && fechaFinal ? `${fechaInicio}...${fechaFinal}` : ''
      }`;
    console.log(uri);
    const { data } = await axios.get(uri);

    // Extraer los resultados y el conteo total de documentos
    const { results, count } = data;

    // Verificación para asegurar que count esté definido
    const totalDocs = count || 0;

    // Construcción de la respuesta con un totalPage válido
    const response = {
      docs: results?.map((doc, index) => {
        return {
          ...doc,
          numero: (index + 1) * page
        };
      }),
      totalDocs,
      limit: parseInt(limit),
      totalPage: totalDocs > 0 ? Math.ceil(totalDocs / limit) : null,
      page: parseInt(page)
    };

    return res.status(200).json(response);
  } catch (error) {
    console.error('Error en Jurisprudencia:', error);
    return res.status(500).json({ message: error.message });
  }
};

const getJurisprudenciaById = async (req, res) => {
  const { id } = req.params;

  if (!id) {
    return res.status(400).json({ message: 'Es obligatorio el id' });
  }
  try {
    const uri = `https://jurisprudencia.corteidh.or.cr/vid/${id}.json?include=abstract,parent,meta,formats,children,properties_with_ids,library,source&fat=1&locale=es&hide_ct6=true&t=1722929840&`;

    const response = await axios.get(uri);
    const doc = response.data;

    const docs = doc.expressions.map((item, index) => {
      const matchingItems = doc.formats.filter(
        (secondItem) => secondItem.id === item.vid && secondItem.locale === 'es'
      );

      return {
        id: item.vid,
        label: item.label,
        tabConfig: item.tabConfig,
        documents: matchingItems
      };
    });

    const documento = {
      id: doc.id,
      type: doc.type,
      publish: doc.published_at,
      doc_date: doc.doc_date,
      title: doc.title,
      properties: doc.properties,
      docs
    };

    // console.log(documento);

    return res.status(200).json(documento);
  } catch (error) {
    console.error('Error en Jurisprudencia:', error);
    return res.status(500).json({ message: error.message });
  }
};

const getPdf = async (req, res) => {
  const { url } = req.query;

  if (!url) {
    return res.status(400).json({ message: 'Es necesario la url' });
  }

  try {
    const uri = `https://jurisprudencia.corteidh.or.cr/${url}`;

    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    const contentDisposition = response.headers['content-disposition'];
    let filename = 'document.pdf';

    if (contentDisposition && contentDisposition.includes('filename=')) {
      const match = contentDisposition.match(/filename="(.+)"/);
      if (match && match[1]) {
        filename = match[1];
      }
    }

    res.setHeader('Content-Disposition', `inline; filename="${filename}"`);
    res.setHeader('Content-Type', 'application/pdf');
    return res.status(200).send(response.data);
  } catch (error) {
    console.error('Error en Jurisprudencia:', error);
    return res.status(500).json({ message: error.message });
  }
};

const downloadPDF = async (req, res) => {
  const { url } = req.query;

  if (!url) {
    return res.status(400).json({ message: 'Es necesario la url' });
  }

  try {
    const uri = `https://jurisprudencia.corteidh.or.cr/${url}`;

    const response = await axios.get(uri, { responseType: 'arraybuffer' });

    const contentDisposition = response.headers['content-disposition'];
    let filename = 'document.pdf';

    if (contentDisposition && contentDisposition.includes('filename=')) {
      const match = contentDisposition.match(/filename="(.+)"/);
      if (match && match[1]) {
        filename = match[1];
      }
    }

    res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
    res.setHeader('Content-Type', 'application/pdf');
    return res.status(200).send(response.data);
  } catch (error) {
    console.error('Error en Jurisprudencia:', error);
    return res.status(500).json({ message: error.message });
  }
};

module.exports = {
  getJurisprudencia,
  getOrdenamiento,
  getPdf,
  downloadPDF,
  getJurisprudenciaById
};
