const IngresosModel = require('../models//Ingresos.js');
const ExpedientesModel = require('../models/Expedientes.js');
const ClientesModel = require('../models/Clientes.js');
const MovimientosBancos = require('../models/MovimientosBancos.js');
const moment = require('moment-timezone');
const { uploadFile, buildFileUri, deleteFileS3 } = require('../config/s3.js');

const getIngresos = async (req, res) => {
  const { despacho } = req.params;
  const { page = 1, limit = 10, estatus, search, cliente } = req.query;

  const options = {
    page: parseInt(page, 10),
    limit: parseInt(limit, 10),
    populate: ['cuentaBancaria', 'cliente', 'referencia.expediente'],
    sort: { fecha: -1, nombre: 1 }
  };

  const query = {
    despacho
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [
      { comentario: { $regex: search, $options: 'i' } },
      { concepto: { $regex: search, $options: 'i' } },
      { 'cliente.nombre': { $regex: search, $options: 'i' } },
      { 'referencia.titulo': { $regex: search, $options: 'i' } }
    ];
  }

  if (cliente) {
    query.cliente = cliente;
  }

  try {
    const ingresos = await IngresosModel.paginate(query, options);
    res.status(200).json({ ingresos });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createIngresos = async (req, res) => {
  const { despacho } = req.params;
  const { concepto, importe, fecha, referencia, estatus, cuentaBancaria, cliente } = req.body;

  if (!concepto || !importe || !referencia || !cliente) {
    return res.status(400).json({ message: 'Faltan campos por llenar' });
  }

  let documentos = req?.files?.comprobantes ?? [];
  console.log(documentos);
  documentos = Array.isArray(documentos) ? documentos : [documentos];

  try {
    const uploadPromises = documentos.map(file => uploadFile(file, `despachos/${despacho}/ingresos`));
    const fileUploads = await Promise.all(uploadPromises);

    const comprobantes = fileUploads.map((doc) => {
      return {
        nombre: doc.originalName,
        archivo: doc.fileName
      };
    }) || [];

    const obj = {
      despacho,
      concepto,
      importe,
      fecha: moment(fecha).tz('America/Mexico_City').format(),
      referencia: JSON.parse(referencia),
      estatus,
      cliente,
      comprobantes: comprobantes || []
    };

    if (cuentaBancaria) {
      obj.cuentaBancaria = cuentaBancaria;
    }

    console.log(obj);
    const newIngresos = await IngresosModel.create(obj);

    const mov = {
      despacho,
      afectacion: 'Abono',
      ligadoa: 'Ingreso',
      folioLiga: { ingreso: newIngresos._id },
      concepto,
      importe,
      estatus: 'Aplicado'
    };

    if (cuentaBancaria) {
      mov.cuentaBancaria = cuentaBancaria;
    }

    await MovimientosBancos.create(mov);
    res.status(201).json({ newIngresos, message: 'Ingreso creado' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getExpedientesSinPaginar = async (req, res) => {
  const { despacho } = req.params;
  const { cliente } = req.query;
  try {
    const expedientes = await ExpedientesModel.find({
      despacho,
      estatus: 'Activo',
      cliente
    }, '_id titulo').sort({ titulo: 1 });
    res.status(200).json({ expedientes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getClientes = async (req, res) => {
  const { despacho } = req.params;
  try {
    const clientes = await ClientesModel.find({ despacho, estatus: 'Activo' }, '_id nombre').sort({ nombre: 1 });
    res.status(200).json({ clientes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteIngresos = async (req, res) => {
  const { id } = req.params;
  try {
    const ingreso = await IngresosModel.findByIdAndRemove(id);

    if (!ingreso) {
      return res.status(404).json({ message: 'No se encontró el ingreso' });
    }

    if (ingreso.estatus === 'Facturado') {
      return res.status(400).json({ message: 'No se puede eliminar un ingreso facturado' });
    }

    const mov = await MovimientosBancos.findOneAndRemove({ 'folioLiga.ingreso': id });

    if (mov) {
      await mov.remove();
    }

    ingreso.delete();
    res.status(200).json({ message: 'Ingreso eliminado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getIngresoById = async (req, res) => {
  const { id } = req.params;
  try {
    const ingreso = await IngresosModel.findById(id);

    if (!ingreso) {
      return res.status(404).json({ message: 'No se encontró el ingreso' });
    }

    ingreso.comprobantes.forEach((doc) => {
      doc.archivo = buildFileUri(`despachos/${ingreso.despacho}/ingresos/${doc.archivo}`);
      console.log(doc.archivo);
    });

    res.status(200).json({ ingreso });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteComprobante = async (req, res) => {
  const { id } = req.params;

  try {
    const ingreso = await IngresosModel.findOne({ 'comprobantes._id': id });

    if (!ingreso) {
      return res.status(404).json({ message: 'Ingreso no encontrado' });
    }

    const comprobante = ingreso.comprobantes.id(id);

    if (comprobante) {
      await deleteFileS3(`despachos/${ingreso.despacho}/ingresos/${comprobante.archivo}`);
      comprobante.remove();
      await ingreso.save();
    }

    res.status(200).json({ message: 'Comprobante eliminado' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getIngresos,
  createIngresos,
  getExpedientesSinPaginar,
  getClientes,
  deleteIngresos,
  getIngresoById,
  deleteComprobante
};
