const IncidentesModel = require("../models/IncidenciasDespacho.js");

const getIncidencias = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search, materia } = req.query;

  const options = {
    page,
    limit: 15,
    sort: { materia: 1, nombre: 1 },
    populate: "materia",
  };

  const query = {
    despacho,
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.nombre = { $regex: search, $options: "i" };
  }

  if (materia) {
    query.materia = materia;
  }

  try {
    const incidencias = await IncidentesModel.paginate(query, options);
    res.status(200).json({ incidencias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getIncidenciasSinPaginar = async (req, res) => {
  const { despacho } = req.params;
  const { materia } = req.query;

  try {
    const query = {
      despacho,
    };

    if (materia) {
      query.materia = materia;
    }

    query.estatus = "Activo";

    const incidenciasFind = await IncidentesModel.find(query).sort({
      nombre: 1,
    });

    const incidencias = incidenciasFind.map((materia) => {
      return {
        value: materia._id,
        label: materia.nombre,
        key: materia._id,
      };
    });

    res.status(200).json({ incidencias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createIncidencias = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, estatus, materia } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }

  if (!materia) {
    return res.status(400).json({ message: "La materia es requerida" });
  }

  const obj = {
    despacho,
    nombre,
    materia,
    estatus: estatus || "Activo",
  };

  try {
    const incidencias = await IncidentesModel.create(obj);
    res
      .status(201)
      .json({ incidencias, message: "Contratiempo creada correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getIncidenciaById = async (req, res) => {
  const { despacho, id } = req.params;

  try {
    const incidencias = await IncidentesModel.findOne({ _id: id, despacho });

    if (!incidencias) {
      return res
        .status(404)
        .json({ message: "No se encontró el contratiempo" });
    }

    res.status(200).json(incidencias);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateIncidente = async (req, res) => {
  const { id, despacho } = req.params;
  const { nombre, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }

  try {
    const incidente = await IncidentesModel.findOne({ _id: id, despacho });

    if (!incidente) {
      return res
        .status(404)
        .json({ message: "No se encontró el contratiempo" });
    }

    const update = {
      nombre,
      estatus: estatus || "Activo",
    };

    await IncidentesModel.findOneAndUpdate({ _id: id, despacho }, update, {
      new: true,
    });

    res.status(200).json({ message: "Contratiempo actualizado" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteIncidente = async (req, res) => {
  const { id, despacho } = req.params;

  try {
    const incidente = await IncidentesModel.findOne({ _id: id, despacho });

    if (!incidente) {
      return res
        .status(404)
        .json({ message: "No se encontró el contratiempo" });
    }

    await IncidentesModel.findOneAndDelete({ _id: id, despacho });

    return res.status(200).json({ message: "Contratiempo eliminado" });
  } catch (error) {
    res.status(404).json({ message: "Ocurrió un error inténtelo de nuevo." });
  }
};

module.exports = {
  getIncidencias,
  getIncidenciasSinPaginar,
  createIncidencias,
  getIncidenciaById,
  updateIncidente,
  deleteIncidente,
};
