const ExpedientesGastos = require('../models/ExpedientesGastos.js');
const mongoose = require('mongoose');
const { formatDate, formatMoney } = require('../config/functionsGlobal.js');
const { buildFileUri } = require('../config/s3.js');

const getGastosTramites = async (req, res) => {
  const { despacho } = req.params;
  const { search, estatus, expediente, fechaInicio, fechaFinal, page } = req.query;

  try {
    const limit = 10;
    const parsedPage = parseInt(page) || 1;
    const skip = (parsedPage - 1) * limit;

    const query = {
      despacho: mongoose.Types.ObjectId(despacho),
      tipo: 'Gasto'
    };

    if (search) {
      query.$or = [
        { concepto: { $regex: search, $options: 'i' } }
      ];
    }

    if (estatus) {
      query.estatus = estatus;
    }

    if (expediente) {
      query.expediente = mongoose.Types.ObjectId(expediente);
    }

    if (fechaInicio && fechaFinal) {
      query.fecha = {
        $gte: new Date(fechaInicio),
        $lte: new Date(fechaFinal)
      };
    }

    const gastosPipeline = [
      { $match: query },
      {
        $lookup: {
          from: 'expedientes',
          localField: 'expediente',
          foreignField: '_id',
          as: 'expediente'
        }
      },
      { $unwind: '$expediente' },
      { $match: { 'expediente.estatus': 'Activo' } },
      {
        $project: {
          concepto: 1,
          importe: 1,
          fecha: 1,
          'expediente.titulo': 1,
          'expediente.numeroExpediente': 1,
          'adjunto.archivo': 1,
          'adjunto.nombre': 1
        }
      },
      { $sort: { fecha: -1 } },
      { $skip: skip },
      { $limit: limit }
    ];

    const totalPipeline = [
      { $match: query },
      {
        $lookup: {
          from: 'expedientes',
          localField: 'expediente',
          foreignField: '_id',
          as: 'expediente'
        }
      },
      { $unwind: '$expediente' },
      { $match: { 'expediente.estatus': 'Activo' } },
      { $group: { _id: null, total: { $sum: '$importe' }, count: { $sum: 1 } } }
    ];

    const [gastos, totalAggregate] = await Promise.all([
      ExpedientesGastos.aggregate(gastosPipeline),
      ExpedientesGastos.aggregate(totalPipeline)
    ]);

    const total = totalAggregate.length > 0 ? totalAggregate[0].total : 0;
    const totalDocs = totalAggregate.length > 0 ? totalAggregate[0].count : 0;

    const totalPages = Math.ceil(totalDocs / limit);

    gastos.forEach(gasto => {
      gasto.fecha = formatDate(gasto.fecha);

      gasto.importe = formatMoney(gasto.importe);
      if (gasto.adjunto?.archivo) {
        gasto.adjunto.archivo = buildFileUri(`despachos/${despacho}/expedientes/${expediente}/gastos/${gasto.adjunto.archivo}`);
      }
    });

    res.status(200).json({
      gastos: {
        docs: gastos,
        totalDocs,
        limit,
        totalPages,
        page: parsedPage,
        pagingCounter: (parsedPage - 1) * limit + 1,
        hasPrevPage: parsedPage > 1,
        hasNextPage: parsedPage < totalPages,
        prevPage: parsedPage > 1 ? parsedPage - 1 : null,
        nextPage: parsedPage < totalPages ? parsedPage + 1 : null
      },
      total: formatMoney(total)
    });
  } catch (error) {
    console.error('Error en getGastosTramites:', error);
    return res.status(500).json({ message: error.message });
  }
};

module.exports = {
  getGastosTramites
};
