const GastosModel = require("../models/Gastos.js");
const MovimientosBancos = require("../models/MovimientosBancos.js");
const { uploadFile, buildFileUri, deleteFileS3 } = require("../config/s3.js");
const moment = require("moment-timezone");
const getGastos = async (req, res) => {
  const { despacho } = req.params;
  const { page = 1, limit = 10, estatus, search } = req.query;

  const options = {
    page: parseInt(page, 10),
    limit: parseInt(limit, 10),
    populate: ["cuentaBancaria"],
    sort: {
      estatus: -1,
      fecha: -1,
      _id: -1,
    },
  };

  const query = {
    despacho,
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [
      { referencia: { $regex: search, $options: "i" } },
      { comentario: { $regex: search, $options: "i" } },
    ];
  }

  try {
    const gastos = await GastosModel.paginate(query, options);
    res.status(200).json({ gastos });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createGastos = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: "Falta el despacho" });
  }

  const {
    cuentaBancaria,
    conceptos,
    fecha,
    total,
    referencia,
    creadoPor,
    comentario,
    estatus,
    usuarioAsignado,
  } = req.body;

  try {
    const conceptos2 = JSON.parse(conceptos) || [];

    let documentos = req?.files?.comprobantes ?? [];
    documentos = Array.isArray(documentos) ? documentos : [documentos];

    if (conceptos2.length === 0) {
      return res.status(400).json({ message: "Faltan los detalles" });
    }

    const uploadPromises = documentos.map((file) =>
      uploadFile(file, `despachos/${despacho}/gastos`)
    );
    const fileUploads = await Promise.all(uploadPromises);

    const comprobantes = fileUploads.map((doc) => {
      return {
        nombre: doc.originalName,
        archivo: doc.fileName,
      };
    });

    const obj = {
      despacho,
      conceptos: conceptos2,
      fecha: moment(fecha).tz("America/Mexico_City").format(),
      total,
      referencia,
      creadoPor,
      comentario,
      estatus,
      comprobantes,
      usuarioAsignado: usuarioAsignado || null,
      cuntaBancaria: cuentaBancaria || null,
    };

    const newGasto = await GastosModel.create(obj);

    if (cuentaBancaria) {
      const mov = {
        despacho,
        afectacion: "Cargo",
        ligadoa: "Egreso",
        folioLiga: { gasto: newGasto._id },
        concepto: referencia,
        importe: total,
        estatus: "Aplicado",
        cuentaBancaria,
      };

      await MovimientosBancos.create(mov);
    }

    return res.status(201).json({ newGasto, message: "Gasto creado" });
  } catch (error) {
    console.log(error);
    return res.status(409).json({ message: error.message }); // Ensure we return to avoid attempting to set headers after they're already sent
  }
};

const getGasto = async (req, res) => {
  const { id } = req.params;

  try {
    const gasto = await GastosModel.findById(id);

    if (!gasto) {
      return res.status(404).json({ message: "Gasto no encontrado" });
    }

    gasto.comprobantes = gasto.comprobantes.map((doc) => {
      if (doc?.archivo) {
        doc.archivo = buildFileUri(
          `despachos/${gasto.despacho}/gastos/${doc.archivo}`
        );
      }
      return doc;
    });

    res.status(200).json(gasto);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteGasto = async (req, res) => {
  const { id } = req.params;

  try {
    const gastos = await GastosModel.findById(id);

    if (gastos) {
      const query = { folioLiga: { gasto: gastos._id } };
      const mov = await MovimientosBancos.findOne(query);

      if (mov) {
        await mov.remove();
      }
      await gastos.remove();
    }

    res.status(200).json({ message: "Gasto eliminado" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateGasto = async (req, res) => {
  const { id } = req.params;
  const {
    cuentaBancaria,
    conceptos = [],
    fecha = moment().tz("America/Mexico_City").format(),
    total,
    referencia,
    comentario,
    estatus,
    despacho,
  } = req.body;

  if (!id) {
    return res.status(400).json({ message: "Falta el id" });
  }

  try {
    const gasto = await GastosModel.findById(id);

    if (!gasto) {
      return res.status(404).json({ message: "Gasto no encontrado" });
    }

    const conceptos2 = JSON.parse(conceptos) || [];

    if (conceptos2.length === 0) {
      return res.status(400).json({ message: "Faltan los detalles" });
    }

    let documentos = req?.files?.comprobantes ?? [];
    documentos = Array.isArray(documentos) ? documentos : [documentos];

    const uploadPromises = documentos.map((file) =>
      uploadFile(file, `despachos/${despacho}/gastos`)
    );
    const fileUploads = await Promise.all(uploadPromises);

    if (documentos.length > 0) {
      const comprobantes = fileUploads.map((doc) => {
        return {
          nombre: doc.originalName,
          archivo: doc.fileName,
        };
      });
      gasto.comprobantes = gasto.comprobantes.concat(comprobantes);
    }

    gasto.conceptos = conceptos2;
    gasto.fecha = moment(fecha).tz("America/Mexico_City").format();
    gasto.total = total;
    gasto.referencia = referencia;
    gasto.comentario = comentario;
    gasto.estatus = estatus;

    if (cuentaBancaria !== "null" && cuentaBancaria) {
      gasto.cuentaBancaria = cuentaBancaria;
    }

    await gasto.save();

    if (cuentaBancaria !== "null" && cuentaBancaria) {
      const findMov = await MovimientosBancos.findOne({
        "folioLiga.gasto": gasto._id,
      });

      if (findMov) {
        findMov.importe = total;
        findMov.concepto = referencia;
        findMov.cuentaBancaria = cuentaBancaria;
        findMov.estatus = estatus === "Vigente" ? "Aplicado" : estatus;
        findMov.save();
      } else {
        const mov = {
          despacho: gasto.despacho,
          afectacion: "Cargo",
          ligadoa: "Egreso",
          folioLiga: { gasto: gasto._id },
          concepto: gasto.referencia,
          importe: gasto.total,
          estatus: estatus === "Vigente" ? "Aplicado" : estatus,
          cuentaBancaria,
        };

        await MovimientosBancos.create(mov);
      }
    } else {
      const mov = await MovimientosBancos.findOne({
        "folioLiga.gasto": gasto._id,
      });

      if (mov) {
        await mov.remove();
      }
    }

    res.status(200).json({ gasto, message: "Gasto actualizado" });
  } catch (error) {
    console.log(error);
    res.status(409).json({ message: error.message });
  }
};

const deleteComprobante = async (req, res) => {
  const { id } = req.params;

  try {
    const gasto = await GastosModel.findOne({ "comprobantes._id": id });

    if (!gasto) {
      return res.status(404).json({ message: "Gasto no encontrado" });
    }

    const comprobante = gasto.comprobantes.id(id);

    if (comprobante) {
      console.log(`despachos/${gasto.despacho}/gastos/${comprobante.archivo}`);
      await deleteFileS3(
        `despachos/${gasto.despacho}/gastos/${comprobante.archivo}`
      );
      comprobante.remove();
      await gasto.save();
    }

    res.status(200).json({ message: "Comprobante eliminado" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getGastos,
  createGastos,
  getGasto,
  deleteGasto,
  updateGasto,
  deleteComprobante,
};
