const { buildFileUri } = require('../config/s3.js');
const Chat = require('../models/ChatV2.js');
const ChatViejo = require('../models/Chat.js');
const Despachos = require('../models/Despachos.js');
const OpenAI = require('openai');
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY
});

const historial = async (req, res) => {
  const { despacho, usuario } = req.params;
  const { page = 1 } = req.query;

  try {
    const options = {
      page,
      limit: 10,
      sort: {
        timestamp: -1,
        _id: 1
      },
      populate: {
        path: 'usuario',
        select: 'nombre foto'
      },
      lean: true
    };

    const query = {
      despacho,
      estatus: 'Activo',
      usuario
    };

    const chats = await Chat.paginate(query, options);

    const mensajes = chats.docs.map(chat => {
      const { _id, role, content, timestamp, usuario } = chat;
      return {
        _id,
        position: role === 'user' ? 'right' : 'left',
        text: content,
        timestamp,
        favorito: !!chat.favorito,
        usuario: usuario
          ? { nombre: usuario.nombre, foto: usuario.foto ? buildFileUri(`usuarios/${usuario.foto}`) : buildFileUri('default/icono_usuario_100x100_04.jpg') }
          : null
      };
    });

    chats.docs = mensajes.sort((a, b) => a.timestamp - b.timestamp);

    res.status(200).json(chats);
  } catch (error) {
    console.error('Error fetching chat history:', error);
    res.status(500).json({ message: error });
  }
};

const contadorChat = async (req, res) => {
  const { despacho } = req.params;
  try {
    const findDespacho = await Despachos.findOne({ _id: despacho });

    if (!findDespacho) {
      return res.status(404).json({ error: 'Despacho no encontrado' });
    }

    const { contadorChatGPT } = findDespacho;

    const gpt3 = contadorChatGPT.diario.limite - contadorChatGPT?.diario.contador;

    const gpt4 = contadorChatGPT?.comprado.limite - contadorChatGPT?.comprado.contador;

    res.json({
      gpt3,
      gpt4
    });
  } catch (error) {
    console.error('Error fetching chat history:', error);
    res.status(500).json({ message: error });
  }
};

const deleteMessages = async (req, res) => {
  const { _id } = req.params;

  try {
    const findChat = await Chat.findById(_id);

    if (!findChat) {
      return res.status(404).json({ error: 'Chat no encontrado' });
    }

    findChat.estatus = 'Inactivo';
    await findChat.save();
    res.json({ message: 'Chat eliminado' });
  } catch (error) {
    console.error('Error deleting messages:', error);
    res.status(500).json({ message: error });
  }
};
const meterChatsViejosAlNuevo = async (req, res) => {
  // Solo guarda los mensajes si no existen previamente en el nuevo esquema
  const existingChat = await Chat.countDocuments();
  if (existingChat > 0) return;

  const chats = await ChatViejo.find().sort({ timestamp: 1 }).exec();

  for (const chat of chats) {
    const { messages, despacho } = chat;

    // Crea un array para los nuevos mensajes
    let usuarioPadre = null;
    const nuevosMensajes = messages.map(message => {
      const { usuario, content, timestamp } = message;
      const role = usuario ? 'user' : 'assistant';
      // usuarioPadre = usuario ;
      if (usuario) {
        usuarioPadre = usuario;
      }

      return {
        role,
        content,
        timestamp,
        usuario: usuarioPadre,
        despacho
      };
    });

    await Chat.insertMany(nuevosMensajes); // Inserta todos los mensajes de un chat a la vez
  }

  console.log('Chats viejos metidos al nuevo');

  res.json({ message: 'Chats viejos metidos al nuevo' });
};

// const lista = async (req, res) => {
//   const { despacho } = req.params;
//   try {
//     const chats = await Chat.find({ despacho }).sort({ 'messages.timestamp': -1 }).exec();
//     const listaChats = chats.map(chat => {
//       const primerMensaje = chat.messages.length > 0 ? chat.messages[0].content : 'No hay mensajes';
//       return {
//         id: chat.conversationId,
//         name: primerMensaje,
//         timestamp: chat.messages[0]?.timestamp || new Date()
//       };
//     });
//     res.json(listaChats);
//   } catch (error) {
//     console.error('Error fetching chat list:', error);
//     res.status(500).json({ error: 'Error al obtener la lista de chats' });
//   }
// };

const obtenerResumenIA = async (req, res) => {
  const { despacho, message } = req.body;

  if (!despacho) {
    return res.status(400).json({
      message: 'Es obligatorio el despacho'
    });
  }

  if (!message) {
    return res.status(400).json({
      message: 'Es obligatorio el mensaje'
    });
  }

  const isChatValid = await validarContadorChat({ despacho });

  console.log(isChatValid);

  if (!isChatValid.valido) {
    return res.status(400).json({
      message: 'No se puede realizar la petición, se ha excedido el número de mensajes permitidos'
    });
  }

  const systemMessage = {
    role: 'system',
    content: 'Quiero que hagas un resumen de los puntos y texto que te proporciono. Asegúrate de formatear la respuesta usando Markdown, con listas, encabezados y una estructura clara.'
  };

  const response = await openai.chat.completions.create({
    model: isChatValid.tipo,
    messages: [
      systemMessage,
      {
        role: 'user',
        content: message,
        usuario: 'user'
      }
    ],
    max_tokens: 500,
    temperature: 0.5
  });

  const gptMessage = response.choices[0].message.content.trim().replace('```', '').replace('markdown\n', '');

  return res.status(200).json({
    message: gptMessage
  });
};

const validarContadorChat = async ({ despacho }) => {
  const findDespacho = await Despachos.findById(despacho, 'contadorChatGPT');

  if (!findDespacho) {
    return {
      valido: false,
      mensaje: 'No se encontró el despacho',
      tipo: null
    };
  }

  const diario = findDespacho.contadorChatGPT.diario || null;
  const comprado = findDespacho.contadorChatGPT.comprado || null;

  // Validación inicial
  if (diario.contador < diario.limite) {
    // 'gpt-3.5-turbo' : 'gpt-4o-mini',
    diario.contador += 1;

    await Despachos.findByIdAndUpdate(despacho, {
      $set: {
        'contadorChatGPT.diario.contador': diario.contador
      }
    });

    return {
      valido: true,
      mensaje: 'Se puede realizar la petición',
      tipo: 'gpt-3.5-turbo'
    };
  }

  if (comprado.contador < comprado.limite) {
    comprado.contador += 1;

    await Despachos.findByIdAndUpdate(despacho, {
      $set: {
        'contadorChatGPT.comprado.contador': comprado.contador
      }
    });

    return {
      valido: true,
      mensaje: 'Se puede realizar la petición',
      tipo: 'gpt-4o-mini'
    };
  }
};

module.exports = {
  historial,
  contadorChat,
  deleteMessages,
  meterChatsViejosAlNuevo,
  obtenerResumenIA
  // lista,

};
