const FiscaliasModel = require('../models/Fiscalias.js');

const getFiscalias = async (req, res) => {
  const { page = 1, search, estado, estatus } = req.query;
  const options = {
    page,
    limit: 20,
    sort: { estado: 1, nombre: 1 },
    populate: 'estado'
  };

  const query = {};

  if (search) {
    query.$or = [
      { nombre: { $regex: search, $options: 'i' } }
    ];
  }

  if (estado) {
    query.estado = estado;
  }

  if (estado) {
    query.estado = estado;
  }

  if (estatus) {
    query.estatus = estatus;
  }

  try {
    const findFiscalias = await FiscaliasModel.paginate(query, options);
    res.status(200).json({ fiscalias: findFiscalias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createFiscalias = async (req, res) => {
  const { nombre, direccion, telefonos, estado, paginaWeb } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }

  try {
    const newTelefonos = telefonos.length > 0
      ? telefonos.split(',').map(tel => tel.trim())
      : [];

    const newLegislacion = new FiscaliasModel({
      nombre,
      direccion,
      telefonos: newTelefonos,
      estado,
      liga: paginaWeb
    });

    await newLegislacion.save();
    res.status(201).json({ message: 'Legislación creada' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateFiscalia = async (req, res) => {
  const { id } = req.params;
  const { nombre, direccion, telefonos, estado, paginaWeb, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }

  const newTelefonos = telefonos.length > 0
    ? telefonos.split(',').map(tel => tel.trim())
    : [];

  try {
    const updatedFiscalia = await FiscaliasModel.findByIdAndUpdate(id, {
      nombre,
      estado,
      liga: paginaWeb,
      estatus: estatus || 'Activo',
      direccion,
      telefonos: newTelefonos

    }, { new: true });

    res.status(200).json({ message: 'Fiscalía actualizada correctamente', fiscalia: updatedFiscalia });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getLegislacionById = async (req, res) => {
  const { id } = req.params;
  try {
    const fiscalia = await FiscaliasModel.findById(id);

    return res.status(200).json(fiscalia);
  } catch (error) {
    return res.status(404).json({ message: 'Fiscalía no encontrada' });
  }
};

const deleteFiscalia = async (req, res) => {
  const { id } = req.params;

  try {
    await FiscaliasModel.findByIdAndDelete(id);
    res.status(200).json({ message: 'Fiscalía eliminada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getFiscalias,
  createFiscalias,
  updateFiscalia,
  getLegislacionById,
  deleteFiscalia
};
