const ExpedientesUsuariosModel = require("../models/ExpedientesUsuarios.js");
const UsuarioModel = require("../models/Usuarios.js");
const ExpedienteModel = require("../models/Expedientes.js");
const {
  InvitacionExpedienteHTML,
} = require("../Mail/InvitacionExpedienteHTML.js");
const { sendMail } = require("../config/mail.js");
const { buildFileUri } = require("../config/s3.js");
const { notificarExpedienteUsuario } = require("./Expedientes.Controller.js");
const DESPACHO_APP = process.env.DESPACHO_APP;

const getUsuarios = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 20,
    populate: {
      path: "usuario",
      select: "nombre apellidoPaterno apellidoMaterno email foto",
    },
    select: "usuario notificaciones",
  };

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  const query = { despacho, expediente };

  try {
    const usuarios = await ExpedientesUsuariosModel.paginate(query, options);

    usuarios.docs.forEach((usuario) => {
      if (!usuario?.usuario) {
        return;
      }
      if (usuario?.usuario?.foto) {
        usuario.usuario.foto = buildFileUri(`usuarios/${usuario.usuario.foto}`);
      } else {
        usuario.usuario.foto = buildFileUri(
          "default/icono_usuario_100x100_04.jpg"
        );
      }
    });

    res.status(200).json({ usuarios });
  } catch (error) {
    console.log("error", error);
    res.status(404).json({ message: error.message });
  }
};

const getUsuarioExpedienteSinAsignar = async (req, res) => {
  const { despacho, expediente } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  const query = {
    estatus: "Activo",
    despacho,
  };

  try {
    const findUsuariosActivos = await UsuarioModel.find(query)
      .select([
        "_id",
        "nombre",
        "apellidoPaterno",
        "apellidoMaterno",
        "email",
        "foto",
      ])
      .sort({ nombre: 1 });

    const queryEU = {
      despacho,
      expediente,
    };
    const findUsuariosAsignados = await ExpedientesUsuariosModel.find(queryEU);

    const usuarios = findUsuariosActivos.filter((usuario) => {
      return !findUsuariosAsignados.some((usuarioAsignado) =>
        usuarioAsignado.usuario.equals(usuario._id)
      );
    });

    usuarios.forEach((usuario) => {
      if (usuario?.foto) {
        usuario.foto = buildFileUri(`usuarios/${usuario.foto}`);
      } else {
        usuario.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
      }
    });
    res.status(200).json({ usuarios });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteUsuarioExpediente = async (req, res) => {
  try {
    const { _id } = req.params;
    const { usuario } = req.query;

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    const usuarioExpediente = await ExpedientesUsuariosModel.findByIdAndDelete(
      _id
    ).populate("usuario", "nombre apellidoPaterno apellidoMaterno");

    if (!usuarioExpediente) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    const { despacho, expediente } = usuarioExpediente;
    const {
      nombre = "",
      apellidoPaterno = "",
      apellidoMaterno = "",
    } = usuarioExpediente.usuario;

    const descripcionMovimiento = `Se ha eliminado a ${nombre} ${
      apellidoPaterno || ""
    } ${apellidoMaterno || ""} del expediente`;

    const tituloMovimiento = "Usuarios";

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(200).json({ message: "Usuario eliminado" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createUsuariosExpediente = async (req, res) => {
  try {
    const { despacho, expediente } = req.params;

    const {
      usuarios: usuariosIDS = [],
      permisos,
      notificaciones = true,
      usuario: responsable,
    } = req.body;

    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!usuariosIDS.length) {
      return res.status(400).json({ message: "Los usuarios son requeridos" });
    }

    const usuariosObjExpedientes = usuariosIDS.map((usuario) => {
      return {
        despacho,
        expediente,
        usuario,
        permisos,
        notificaciones,
      };
    });

    const usuariosExpedientes = await ExpedientesUsuariosModel.insertMany(
      usuariosObjExpedientes
    );

    const findUsuarios = await UsuarioModel.find({
      _id: { $in: usuariosIDS },
    }).select("email nombre apellidoPaterno apellidoMaterno");
    const findExpediente = await ExpedienteModel.findById(expediente).select(
      "titulo"
    );

    if (!findExpediente) {
      return res.status(404).json({ message: "Expediente no encontrado" });
    }

    const { titulo: nombreExpediente } = findExpediente;

    findUsuarios.forEach((usuario) => {
      const {
        email,
        nombre = "",
        apellidoPaterno = "",
        apellidoMaterno = "",
      } = usuario;

      const html = InvitacionExpedienteHTML({
        nombreDestinatario: `${nombre} ${apellidoPaterno} ${apellidoMaterno}`,
        nombreExpediente,
        enlaceExpediente: `${DESPACHO_APP}/expedientes/${expediente}/editar`,
      });

      sendMail(html, "Invitación a expediente", email);

      const descripcionMovimiento = `Se ha agregado a ${nombre} ${
        apellidoPaterno || ""
      } ${apellidoMaterno || ""} al expediente`;
      const tituloMovimiento = "Usuarios";

      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario: responsable,
      });
    });

    res.status(201).json({ usuariosExpedientes });
  } catch (error) {
    console.log("error", error);
    res.status(409).json({ message: error.message });
  }
};

const getUsuariosExpedienteSinPaginar = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { sin } = req.query;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  const query = { despacho, expediente };

  if (sin) {
    query.usuario = { $ne: sin };
  }
  try {
    const usuarios = await ExpedientesUsuariosModel.find(query).populate(
      "usuario",
      "nombre apellidoPaterno apellidoMaterno email foto"
    );

    usuarios.forEach((usuario) => {
      if (usuario?.usuario?.foto) {
        usuario.usuario.foto = buildFileUri(`usuarios/${usuario.usuario.foto}`);
      } else {
        usuario.usuario.foto = buildFileUri(
          "default/icono_usuario_100x100_04.jpg"
        );
      }
    });

    res.status(200).json({ usuarios });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateUsuarioExpediente = async (req, res) => {
  try {
    const { _id } = req.params;

    const { permisos, notificaciones, responsable } = req.body;

    const usuarioExpediente = await ExpedientesUsuariosModel.findByIdAndUpdate(
      _id,
      { permisos, notificaciones },
      { new: true }
    ).populate("usuario", "nombre apellidoPaterno apellidoMaterno");

    if (!usuarioExpediente) {
      return res.status(404).json({ message: "Usuario no encontrado" });
    }

    const { despacho, expediente } = usuarioExpediente;
    const {
      nombre = "",
      apellidoPaterno = "",
      apellidoMaterno = "",
    } = usuarioExpediente.usuario;

    const descripcionMovimiento = `Se ha actualizado a ${nombre} ${
      apellidoPaterno || ""
    } ${apellidoMaterno || ""} en el expediente`;
    const tituloMovimiento = "Usuarios";

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario: responsable,
    });

    res.status(200).json({ usuarioExpediente });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getUsuarios,
  getUsuarioExpedienteSinAsignar,
  deleteUsuarioExpediente,
  createUsuariosExpediente,
  getUsuariosExpedienteSinPaginar,
  updateUsuarioExpediente,
};
