const PautasModel = require("../models/ExpedientesPautas.js");
const { buildFileUri } = require("../config/s3.js");
const { notificarExpedienteUsuario } = require("./Expedientes.Controller.js");

const create = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { nombre, usuario } = req.body;
  try {
    if (!nombre)
      return res
        .status(400)
        .json({ message: "El nombre de la pauta es requerido" });

    if (!usuario)
      return res.status(400).json({ message: "El usuario es requerido" });

    const pauta = {
      nombre,
      creadoPor: usuario,
      expediente,
      despacho,
    };

    const newPauta = await PautasModel.create(pauta);

    const tituloMovimiento = "Minutas";
    const descripcionMovimiento = `Se creó la pauta ${nombre}`;

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(201).json(newPauta);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getPautas = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 20,
    sort: { nombre: 1 },
    populate: [
      {
        path: "creadoPor",
        select: "nombre apellidoPaterno apellidoMaterno foto",
      },
      {
        path: "editadoPor",
        select: "nombre apellidoPaterno apellidoMaterno foto",
      },
    ],
    select: "nombre ultimoMovimiento fechaCreacion creadoPor editadoPor",
  };

  const query = {
    despacho,
    expediente,
  };

  try {
    const pautas = await PautasModel.paginate(query, options);

    pautas.docs.forEach((pauta) => {
      if (pauta.creadoPor.foto) {
        pauta.creadoPor.foto = buildFileUri(`usuarios/${pauta.creadoPor.foto}`);
      }

      if (pauta.editadoPor) {
        pauta.editadoPor.foto = buildFileUri(
          `usuarios/${pauta.editadoPor.foto}`
        );
      }
    });

    res.status(200).json({ pautas });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getPautaById = async (req, res) => {
  const { id } = req.params;
  try {
    const pauta = await PautasModel.findById(id).populate([
      { path: "expediente", select: "titulo" },
    ]);
    if (!pauta) return res.status(404).json({ message: "Pauta no encontrada" });

    res.status(200).json(pauta);
  } catch (error) {
    res.status(404).json({ message: "No se pudo encontrar la pauta" });
  }
};

const updatePauta = async (req, res) => {
  const { id } = req.params;
  const { documento = "" } = req.body;

  if (!documento)
    return res.status(400).json({ message: "El documento es requerido" });

  try {
    const pauta = await PautasModel.findByIdAndUpdate(
      id,
      { documento, ultimoMovimiento: new Date() },
      { new: true }
    );

    res.status(200).json(pauta);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deletePauta = async (req, res) => {
  const { id } = req.params;
  const { responsable } = req.query;
  try {
    const pauta = await PautasModel.findByIdAndRemove(id);

    if (!pauta) return res.status(404).json({ message: "Pauta no encontrada" });

    const tituloMovimiento = "Minutas";
    const descripcionMovimiento = `Se eliminó la pauta ${pauta.nombre}`;

    notificarExpedienteUsuario({
      despacho: pauta.despacho,
      expediente: pauta.expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario: responsable,
    });

    res.status(200).json({ message: "Pauta eliminada" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  create,
  getPautas,
  getPautaById,
  updatePauta,
  deletePauta,
};
