const RecursosIncidenciasExpedienteModel = require("../models/ExpedientesRecursosIncidencias.js");
const { formatDateTime } = require("../config/functionsGlobal.js");
const getRecursosIncidencias = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { tipo, page = 1 } = req.query;

  try {
    const options = {
      page,
      limit: 15,
      sort: { fecha: -1 },
      populate: [
        { path: "creadoPor", select: "nombre apellidoPaterno apellidoMaterno" },
        { path: "recurso" },
        { path: "incidencia" },
      ],
    };
    const query = {
      despacho,
      expediente,
    };

    if (tipo) {
      query.tipo = tipo;
    }

    const recursosIncidencias =
      await RecursosIncidenciasExpedienteModel.paginate(query, options);

    recursosIncidencias.docs = recursosIncidencias.docs.map(
      (recursoIncidencia) => {
        return {
          ...recursoIncidencia._doc,
          fecha: formatDateTime(recursoIncidencia.fecha),
        };
      }
    );

    res.status(200).json({ recursosIncidencias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const nuevoRecursoIncidencia = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { tipo, recurso, incidencia, comentario, creadoPor, fecha } = req.body;

  const nuevoRecursoIncidencia = new RecursosIncidenciasExpedienteModel({
    despacho,
    expediente,
    tipo,
    recurso,
    fecha,
    incidencia,
    creadoPor,
    comentario,
  });

  try {
    await nuevoRecursoIncidencia.save();
    res.status(201).json(nuevoRecursoIncidencia);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const eliminarRecursoIncidencia = async (req, res) => {
  const { despacho, expediente, _id } = req.params;

  try {
    const recursoIncidencia =
      await RecursosIncidenciasExpedienteModel.findOneAndDelete({
        _id,
        despacho,
        expediente,
      });

    if (!recursoIncidencia) {
      return res
        .status(404)
        .json({ message: "Recurso o incidencia no encontrado" });
    }

    res
      .status(200)
      .json({ message: "Recurso o contratiempo eliminado correctamente" });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const actualizarRecursoIncidencia = async (req, res) => {
  const { despacho, expediente, _id } = req.params;
  const {
    tipo,
    recurso = null,
    incidencia = null,
    comentario,
    fecha,
  } = req.body;

  try {
    const recursoIncidencia =
      await RecursosIncidenciasExpedienteModel.findOneAndUpdate(
        { _id, despacho, expediente },
        {
          tipo,
          recurso: recurso ? recurso : null,
          incidencia: incidencia ? incidencia : null,
          comentario,
          fecha,
        },
        { new: true }
      );

    if (!recursoIncidencia) {
      return res
        .status(404)
        .json({ message: "Recurso o incidencia no encontrado" });
    }

    res.status(200).json(recursoIncidencia);
  } catch (error) {
    console.log(error);
    res.status(500).json({ message: error.message });
  }
};

module.exports = {
  getRecursosIncidencias,
  nuevoRecursoIncidencia,
  eliminarRecursoIncidencia,
  actualizarRecursoIncidencia,
};
