const ExpedienteGastosModel = require("../models/ExpedientesGastos.js");
const { uploadFile, buildFileUri, deleteFileS3 } = require("../config/s3.js");
const { formatDate, formatMoney } = require("../config/functionsGlobal.js");
const moment = require("moment-timezone");
const { notificarExpedienteUsuario } = require("./Expedientes.Controller.js");

const create = async (req, res) => {
  const { despacho, expediente, usuario } = req.params;
  const { tipo, fecha, concepto, importe } = req.body;

  const comprobante = req.files?.comprobante || null;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  if (!usuario) {
    return res.status(400).json({ message: "El usuario es requerido" });
  }

  if (!concepto) {
    return res.status(400).json({ message: "El concepto es requerido" });
  }

  if (!importe) {
    return res.status(400).json({ message: "El importe es requerido" });
  }

  if (!tipo) {
    return res.status(400).json({ message: "El tipo es requerido" });
  }

  const importeNumber = parseFloat(importe) || 0;

  // Set the date with the current time in Mexico timezone
  const fechaConHora = moment.tz(fecha, "America/Mexico_City").toISOString();

  const obj = {
    despacho,
    expediente,
    usuario,
    tipo,
    fecha: fechaConHora,
    concepto,
    importe: importeNumber,
  };

  if (comprobante) {
    const upload = await uploadFile(
      comprobante,
      `despachos/${despacho}/expedientes/${expediente}/gastos`
    );

    obj.adjunto = {
      archivo: upload.fileName,
      nombre: upload.originalName,
    };
  }

  try {
    await ExpedienteGastosModel.create(obj);

    // const tituloMovimiento = `Se ha registrado un ${tipo}`;
    const tituloMovimiento = `Gastos`;
    const descripcionMovimiento = `Se ha registrado un ${tipo} con el concepto: ${concepto}`;

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(201).json({ message: "Gasto creado correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const get = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { tipo = "Gasto" } = req.query;
  const { page = 1 } = req.query;

  const options = {
    page,
    limit: 10,
    sort: { fecha: -1 },
  };

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  try {
    const gastos = await ExpedienteGastosModel.paginate(
      { despacho, expediente, tipo },
      options
    );

    const updatedGastos = gastos.docs.map((gasto) => {
      if (gasto.adjunto?.archivo) {
        gasto.adjunto.archivo = buildFileUri(
          `despachos/${despacho}/expedientes/${expediente}/gastos/${gasto.adjunto.archivo}`
        );
      }
      return {
        ...gasto._doc,
        fecha: formatDate(gasto.fecha),
      };
    });

    res.status(200).json({ gastos: { ...gastos, docs: updatedGastos } });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteGasto = async (req, res) => {
  const { id } = req.params;
  const { responsable } = req.query;
  try {
    const gasto = await ExpedienteGastosModel.findById(id);
    if (gasto.adjunto?.archivo) {
      await deleteFileS3(
        `despachos/${gasto.despacho}/expedientes/${gasto.expediente}/gastos/${gasto.adjunto.archivo}`
      );
    }
    const deletedGasto = await ExpedienteGastosModel.findByIdAndDelete(id);

    const tituloMovimiento = `Gastos`;
    const descripcionMovimiento = `Se ha eliminado un ${deletedGasto.tipo} con el concepto: ${deletedGasto.concepto}`;

    notificarExpedienteUsuario({
      despacho: deletedGasto.despacho,
      expediente: deletedGasto.expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario: responsable,
    });
    res.status(200).json({ message: "Gasto eliminado correctamente" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const balance = async (req, res) => {
  const { despacho, expediente } = req.params;
  try {
    const gastos = await ExpedienteGastosModel.find({
      despacho,
      expediente,
      tipo: "Gasto",
    });
    const ingresos = await ExpedienteGastosModel.find({
      despacho,
      expediente,
      tipo: "Ingreso",
    });
    const costos = await ExpedienteGastosModel.find({
      despacho,
      expediente,
      tipo: "Costo",
    });

    const gastoTotal = gastos.reduce((acc, gasto) => acc + gasto.importe, 0);
    const ingresoTotal = ingresos.reduce(
      (acc, ingreso) => acc + ingreso.importe,
      0
    );
    const costoTotal = costos.reduce((acc, costo) => acc + costo.importe, 0);

    // Calcula el saldo como ingresos menos gastos
    const saldo = costoTotal - ingresoTotal;

    // Calcula el total como ingresos menos la suma de gastos
    const total = ingresoTotal - gastoTotal;

    console.log(formatMoney(costoTotal));
    res.status(200).json({
      total: formatMoney(total), // Total (Ingresos - Gastos - Costos)
      gastos: formatMoney(gastoTotal),
      ingresos: formatMoney(ingresoTotal),
      saldo: formatMoney(saldo), // Saldo (Ingresos - Gastos)
      costo: formatMoney(costoTotal),
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  create,
  get,
  deleteGasto,
  balance,
};
