const FoliosModel = require("../models/ExpedientesFolios.js");
const MaterialModel = require("../models/MateriasDespacho.js");
const ExpedientesModel = require("../models/Expedientes.js");

const getFolios = async (req, res) => {
  const { despacho } = req.params;
  const { page = 1, search, materia } = req.query;
  const options = {
    page,
    limit: 20,
    sort: { folio: 1 },
    populate: "materia",
  };

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  const query = {
    despacho,
  };

  if (materia) {
    query.materia = materia;
  }

  if (search) {
    query.$or = [
      { clave: { $regex: search, $options: "i" } },
      { folio: { $regex: search, $options: "i" } },
    ];
  }

  try {
    const folios = await FoliosModel.paginate(query, options);

    res.status(200).json({ folios });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const createFolio = async (req, res) => {
  const { despacho } = req.params;
  const { materia, clave, folio, creadoPor } = req.body;

  try {
    // const findFolio = await FoliosModel.findOne({ clave, despacho }).populate('materia');

    // if (findFolio) {
    //   return res.status(409).json({ message: 'La clave ya existe en la materia ' + findFolio?.materia?.nombre });
    // }

    const newFolio = {
      despacho,
      materia,
      clave,
      folio,
      creadoPor,
    };

    const nuevoFolio = await FoliosModel.create(newFolio);

    res.status(201).json(nuevoFolio);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getMateriaNoAsignadas = async (req, res) => {
  const { despacho } = req.params;

  try {
    const materias = await MaterialModel.find({ estatus: "Activo", despacho });

    const folios = await FoliosModel.find({ despacho });

    const materiasSinAsignarFolio = materias.filter((materia) => {
      const materiaNoAsignada = folios.every(
        (folio) => folio.materia.toString() !== materia._id.toString()
      );
      return materiaNoAsignada;
    });

    const materiasANo = materiasSinAsignarFolio.map((materia) => {
      return {
        value: materia._id,
        label: materia.nombre,
        key: materia._id,
      };
    });

    res.status(200).json(materiasANo);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteFolio = async (req, res) => {
  const { id } = req.params;
  try {
    await FoliosModel.findByIdAndRemove(id);
    res.status(200).json({ message: "Folio eliminado" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const proximoFolio = async (req, res) => {
  const { materia } = req.params;

  try {
    const folio = await FoliosModel.findOne({ materia });

    if (!folio) {
      return res.status(404).json({ message: "No se encontró el folio" });
    }

    const folioNuevo = {
      ...folio.toObject(),
      proximo: `${folio?.clave}-${folio?.folio}`,
    };

    res.status(200).json(folioNuevo);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const proximoFolioSinMateria = async (req, res) => {
  const { despacho } = req.params;

  try {
    const expedientes = await ExpedientesModel.countDocuments({
      despacho,
      procedimiento: "No litigioso",
    });
    const proximo = "NL-" + (expedientes + 1);

    res.status(200).json({ proximo });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

module.exports = {
  getFolios,
  createFolio,
  getMateriaNoAsignadas,
  deleteFolio,
  proximoFolioSinMateria,
  proximoFolio,
};
