/* eslint-disable array-callback-return */
const ExpedienteFolder = require("../models/ExpedienteFolder");
const File = require("../models/File");
const { formatDate, convertSize } = require("../config/functionsGlobal.js");
const { uploadFile, buildFileUri, deleteFileS3 } = require("../config/s3.js");
const { notificarExpedienteUsuario } = require("./Expedientes.Controller.js");
const createFolder = async (req, res) => {
  const { name, parentFolder, despacho, expediente, responsable } = req.body;

  console.log(parentFolder, "parentFolder");

  if (!name || !despacho || !expediente) {
    return res
      .status(400)
      .json({ message: "Name, Despacho y Expediente son requeridos" });
  }

  try {
    const newFolder = new ExpedienteFolder({
      name,
      parentFolder,
      despacho,
      expediente,
    });

    if (parentFolder) {
      const parent = await ExpedienteFolder.findById(parentFolder);
      if (!parent) {
        return res.status(404).json({ message: "Carpeta padre no encontrada" });
      }
      parent.subfolders.push(newFolder._id);
      await parent.save();
    }

    await newFolder.save();

    const tituloMovimiento = "Documentos";
    const descripcionMovimiento = `Se ha creado la carpeta ${name} en el expediente`;

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario: responsable,
    });

    return res
      .status(201)
      .json({ newFolder, message: "Carpeta creada correctamente" });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ message: "Error en el servidor" });
  }
};

const createFile = async (req, res) => {
  const {
    filename,
    fileType,
    size,
    url,
    folder,
    despacho,
    expediente,
    responsable,
  } = req.body;

  if (!filename || !fileType || !size || !url || !despacho || !expediente) {
    return res.status(400).json({ message: "Todos los campos son requeridos" });
  }

  try {
    const newFile = new File({
      filename,
      fileType,
      size,
      url,
      folder,
      despacho,
      expediente,
    });

    if (folder) {
      const parentFolder = await ExpedienteFolder.findById(folder);
      if (!parentFolder) {
        return res.status(404).json({ message: "Carpeta padre no encontrada" });
      }
      parentFolder.files.push(newFile._id);
      await parentFolder.save();
    }

    const tituloMovimiento = "Documentos";
    const descripcionMovimiento = `Se ha agregado el archivo ${filename} en el expediente`;

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario: responsable,
    });

    await newFile.save();
    return res
      .status(201)
      .json({ newFile, message: "Archivo creado correctamente" });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ message: "Error en el servidor" });
  }
};
const uploadsFiles = async (req, res) => {
  const { folder, despacho, expediente, responsable } = req.body;
  let { files } = req?.files || [];

  if (!files) {
    return res.status(400).json({
      message: "Archivos son requeridos",
    });
  }

  if (!despacho || !expediente) {
    return res.status(400).json({ message: "Todos los campos son requeridos" });
  }

  // Asegúrate de que `files` sea un array
  files = Array.isArray(files) ? files : [files];

  try {
    // Subir archivos en paralelo
    const uploadPromises = files.map((file) =>
      uploadFile(
        file,
        `despachos/${despacho}/expedientes/${expediente}/documentos`
      )
    );
    const fileUploads = await Promise.all(uploadPromises);

    // Crear documentos de archivos
    const newFiles = fileUploads.map((fileUpload, index) => {
      const file = files[index];

      const tituloMovimiento = "Documentos";
      const descripcionMovimiento = `Se ha agregado el archivo ${file.name} en el expediente`;

      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario: responsable,
      });

      return {
        filename: file.name,
        fileType: file.name.split(".").pop(),
        size: file.size,
        url: fileUpload.fileName,
        folder,
        despacho,
        expediente,
      };
    });

    // Insertar todos los documentos de archivos en la base de datos en una sola operación
    const insertedFiles = await File.insertMany(newFiles);

    // Si la carpeta existe, actualiza sus archivos en una sola operación
    if (folder) {
      const parentFolder = await ExpedienteFolder.findByIdAndUpdate(
        folder,
        { $push: { files: { $each: insertedFiles.map((file) => file._id) } } },
        { new: true }
      );

      if (!parentFolder) {
        return res.status(404).json({ message: "Carpeta padre no encontrada" });
      }
    }

    return res.status(201).json({
      newFiles: insertedFiles,
      message: "Archivos subidos correctamente",
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ message: error.message });
  }
};

const getFolders = async (req, res) => {
  // Extraemos los parámetros de la consulta (query) del request
  const { despacho, expediente, parentFolder } = req.query;

  // Verificamos que los parámetros 'despacho' y 'expediente' estén presentes
  if (!despacho || !expediente) {
    return res
      .status(400)
      .json({ message: "Despacho y Expediente son requeridos" });
  }

  try {
    // Construimos la consulta base para las carpetas
    const folderQuery = { despacho, expediente };

    folderQuery.parentFolder = parentFolder || null;

    const fileQuery = { despacho, expediente, folder: parentFolder || null };

    const [files, folders] = await Promise.all([
      File.find(fileQuery).sort("name").lean(),
      ExpedienteFolder.find(folderQuery).sort("name").populate("files").lean(),
    ]);

    // Mapeamos las carpetas para calcular el tamaño total de sus archivos en megabytes
    const folderSizes = folders.map((folder) => {
      const sizeFolder = folder.files.reduce((acc, file) => acc + file.size, 0);
      return { folder, sizeFolder };
    });

    const data = [];

    // Iteramos sobre las carpetas para agregar su tamaño total en megabytes

    folders.forEach((folder) => {
      const folderSize = folderSizes.find(
        (size) => size.folder._id.toString() === folder._id.toString()
      );
      data.push({
        name: folder.name,
        fileItems: folder.files,
        lastModified:
          formatDate(folder.updatedAt) ||
          formatDate(folder.createdAt) ||
          formatDate(new Date()),
        size: folderSize ? convertSize(folderSize.sizeFolder) : 0,
        type: "folder",
        _id: folder._id,
        subfolders: folder.subfolders,
      });
    });

    // Iteramos sobre los archivos para agregar su tamaño en megabytes
    files.forEach((file) => {
      data.push({
        name: file.filename,
        lastModified:
          formatDate(file.lastModified) ||
          formatDate(file.createdAt) ||
          formatDate(new Date()),
        size: convertSize(file.size),
        type: "file",
        url: buildFileUri(
          `despachos/${despacho}/expedientes/${expediente}/documentos/${file.url}`
        ),
        fileType: file.fileType,
        _id: file._id,
      });
    });

    return res.status(200).json(data);
  } catch (error) {
    // Si ocurre un error, lo registramos en la consola y enviamos una respuesta con el estado 500 (Error en el servidor)
    console.error(error);
    return res.status(500).json({ message: "Error en el servidor" });
  }
};

const getFolderName = async (req, res) => {
  const { folderId } = req.params;

  if (!folderId) {
    return res.status(400).json({ message: "Folder ID is required" });
  }

  try {
    const folder = await ExpedienteFolder.findById(folderId).lean();

    console.log(folder);

    if (!folder) {
      return res.status(404).json({ message: "Folder not found" });
    }

    return res.status(200).json({ name: folder.name });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ message: "Server error" });
  }
};

const deleteFolders = async (req, res) => {
  const { folderIds, despacho, expediente, responsable } = req.body;

  if (!folderIds) {
    return res.status(400).json({ message: "Folder IDs are required" });
  }

  try {
    // Filtrar IDs de carpetas y archivos
    const soloIdsFolders = folderIds
      .filter((folder) => folder.type === "folder")
      .map((folder) => folder._id);
    const soloIdsFiles = folderIds
      .filter((folder) => folder.type === "file")
      .map((folder) => folder._id);

    // Buscar carpetas y archivos en paralelo
    const [folders, files] = await Promise.all([
      ExpedienteFolder.find({ _id: { $in: soloIdsFolders } }).lean(),
      File.find({ _id: { $in: soloIdsFiles } }).lean(),
    ]);

    // Eliminar archivos en paralelo
    const filesDeletes = await Promise.all(
      files.map((file) => File.findByIdAndDelete(file._id))
    );

    for (const file of filesDeletes) {
      const urlFile = `despachos/${file.despacho}/expedientes/${file.expediente}/documentos/${file.url}`;
      deleteFileS3(urlFile);

      const tituloMovimiento = "Documentos";
      const descripcionMovimiento = `Se ha eliminado el archivo ${file.filename} en el expediente`;

      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario: responsable,
      });
    }

    // Eliminar _id de los archivos de la carpeta

    await Promise.all(
      files.map(async (file) => {
        const parentFolder = await ExpedienteFolder.findById(file.folder);

        if (parentFolder) {
          parentFolder.files = parentFolder.files.filter(
            (f) => f.toString() !== file._id.toString()
          );
          await parentFolder.save();
        }
      })
    );

    // Eliminar _id de las carpetas de las carpetas que se eliminarán
    await Promise.all(
      folders.map(async (folder) => {
        const parentFolder = await ExpedienteFolder.findById(
          folder.parentFolder
        );

        if (parentFolder) {
          parentFolder.subfolders = parentFolder.subfolders.filter(
            (subfolder) => subfolder.toString() !== folder._id.toString()
          );
          await parentFolder.save();
        }
      })
    );

    // Verificar si las carpetas tienen subcarpetas o archivos
    const foldersToDelete = folders.filter(
      (folder) =>
        folder?.subfolders?.length === 0 && folder?.files?.length === 0
    );
    const foldersWithSubfolders = folders.filter(
      (folder) => folder?.subfolders?.length > 0
    );
    const foldersWithFiles = folders.filter(
      (folder) => folder?.files?.length > 0
    );

    if (foldersWithSubfolders?.length > 0) {
      return res.status(400).json({
        message: `Las siguientes carpetas tienen sub-carpetas: ${foldersWithSubfolders
          .map((folder) => folder.name)
          .join(", ")}`,
      });
    }

    if (foldersWithFiles?.length > 0) {
      return res.status(400).json({
        message: `Las siguientes carpetas tienen archivos: ${foldersWithFiles
          .map((folder) => folder.name)
          .join(", ")}`,
      });
    }

    // Eliminar carpetas en paralelo
    await Promise.all(
      foldersToDelete.map((folder) =>
        ExpedienteFolder.findByIdAndDelete(folder._id)
      )
    );

    return res.status(200).json({ message: "Carpetas y archivos eliminados" });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ message: "Server error", error });
  }
};

module.exports = {
  createFolder,
  createFile,
  getFolders,
  getFolderName,
  deleteFolders,
  uploadsFiles,
};
