const ExpedienteModel = require("../models/Expedientes.js");
const JuzgadoModel = require("../models/Juzgados.js");
const ClienteModel = require("../models/Clientes.js");
const ExpedientesUsuarioModel = require("../models/ExpedientesUsuarios.js");
const FoliosModel = require("../models/ExpedientesFolios.js");
const EtapasProcesalesModel = require("../models/EtapasProcesalesDespachos.js");
const MateriaModel = require("../models/MateriasDespacho.js");
const JuzgadosModel = require("../models/Juzgados.js");
const ExpedientesMovimientosModel = require("../models/ExpedientesMovimientos.js");
const AsuntosModel = require("../models/Asuntos.js");
const DespachoModel = require("../models/Despachos.js");
const {
  MovimientosExpedienteHTML,
} = require("../Mail/MovimientosExpedienteHTML.js");
const { sendMail } = require("../config/mail.js");
const RecursosExpedienteModel = require("../models/RecursosDespacho.js");
const ExpedienteRecursosIncidencias = require("../models/ExpedientesRecursosIncidencias.js");
const IncidenciasModel = require("../models/IncidenciasDespacho.js");
const ExpedienteAgenda = require("../models/ExpedientesAgenda.js");
const ExpedienteAgendaUsuarios = require("../models/ExpedientesAgendaUsuarios.js");
const ExpedientesPartes = require("../models/ExpedientesPartesInvolucradas.js");
const ExpedientesNotas = require("../models/ExpedientesNotas.js");
const ExpedientesGastos = require("../models/ExpedientesGastos.js");
const ExpedienteFile = require("../models/File.js");
const ExpedientesPautas = require("../models/ExpedientesPautas.js");
const moment = require("moment-timezone");
const APP_URL = process.env.DESPACHO_APP || "http://localhost:3000";
const {
  deleteFile,
  formatDate,
  formatDateTime,
} = require("../config/functionsGlobal.js");
const ExpedientesRecursosIncidencias = require("../models/ExpedientesRecursosIncidencias.js");
const path = require("path");
const { buildFileUri } = require("../config/s3.js");

const createExpediente = async (req, res) => {
  try {
    const { despacho } = req.params;
    const fecha = new Date();
    const {
      numeroExpediente = "",
      titulo,
      fechaInicio = new Date(),
      cliente,
      procedimiento,
      juzgado,
      materia,
      etapaProcesal,
      etapaOpcional,
      asunto,
      estado,
      municipio,
      objetivos,
      usuario,
    } = req.body;

    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!titulo) {
      return res.status(400).json({ message: "El título es requerido" });
    }

    if (!cliente) {
      return res.status(400).json({ message: "El cliente es requerido" });
    }

    if (!procedimiento) {
      return res.status(400).json({ message: "El procedimiento es requerido" });
    }

    // if (!juzgado) {
    //   return res.status(400).json({ message: 'El juzgado es requerido' });
    // }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!asunto) {
      return res.status(400).json({ message: "El asunto es requerido" });
    }

    let numeroExpedienteInterno = "";
    if (materia) {
      const findFolio = await FoliosModel.findOne({ despacho, materia });

      if (findFolio) {
        const { clave, folio } = findFolio;
        numeroExpedienteInterno = `${clave}-${folio}`;

        findFolio.folio = folio + 1;
        await findFolio.save();
      }
    }

    if (procedimiento === "No litigioso") {
      const expedientes = await ExpedienteModel.countDocuments({
        despacho,
        procedimiento: "No litigioso",
      });
      const proximo = "NL-" + (expedientes + 1);

      numeroExpedienteInterno = proximo;
    }
    // busca el usuario que lo va crear

    const findJuzgado = juzgado ? await JuzgadosModel.findById(juzgado) : null;

    const { _id, nombre } = findJuzgado || {};

    const juzgadoOBJ = { nombre, juzgado: _id };
    const fechaInicioMexico = moment(fechaInicio)
      .tz("America/Mexico_City")
      .format("YYYY-MM-DD");

    const objExpediente = {
      despacho,
      procedimiento,
      numeroExpediente,
      numeroExpedienteInterno,
      titulo,
      fechaInicio: fechaInicioMexico,
      cliente,
      asunto,
      estado,
      municipio,
      objetivos,
      juzgado: juzgadoOBJ,
      creadoPor: usuario,
    };

    if (procedimiento === "No litigioso") {
      objExpediente.materia = undefined;
      objExpediente.etapaProcesal = undefined;
    } else {
      if (materia) {
        // el id de la materia
        const findMateria = await MateriaModel.findById(materia);
        const { _id, nombre } = findMateria;

        objExpediente.materia = {
          nombre,
          materia: _id,
        };
      }

      if (etapaProcesal) {
        // el id de la etapa
        const findEtapaProcesal = await EtapasProcesalesModel.findById(
          etapaProcesal
        );
        const { _id, nombre } = findEtapaProcesal;

        objExpediente.etapaProcesal = {
          nombre,
          etapa: _id,
        };
      } else {
        objExpediente.etapaProcesal = {
          nombre: etapaOpcional ?? "",
          etapa: undefined,
        };
      }
    }

    const despachoObj = await DespachoModel.findById(despacho);

    if (!despachoObj) {
      return res.status(404).json({ message: "El despacho no existe" });
    }

    const { contadorExp } = despachoObj;

    const { contador, limite, vigencia } = contadorExp;

    if (vigencia < fecha) {
      return res
        .status(400)
        .json({ message: "El contador de expedientes ha caducado" });
    }

    if (contador >= limite) {
      return res
        .status(400)
        .json({ message: "El contador de expedientes ha llegado a su límite" });
    }

    despachoObj.contadorExp.contador = contador + 1;

    await despachoObj.save();

    const expediente = await ExpedienteModel.create(objExpediente);

    const expedienteUsuarioObj = {
      despacho,
      expediente,
      usuario,
      rol: "Creador",
      notificaciones: true,
      permisos: {
        "Información General": "Si",
        Agenda: "Si",
        Usuarios: "Si",
        "Partes Involucradas": "Si",
        Notas: "Si",
        "Gastos e Ingresos": "Si",
        Documentos: "Si",
        Historial: "Si",
        "Recursos e Incidentes": "Si",
        Minutas: "Si",
      },
    };

    await ExpedientesUsuarioModel.create(expedienteUsuarioObj);

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: "El expediente fue creado",
      accionRealizada: "Creación del expediente",
      usuario,
    });
    return res.status(201).json({
      message: "El expediente se creó correctamente",
      expediente,
      despacho: despachoObj,
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getExpedientesByUsuario = async (req, res) => {
  try {
    const { despacho, usuario } = req.params;
    const {
      estatus,
      page = 1,
      search,
      cliente,
      asunto,
      materia,
      etapaProcesal = "",
      juzgado = "",
    } = req.query;

    // Verificar si despacho y usuario están presentes en la solicitud
    if (!despacho || !usuario) {
      return res
        .status(400)
        .json({ message: "El despacho y el usuario son requeridos" });
    }

    const options = {
      page,
      limit: 10,
      sort: {
        estatus: 1,
        fechaMovimiento: -1,
        ultimoCambio: -1,
      },
      select:
        "titulo numeroExpediente cliente asunto juzgado estatus materia etapaProcesal", // Only fetch required fields
      populate: [
        { path: "cliente", select: "nombre" },
        { path: "asunto", select: "nombre" },
      ],
    };

    const queryFiltros = {
      despacho,
    };

    if (search) {
      const [clienteSearchResults, asuntoSearchResults] = await Promise.all([
        ClienteModel.find({
          nombre: { $regex: search.trim(), $options: "i" },
          despacho,
        }).select("_id"),
        AsuntosModel.find({
          nombre: { $regex: search.trim(), $options: "i" },
          despacho,
        }).select("_id"),
      ]);

      const clienteIds = clienteSearchResults.map((cliente) => cliente._id);

      const asuntoIds = asuntoSearchResults.map((asunto) => asunto._id);

      queryFiltros.$or = [
        { titulo: { $regex: search.trim(), $options: "i" } },
        { numeroExpediente: { $regex: search.trim(), $options: "i" } },
        // { numeroExpedienteInterno: { $regex: search.trim(), $options: 'i' } },
        { cliente: { $in: clienteIds } },
        { asunto: { $in: asuntoIds } },
        { "juzgado.nombre": { $regex: search.trim(), $options: "i" } },
        { "materia.nombre": { $regex: search.trim(), $options: "i" } },
        { "etapaProcesal.nombre": { $regex: search, $options: "i" } },
      ];
    }

    if (cliente) {
      queryFiltros.cliente = cliente;
    }

    if (estatus) {
      queryFiltros.estatus = estatus;
    }

    if (asunto) {
      queryFiltros.asunto = asunto;
    }

    if (materia) {
      queryFiltros["materia.materia"] = materia;
    }

    if (etapaProcesal) {
      queryFiltros["etapaProcesal.etapa"] = etapaProcesal;
    }

    if (juzgado) {
      queryFiltros["juzgado.juzgado"] = juzgado;
    }

    const findDespacho = await DespachoModel.findById(despacho);

    if (findDespacho) {
      const { creadoPor } = findDespacho;

      if (creadoPor.toString() === usuario.toString()) {
        const expedientes = await ExpedienteModel.paginate(
          queryFiltros,
          options
        );

        return res.status(200).json(expedientes);
      }
    }

    const expedientesUsuarios = await ExpedientesUsuarioModel.find({
      despacho,
      usuario,
    }).select("expediente");

    const expedienteIds = expedientesUsuarios.map((exp) => exp.expediente);

    const combinedQuery = { ...queryFiltros, _id: { $in: expedienteIds } };

    const expedientes = await ExpedienteModel.paginate(combinedQuery, options);

    res.status(200).json(expedientes);
  } catch (error) {
    console.log(error.message);
    res.status(409).json({ message: error.message });
  }
};

const getExpedientesByUsuarioSinPaginate = async (req, res) => {
  try {
    const construyeRespuesta = (expedientes) => {
      const result = expedientes.map((expediente) => {
        const { _id, titulo, numeroExpedienteInterno, numeroExpediente } =
          expediente;

        // Define partes del label
        const parteInterno = numeroExpedienteInterno || "";
        const parteExpediente = numeroExpediente || "";

        // Construye el label
        const label = `${titulo}${parteInterno ? ` - ${parteInterno}` : ""}${
          parteExpediente ? ` - ${parteExpediente}` : ""
        }`;

        return {
          key: _id,
          label,
          titulo,
          numeroExpedienteInterno,
          numeroExpediente,
        };
      });

      return result;
    };

    const { despacho, usuario } = req.params;

    // Verificar si despacho y usuario están presentes en la solicitud
    if (!despacho || !usuario) {
      return res
        .status(400)
        .json({ message: "El despacho y el usuario son requeridos" });
    }

    const queryFiltros = {
      despacho,
      estatus: "Activo", // Filtro para estatus "Activo"
    };

    const findDespacho = await DespachoModel.findById(despacho);

    if (findDespacho) {
      const { creadoPor } = findDespacho;

      if (creadoPor.toString() === usuario.toString()) {
        // Consulta los expedientes directamente sin paginación
        const expedientes = await ExpedienteModel.find(queryFiltros).select(
          "titulo numeroExpedienteInterno numeroExpediente"
        );

        return res.status(200).json(construyeRespuesta(expedientes));
      }
    }

    const expedientesUsuarios = await ExpedientesUsuarioModel.find({
      despacho,
      usuario,
    }).select("expediente");

    const expedienteIds = expedientesUsuarios.map((exp) => exp.expediente);

    const combinedQuery = { ...queryFiltros, _id: { $in: expedienteIds } };

    const expedientes = await ExpedienteModel.find(combinedQuery).select(
      "titulo numeroExpedienteInterno numeroExpediente"
    );

    return res.status(200).json(construyeRespuesta(expedientes));
  } catch (error) {
    console.log(error.message);
    res.status(409).json({ message: error.message });
  }
};

const getExpedienteById = async (req, res) => {
  try {
    const { despacho, usuario, expediente } = req.params;

    if (!expediente || !despacho || !usuario) {
      return res.status(400).json({
        message: `Faltan parámetros: ${
          !expediente ? "Expediente" : !despacho ? "Despacho" : "Usuario"
        }`,
      });
    }

    // Estructura para el populate
    const populate = [
      { path: "cliente", select: "nombre correo telefono" },
      { path: "asunto", select: "nombre" },
      { path: "juzgado.juzgado", select: "nombre" },
      { path: "materia.materia", select: "nombre" },
      { path: "etapaProcesal.etapa", select: "nombre _id" },
      { path: "creadoPor", select: "nombre apellidoPaterno apellidoMaterno" },
    ];

    // Consultar despacho y expediente de usuario en paralelo
    const [findDespacho, expedienteData] = await Promise.all([
      DespachoModel.findById(despacho).select("creadoPor"),
      ExpedienteModel.findById(expediente).populate(populate),
    ]);

    if (!findDespacho) {
      return res.status(404).json({ message: "El despacho no existe" });
    }

    if (!expedienteData) {
      return res.status(404).json({
        message: "El expediente no existe o no tienes permisos para verlo",
      });
    }

    // Verificar si el usuario es el creador del despacho
    const esCreador = findDespacho.creadoPor.toString() === usuario.toString();

    return res.status(200).json({
      expediente: expedienteData,
      permisos: esCreador
        ? { rol: "Creador", notificaciones: false }
        : undefined,
    });
  } catch (error) {
    return res.status(409).json({ message: error.message });
  }
};

const getPermisosExpediente = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;
  try {
    const findDespacho = await DespachoModel.findById(despacho).select(
      "creadoPor"
    );

    if (findDespacho) {
      const { creadoPor } = findDespacho;
      if (creadoPor.toString() === usuario.toString()) {
        const permisos = {
          rol: "Creador",
          notificaciones: true,
          permisos: {
            "Información General": "Si",
            Agenda: "Si",
            Usuarios: "Si",
            "Partes Involucradas": "Si",
            Notas: "Si",
            "Gastos e Ingresos": "Si",
            Documentos: "Si",
            Historial: "Si",
            "Recursos e Incidentes": "Si",
            Minutas: "Si",
          },
        };

        return res.status(200).json(permisos);
      }
    }

    const findPermisos = await ExpedientesUsuarioModel.findOne({
      despacho,
      usuario,
      expediente,
    });

    if (!findPermisos) {
      return res
        .status(404)
        .json({ message: "No tienes permisos para ver este expediente" });
    }

    return res.status(200).json(findPermisos);
  } catch (error) {
    return res.status(409).json({ message: error.message });
  }
};

const updateEstatus = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { estatus, descripcion } = req.body;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!usuario) {
    return res.status(400).json({ message: "El usuario es requerido" });
  }

  if (!expediente) {
    return res.status(400).json({ message: "El expediente es requerido" });
  }

  try {
    const tipos = ["Activo", "Inactivo", "Concluido", "Suspendido"];

    if (!tipos.includes(estatus)) {
      return res.status(400).json({ message: "El estatus no es válido" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const { estatus: estatusAnterior } = findExpediente;

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `El estatus del expediente cambió de ${estatusAnterior} a ${estatus}`;

    findExpediente.estatus = estatus;
    findExpediente.ultimoMovimiento = new Date();

    if (
      estatus === "Concluido" ||
      estatus === "Inactivo" ||
      estatus === "Suspendido"
    ) {
      findExpediente.fechaTermino = new Date();
    }

    if (estatus === "Activo") {
      findExpediente.fechaTermino = undefined;
    }

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(200).json({
      message: "El estatus del expediente se actualizó correctamente",
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateTitulo = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { titulo, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!titulo) {
      return res.status(400).json({ message: "El título es requerido" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `El título del expediente cambió de ${findExpediente.titulo} a ${titulo}`;

    findExpediente.titulo = titulo;
    findExpediente.ultimoMovimiento = new Date();

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res
      .status(200)
      .json({ message: "El título del expediente se actualizó correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateFechaInicio = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { fechaInicio, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!fechaInicio) {
      return res
        .status(400)
        .json({ message: "La fecha de inicio es requerida" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `La fecha de inicio del expediente cambió de ${
        findExpediente.fechaInicio.toISOString().split("T")[0]
      } a ${fechaInicio.split("T")[0]}`;

    const movimiento = {
      despacho,
      expediente,
      creadoPor: usuario,
      titulo: tituloMovimiento,
      fecha: new Date(),
      descripcion: descripcionMovimiento,
    };

    await ExpedientesMovimientosModel.create(movimiento);

    findExpediente.fechaInicio = fechaInicio;
    findExpediente.ultimoMovimiento = new Date();
    // findExpediente.ultimoCambio = new Date();

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(200).json({
      message: "La fecha de inicio del expediente se actualizó correctamente",
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateFechaFinal = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { fechaFinal, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!fechaFinal) {
      return res.status(400).json({ message: "La fecha final es requerida" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `La fecha de inicio del expediente cambió de ${
        findExpediente.fechaInicio.toISOString().split("T")[0]
      } a ${fechaFinal.split("T")[0]}`;

    findExpediente.fechaTermino = fechaFinal;
    findExpediente.ultimoMovimiento = new Date();
    // findExpediente.ultimoCambio = new Date();

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });
    res.status(200).json({
      message: "La fecha final del expediente se actualizó correctamente",
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateNumeroExpediente = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { numeroExpediente, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!numeroExpediente) {
      return res
        .status(400)
        .json({ message: "El número de expediente es requerido" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `El número de expediente cambió de ${findExpediente.numeroExpediente} a ${numeroExpediente}`;

    findExpediente.numeroExpediente = numeroExpediente;
    findExpediente.ultimoMovimiento = new Date();

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res
      .status(200)
      .json({ message: "El número de expediente se actualizó correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateJuzgado = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { juzgado, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!juzgado) {
      return res.status(400).json({ message: "El juzgado es requerido" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const findJuzgado = await JuzgadoModel.findById(juzgado);

    if (!findJuzgado) {
      return res.status(404).json({ message: "El juzgado no existe" });
    }

    const tituloMovimiento = "Información general";
    const descripcionMovimiento =
      descripcion ||
      `El juzgado del expediente cambió de ${findExpediente.juzgado.nombre} a ${findJuzgado.nombre}`;

    findExpediente.juzgado = {
      nombre: findJuzgado.nombre,
      juzgado,
    };
    findExpediente.ultimoMovimiento = new Date();

    await findExpediente.save();

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: tituloMovimiento,
      usuario,
    });

    res.status(200).json({
      message: "El juzgado del expediente se actualizó correctamente",
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};
// asunto
const updateJuicio = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const { nombreJuicio, descripcion } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!nombreJuicio) {
      return res
        .status(400)
        .json({ message: "El nombre del trabajos o servicios es requerido" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query).populate(
      "asunto"
    );

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    const findAsunto = await AsuntosModel.findOne({
      _id: findExpediente.asunto,
      despacho,
    });

    if (!findAsunto) {
      return res.status(404).json({ message: "El asunto no existe" });
    }

    findAsunto.nombre = nombreJuicio;
    findAsunto.save();

    const titiloMovimiento = "Cambio de trabajo o servicios";
    const descripcionMovimiento =
      descripcion ||
      `El trabajo o servicio del expediente cambió de ${findExpediente.asunto.nombre} a ${nombreJuicio}`;

    notificarExpedienteUsuario({
      despacho,
      expediente,
      descripcion: descripcionMovimiento,
      accionRealizada: titiloMovimiento,
      usuario,
    });

    res.status(200).json({
      message:
        "El trabajo o servicio del expediente se actualizó correctamente",
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateEtapaProcesal = async (req, res) => {
  const { despacho, usuario, expediente } = req.params;

  const {
    etapaProcesal,
    descripcion = "",
    recurso = null,
    incidencia = null,
    descripcionRC = "",
    materia,
    isRecursos,
    isIncidentes,
    fechaInicio,
  } = req.body;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    if (!usuario) {
      return res.status(400).json({ message: "El usuario es requerido" });
    }

    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    if (!materia) {
      return res.status(400).json({ message: "La materia es requerida" });
    }

    const query = {
      despacho,
      _id: expediente,
    };

    const findExpediente = await ExpedienteModel.findOne(query);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    if (etapaProcesal !== findExpediente?.etapaProcesal?.etapa?.toString()) {
      const nuevaEtapa = await EtapasProcesalesModel.findOne({
        _id: etapaProcesal,
        despacho,
      });

      if (!nuevaEtapa) {
        return res.status(404).json({ message: "La etapa  no existe" });
      }

      // movimientos de incidencias
      const movimeintoDescripcion =
        descripcion ||
        `La etapa del expediente cambió de ${findExpediente.etapaProcesal.nombre} a ${nuevaEtapa?.nombre}`;
      const accionRealizada = "Cambio de etapa";

      findExpediente.etapaProcesal = {
        nombre: nuevaEtapa.nombre,
        etapa: etapaProcesal,
      };

      findExpediente.ultimoMovimiento = new Date();

      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: movimeintoDescripcion,
        accionRealizada,
        usuario,
      });
    }

    if (materia !== findExpediente?.materia?.materia?.toString()) {
      const nuevaMateria = await MateriaModel.findOne({
        _id: materia,
        despacho,
      });

      if (!nuevaMateria) {
        return res.status(404).json({ message: "La materia no existe" });
      }

      const movimeintoDescripcion =
        descripcion ||
        `La materia del expediente cambió de ${findExpediente.materia.nombre} a ${nuevaMateria.nombre}`;
      const accionRealizada = "Cambio de materia";

      findExpediente.materia = {
        nombre: nuevaMateria.nombre,
        materia,
      };

      findExpediente.ultimoMovimiento = new Date();

      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: movimeintoDescripcion,
        accionRealizada,
        usuario,
      });
    }

    findExpediente.ultimoMovimiento = new Date();
    await findExpediente.save();

    if (recurso && isRecursos) {
      const recursosExpediente = await RecursosExpedienteModel.findOne({
        despacho,
        _id: recurso,
      });

      if (recursosExpediente) {
        const newRecurso = {
          despacho,
          expediente,
          tipo: "Recurso",
          recurso,
          incidencia: null,
          creadoPor: usuario,
          fecha: fechaInicio || new Date(),
          comentario: descripcionRC,
        };

        ExpedientesRecursosIncidencias.create(newRecurso);

        notificarExpedienteUsuario({
          expediente,
          despacho,
          descripcion: "Se agregó un recurso al expediente",
          accionRealizada: "Recurso agregado",
          usuario,
        });
      }
    }

    if (incidencia && isIncidentes) {
      const findIncidencia = await IncidenciasModel.findOne({
        despacho,
        _id: incidencia,
      });

      if (findIncidencia) {
        const newIncidencia = {
          despacho,
          expediente,
          tipo: "Incidencia",
          recurso: null,
          incidencia,
          creadoPor: usuario,
          fecha: fechaInicio || new Date(),
          comentario: descripcionRC,
        };

        ExpedientesRecursosIncidencias.create(newIncidencia);

        notificarExpedienteUsuario({
          despacho,
          expediente,
          descripcion: "Se agregó una incidencia al expediente",
          accionRealizada: "Incidencia agregada",
          usuario,
        });
      }
    }

    res
      .status(200)
      .json({ message: "Se actualizó la etapa del expediente correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteExpediente = async (req, res) => {
  const { despacho, expediente } = req.params;

  try {
    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    const findExpediente = await ExpedienteModel.findById(expediente);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    // const expedientesEliminarArchivo = async ({ expediente, despacho }) => {
    //   const eliminarExpedientesAdjuntos = await ExpedientesAdjuntos.find({ expediente, despacho }).select('archivo');

    //   eliminarExpedientesAdjuntos.forEach(async (adjunto) => {
    //     const { archivo } = adjunto;
    //     const folderPath = path.join('src/uploads/documentos', archivo);
    //     deleteFile(folderPath);
    //   });

    //   return await ExpedientesAdjuntos.deleteMany({ expediente, despacho });
    // };

    const expedientesEliminarGastos = async ({ expediente, despacho }) => {
      const eliminarExpedientesGastos = await ExpedientesGastos.find({
        expediente,
        despacho,
      }).select("adjunto");

      eliminarExpedientesGastos.forEach(async (gasto) => {
        const archivo = gasto.adjunto.archivo;

        const folderPath = path.join("src/uploads/expedientes-gastos", archivo);
        deleteFile(folderPath);
      });

      return await ExpedientesGastos.deleteMany({ expediente, despacho });
    };

    const EliminarTodo = Promise.all([
      // expediente asociados a los usuarios
      await ExpedientesUsuarioModel.deleteMany({ expediente, despacho }),

      // expediente asociados a los movimientos
      await ExpedientesMovimientosModel.deleteMany({ expediente, despacho }),

      // expediente asociados a los recursos o incidencias
      await RecursosExpedienteModel.deleteMany({ expediente, despacho }),

      // expediente asociados a agenda

      await ExpedienteAgenda.deleteMany({ expediente, despacho }),

      // expediente asociados a agenda usuarios
      await ExpedienteAgendaUsuarios.deleteMany({ expediente, despacho }),

      // expediente asociados a partes involucradas
      await ExpedientesPartes.deleteMany({ expediente, despacho }),

      // expediente asociados a notas
      await ExpedientesNotas.deleteMany({ expediente, despacho }),

      // expediente asociados a gastos
      await expedientesEliminarGastos({ expediente, despacho }),

      // expediente asociados a adjuntos y tambien eliminar los archivos
      // await expedientesEliminarArchivo({ expediente, despacho }),

      // expediente asociados a pautas
      await ExpedientesPautas.deleteMany({ expediente, despacho }),
    ]);

    await ExpedienteModel.findByIdAndDelete(expediente);

    const findDespacho = await DespachoModel.findById(despacho);

    if (findDespacho) {
      const { contadorExp } = findDespacho;
      const { contador } = contadorExp;
      findDespacho.contadorExp.contador = contador - 1;
      await findDespacho.save();
    }

    res.status(200).json({
      message: "El expediente se eliminó correctamente",
      expediente: findExpediente,
      EliminarTodo,
    });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getExpedienteByCliente = async (req, res) => {
  const { cliente } = req.params;
  const { search, estatus, page = "", materia } = req.query;

  try {
    if (!cliente) {
      return res.status(400).json({ message: "El cliente es requerido" });
    }

    const query = {
      cliente,
    };

    if (search) {
      query.$or = [
        { titulo: { $regex: search, $options: "i" } },
        { numeroExpediente: { $regex: search, $options: "i" } },
        { numeroExpedienteInterno: { $regex: search, $options: "i" } },
      ];
    }

    if (estatus) {
      query.estatus = estatus;
    }

    if (materia) {
      query["materia.materia"] = materia;
    }

    const options = {
      page,
      limit: 10,
      sort: {
        estatus: 1,
        fechaMovimiento: -1,
        ultimoCambio: -1,
      },
      populate: [
        {
          path: "cliente",
          select: "nombre",
        },
        {
          path: "asunto",
          select: "nombre",
        },
      ],
    };

    const expedientes = await ExpedienteModel.paginate(query, options);

    res.status(200).json(expedientes);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getExpedienteByID = async (req, res) => {
  const { expediente } = req.params;

  try {
    if (!expediente) {
      return res.status(400).json({ message: "El expediente es requerido" });
    }

    const findExpediente = await ExpedienteModel.findById(expediente).populate([
      { path: "cliente", select: "nombre telefono correo" },
      { path: "asunto", select: "nombre" },
      { path: "juzgado.juzgado", select: "nombre" },
      { path: "materia.materia", select: "nombre" },
      { path: "etapaProcesal.etapa", select: "nombre _id" },
      { path: "creadoPor", select: "nombre apellidoPaterno apellidoMaterno" },
    ]);

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    res.status(200).json(findExpediente);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getJuzgadosByExpedientes = async (req, res) => {
  const { despacho } = req.params;

  try {
    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    const findExpedientes = await ExpedienteModel.find({ despacho }).select(
      "juzgado"
    );

    const juzgados = [];

    findExpedientes.forEach((expediente) => {
      const { juzgado } = expediente;

      if (juzgado) {
        const { juzgado: juzgadoID } = juzgado;

        if (!juzgados.includes(juzgadoID)) {
          juzgados.push(juzgadoID);
        }
      }
    });

    const findJuzgados = await JuzgadoModel.find({ _id: { $in: juzgados } });

    const juzgadosData = findJuzgados.map((juzgado) => ({
      value: juzgado._id,
      label: juzgado.nombre,
    }));

    res.status(200).json(juzgadosData);
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const notificarExpedienteUsuario = async ({
  expediente,
  despacho,
  descripcion,
  accionRealizada,
  usuario,
}) => {
  try {
    const findUsuarios = await ExpedientesUsuarioModel.find({
      despacho,
      expediente,
      notificaciones: true,
    }).populate(
      "usuario",
      "email nombre apellidoPaterno apellidoMaterno telefono"
    );

    const movimiento = {
      despacho,
      expediente,
      creadoPor: usuario,
      titulo: accionRealizada,
      fecha: moment().format(),
      descripcion,
    };

    ExpedientesMovimientosModel.create(movimiento);
    // Conjunto para almacenar los números de teléfono a los que ya se les ha enviado un mensaje
    // const telefonosNotificados = new Set();

    findUsuarios.forEach(async (expedienteF) => {
      const { email, nombre, apellidoPaterno, apellidoMaterno } =
        expedienteF.usuario;

      const html = MovimientosExpedienteHTML({
        nombreCliente: `${nombre} ${apellidoPaterno} ${apellidoMaterno}`,
        nombreExpediente: await ExpedienteModel.findById(expediente)
          .select("titulo numeroExpediente")
          .then((expediente) => expediente.titulo),
        nombreRemitente: `${nombre} ${apellidoPaterno} ${
          apellidoMaterno || ""
        }`,
        accionRealizada,
        detallesAdicionales: descripcion,
        fechaMovimiento: new Date().toLocaleString(),
        enlaceExpediente: `${APP_URL}/expedientes/${expediente}/editar`,
      });

      sendMail(html, "Movimiento de expediente", email);

      // if (telefono && !telefonosNotificados.has(telefono)) {
      //   sendWhatsappMovimiento({
      //     to: telefono,
      //     var1: `Hola ${nombre} ${apellidoPaterno} ${apellidoMaterno}`,
      //     var2: accionRealizada,
      //     var3: descripcion,
      //     url: `/expedientes/${expediente}/editar`
      //   });
      //   // Agregar el número de teléfono al conjunto
      //   telefonosNotificados.add(telefono);
      // }
    });
  } catch (error) {
    console.log(error);
  }
};

const createExpedienteFile = async (req, res) => {
  const { despacho, expediente } = req.params;
  const { modulos } = req.query;
  const listModulos = null;

  try {
    const listModulos = modulos?.split(",");
    const finders = {
      "Información General": () =>
        ExpedienteModel.findOne({ _id: expediente, despacho }).populate(
          "cliente asunto materia etapaProcesal creadoPor"
        ),
      Agenda: () =>
        ExpedienteAgenda.find({ expediente, despacho })
          .populate("creadoPor")
          .sort({ fecha: -1 }),
      Usuarios: () =>
        ExpedientesUsuarioModel.find({ expediente, despacho }).populate(
          "usuario"
        ),
      "Partes Involucradas": () =>
        ExpedientesPartes.find({ expediente, despacho }).sort({ fecha: -1 }),
      Notas: () =>
        ExpedientesNotas.find({ expediente, despacho })
          .populate("creadoPor")
          .sort({ fecha: -1 }),
      Gastos: () =>
        ExpedientesGastos.find({ expediente, despacho, tipo: "Gasto" }).sort({
          fecha: -1,
        }),
      Ingresos: () =>
        ExpedientesGastos.find({ expediente, despacho, tipo: "Ingreso" }).sort({
          fecha: -1,
        }),
      Documentos: () =>
        ExpedienteFile.find({ expediente, despacho }).sort({ createdAt: -1 }),
      Historial: () =>
        ExpedientesMovimientosModel.find({ expediente, despacho })
          .populate("creadoPor")
          .sort({ fecha: -1 }),
      Recursos: () =>
        ExpedienteRecursosIncidencias.find({
          expediente,
          despacho,
          tipo: "Recurso",
        })
          .populate("creadoPor recurso")
          .sort({ fecha: -1 }),
      Incidentes: () =>
        ExpedienteRecursosIncidencias.find({
          expediente,
          despacho,
          tipo: "Incidencia",
        })
          .populate("creadoPor incidencia")
          .sort({ fecha: -1 }),
      // Pautas: () => []
    };

    if (!listModulos) {
      return res.status(400).json({ message: "Los módulos son requeridos" });
    }
    const results = await Promise.all(
      Object.keys(finders).map((modulo) =>
        listModulos.includes(modulo) ? finders[modulo]() : Promise.resolve([])
      )
    );
    const findDespacho = await DespachoModel.findById(despacho);

    if (findDespacho?.logo) {
      findDespacho.logo = buildFileUri(
        `despachos/${despacho}/logo/${findDespacho.logo}`
      );
    }

    const [
      findExpediente = [],
      findAgenda = [],
      findUsuarios = [],
      findPartesInvolucradas = [],
      findNotas = [],
      findGastos = [],
      findIngresos = [],
      findDocumentos = [],
      findMovimientos = [],
      findRecursos = [],
      findIncidencias = [],
      // findPautass
    ] = results;

    if (!findExpediente) {
      return res.status(404).json({ message: "El expediente no existe" });
    }

    return res.status(200).json({
      logo: findDespacho?.logo || "",
      nombreDespacho: findDespacho?.nombre || "",
      numeroExpediente: findExpediente?.numeroExpediente,
      titulo: findExpediente?.titulo,
      juicio: findExpediente?.asunto?.nombre,
      folioInterno: findExpediente?.numeroExpedienteInterno || "",
      fechaInicio: findExpediente?.fechaInicio,
      ultimoMovimiento: formatDate(findExpediente.ultimoMovimiento),
      estatus: findExpediente?.estatus,
      cliente: findExpediente?.cliente?.nombre,
      procedimiento: findExpediente?.procedimiento,
      juzgado: findExpediente?.juzgado?.nombre,
      materia: findExpediente?.materia?.nombre,
      etapaProcesal: findExpediente?.etapaProcesal?.nombre,
      responsable: `${findExpediente?.creadoPor?.nombre || ""} ${
        findExpediente?.creadoPor?.apellidoPaterno || ""
      } ${findExpediente?.creadoPor?.apellidoMaterno || ""}`,
      movimientos: findMovimientos?.map((movimiento) => ({
        fecha: formatDate(movimiento?.fecha),
        titulo: movimiento?.titulo,
        descripcion: movimiento?.descripcion,
        responsable: `${movimiento?.creadoPor?.nombre || ""} ${
          movimiento?.creadoPor?.apellidoPaterno || ""
        } ${movimiento?.creadoPor?.apellidoMaterno || ""}`,
      })),
      partesInvolucradas: findPartesInvolucradas
        ? findPartesInvolucradas?.map((parte) => ({
            rol: parte?.tipo || "",
            nombre: parte?.nombre || "",
          }))
        : [],
      notas: findNotas?.map((nota) => ({
        fecha: formatDate(nota?.fecha),
        descripcion: nota?.comentario,
        responsable: `${nota?.creadoPor?.nombre || ""} ${
          nota?.creadoPor?.apellidoPaterno || ""
        } ${nota?.creadoPor?.apellidoMaterno || ""}`,
      })),
      gastos: findGastos?.map((gasto) => ({
        fecha: formatDate(gasto?.fecha),
        concepto: gasto?.concepto,
        monto: gasto?.importe,
      })),
      ingresos: findIngresos?.map((ingreso) => ({
        fecha: formatDate(ingreso?.fecha),
        concepto: ingreso?.concepto,
        monto: ingreso?.importe,
      })),
      documentosAdjuntos: findDocumentos?.map((documento) => ({
        nombre: documento?.filename,
        url: buildFileUri(
          `despachos/${despacho}/expedientes/${expediente}/documentos/${documento?.url}`
        ),
      })),
      recursos: findRecursos?.map((recurso) => ({
        fecha: formatDate(recurso?.fecha),
        descripcion: recurso?.comentario,
        nombre: recurso?.recurso?.nombre || "",
        responsable: `${recurso?.creadoPor?.nombre || ""} ${
          recurso?.creadoPor?.apellidoPaterno || ""
        } ${recurso?.creadoPor?.apellidoMaterno || ""}`,
      })),
      incidencias: findIncidencias?.map((incidencia) => ({
        fecha: formatDate(incidencia?.fecha),
        descripcion: incidencia?.comentario,
        nombre: incidencia?.incidencia?.nombre || "",
        responsable: `${incidencia?.creadoPor?.nombre || ""} ${
          incidencia?.creadoPor?.apellidoPaterno || ""
        } ${incidencia?.creadoPor?.apellidoMaterno || ""}`,
      })),
      agenda: findAgenda?.map((agenda) => ({
        fecha: formatDateTime(agenda.fecha),
        descripcion: agenda?.descripcion,
        titulo: agenda?.title,
        responsable: `${agenda?.creadoPor?.nombre || ""} ${
          agenda?.creadoPor?.apellidoPaterno || ""
        } ${agenda?.creadoPor?.apellidoMaterno || ""}`,
      })),
      usuarios: findUsuarios?.map((usuario) => ({
        nombre: `${usuario?.usuario?.nombre} ${
          usuario?.usuario?.apellidoPaterno || ""
        } ${usuario?.usuario?.apellidoMaterno || ""}`,
        rol: usuario?.rol,
      })),
    });
  } catch (error) {
    console.log(error);
    return res
      .status(409)
      .json({ message: error.message, moudlos: listModulos });
  }
};

module.exports = {
  createExpediente,
  getExpedientesByUsuario,
  getExpedienteById,
  updateEstatus,
  updateTitulo,
  updateNumeroExpediente,
  updateJuicio,
  updateEtapaProcesal,
  deleteExpediente,
  getExpedienteByCliente,
  getExpedienteByID,
  getJuzgadosByExpedientes,
  updateFechaInicio,
  updateFechaFinal,
  updateJuzgado,
  createExpedienteFile,
  getExpedientesByUsuarioSinPaginate,
  notificarExpedienteUsuario,
  getPermisosExpediente,
};
