const ExpedientesAgenda = require("../models/ExpedientesAgenda.js");
const UsuariosAgenda = require("../models/ExpedientesAgendaUsuarios.js");
const Expedientes = require("../models/Expedientes.js");
const moment = require("moment-timezone");
const { notificarExpedienteUsuario } = require("./Expedientes.Controller.js");
const UsuarioModel = require("../models/Usuarios.js");
const clientesModel = require("../models/Clientes.js");
const { sendWhatsappEvent } = require("../config/functionsGlobal.js");
const mongoose = require("mongoose");
const prospectosModel = require("../models/ProspectosDespacho.js");

const getEvents = async (req, res) => {
  const { despacho, usuario } = req.params;
  const { expediente = null } = req.query;

  try {
    if (!usuario) {
      return res.status(400).json({ message: "Usuario no especificado" });
    }

    if (!despacho) {
      return res.status(400).json({ message: "Despacho no especificado" });
    }

    const query = {
      despacho: mongoose.Types.ObjectId(despacho),
      usuario: mongoose.Types.ObjectId(usuario),
    };

    if (expediente) {
      query.expediente = expediente;
    }

    const events = await UsuariosAgenda.find(query).populate("agenda").exec();
    // ordenar por fecha ASC
    const orderedEvents = events.sort((a, b) => {
      const fechaHoraInicioA = moment(a.agenda.fecha);
      const fechaHoraInicioB = moment(b.agenda.fecha);
      const horaInicioA = a.agenda.horaInicio;
      const horaInicioB = b.agenda.horaInicio;

      if (fechaHoraInicioA.isBefore(fechaHoraInicioB)) {
        return -1;
      } else if (fechaHoraInicioA.isAfter(fechaHoraInicioB)) {
        return 1;
      } else {
        if (horaInicioA < horaInicioB) {
          return -1;
        } else if (horaInicioA > horaInicioB) {
          return 1;
        } else {
          return 0;
        }
      }
    });

    const formattedEvents = formattedEventsFun(orderedEvents);

    // const eventsWithUsers = await Promise.all(
    //   formattedEvents.map(async (event) => {
    //     const usuariosEvento = await UsuariosAgenda.find({ agenda: event.id }).select('usuario').exec().then((usuarios) => usuarios.map((usuario) => usuario.usuario));
    //     return { ...event, usuariosEvento };
    //   })
    // );

    // const eventos = eventsWithUsers;

    return res.status(200).json({ eventos: formattedEvents });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const createEvent = async (req, res) => {
  const { despacho, usuario } = req.params;
  const {
    expediente = null,
    titulo,
    descripcion = "",
    color,
    fechaInicio,
    horaInicio = null,
    fechaFinal = null,
    horaFinal = null,
    recordar = null,
    usuariosSeleccionados = [],
    clientesSeleccionados = [],
    invitadosSeleccionados = [],
  } = req.body;

  if (!usuario) {
    return res.status(400).json({ message: "Usuario no especificado" });
  }

  if (!despacho) {
    return res.status(400).json({ message: "Despacho no especificado" });
  }

  if (!titulo) {
    return res.status(400).json({ message: "Título no especificado" });
  }

  if (!fechaInicio) {
    return res.status(400).json({ message: "Fecha de inicio no especificada" });
  }

  if (!horaInicio) {
    return res.status(400).json({ message: "Hora de inicio no especificada" });
  }

  if (recordar && !horaInicio) {
    return res.status(400).json({ message: "Hora de inicio no especificada" });
  }

  try {
    const fecha = moment.utc(fechaInicio).toDate();

    if (horaInicio) {
      fecha.setHours(moment(horaInicio).hours());
      fecha.setMinutes(moment(horaInicio).minutes());
    }

    const fechaFin = fechaFinal ? moment.utc(fechaFinal).toDate() : null;

    if (horaFinal && fechaFin) {
      fechaFin.setHours(moment(horaFinal).hours());
      fechaFin.setMinutes(moment(horaFinal).minutes());
    }

    const obj = {
      despacho,
      title: titulo,
      descripcion,
      fecha,
      fechaFin,
      creadoPor: usuario || null,
      fechaCreacion: moment().toDate(),
    };

    if (expediente) {
      obj.expediente = expediente;
    }

    const event = await ExpedientesAgenda.create(obj);

    let fechaRecordatorioMx = null;

    if (recordar !== null) {
      fechaRecordatorioMx = moment(fecha)
        .subtract(parseInt(recordar), "minutes")
        .toDate();
    }

    if (usuariosSeleccionados.length > 0) {
      for (const _id of usuariosSeleccionados) {
        await UsuariosAgenda.create({
          despacho,
          expediente,
          usuario: _id,
          fechaRecordatorio: fechaRecordatorioMx,
          agenda: event._id,
          recordar,
          color,
        });
      }
    }

    if (clientesSeleccionados.length > 0) {
      for (const _id of clientesSeleccionados) {
        await UsuariosAgenda.create({
          despacho,
          expediente,
          cliente: _id,
          fechaRecordatorio: fechaRecordatorioMx,
          agenda: event._id,
          recordar,
          color,
        });
      }
    }

    if (invitadosSeleccionados.length > 0) {
      for (const _id of invitadosSeleccionados) {
        await UsuariosAgenda.create({
          despacho,
          expediente,
          prospecto: _id,
          fechaRecordatorio: fechaRecordatorioMx,
          agenda: event._id,
          recordar,
          color,
        });
      }
    }

    if (expediente) {
      const tituloMovimiento = "Agenda";
      const descripcionMovimiento = `Se ha creado un nuevo evento con el título: ${titulo}. Fecha: ${moment(
        fecha
      ).format("DD/MM/YYYY")}. Hora: ${moment(fecha).format("hh:mm A")}.`;
      notificarExpedienteUsuario({
        despacho,
        expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario,
      });
    }

    const numerosTelefonicos = [];

    // agrega de usuarios usuariosSeleccionados
    const findUsuarios = await UsuarioModel.find({
      _id: { $in: usuariosSeleccionados },
    })
      .select("telefono")
      .exec();
    const telefonosUsuarios = findUsuarios.map((usuario) => usuario.telefono);
    numerosTelefonicos.push(...telefonosUsuarios);

    // agrega de clientes clientesSeleccionados
    const findClientes = await clientesModel
      .find({ _id: { $in: clientesSeleccionados } })
      .select("telefono")
      .exec();
    const telefonosClientes = findClientes.map((cliente) => cliente.telefono);
    numerosTelefonicos.push(...telefonosClientes);

    // agrega de invitados invitadosSeleccionados

    const telefonosInvitados = await prospectosModel
      .find({ _id: { $in: invitadosSeleccionados } })
      .select("telefono")
      .exec();
    const telefonosInvitadosArray = telefonosInvitados.map(
      (invitado) => invitado.telefono
    );
    numerosTelefonicos.push(...telefonosInvitadosArray);

    // limpiar iguales y nulos de la lista
    const numerosTelefonicosUnicos = Array.from(
      new Set(numerosTelefonicos.filter((num) => num != null))
    );

    // enviar mensajes de whatsapp

    if (numerosTelefonicosUnicos.length > 0) {
      const findUsuarioResponsable = await UsuarioModel.findById(usuario)
        .select("nombre")
        .exec();

      const fechaInicioStr = moment(fecha)?.format("DD/MM/YYYY") || " ";
      const horaInicioStr = moment(fecha)?.format("hh:mm A") || " ";
      const fechaFinalStr = fechaFin
        ? moment(fechaFin)?.format("DD/MM/YYYY")
        : " ";
      const horaFinalStr = fechaFin ? moment(fechaFin)?.format("hh:mm A") : " ";

      await Promise.all(
        numerosTelefonicosUnicos.map(async (telefono) => {
          await sendWhatsappEvent({
            responsable: findUsuarioResponsable.nombre,
            to: telefono,
            evento: titulo || " ",
            descripcion: descripcion || " ",
            fechaInicio: fechaInicioStr,
            horaInicio: horaInicioStr,
            fechaFinal: fechaFinalStr,
            horaFinal: horaFinalStr,
          });
        })
      );
    }

    return res.status(201).json({ evento: event, message: "Evento creado" });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const updateEvent = async (req, res) => {
  const {
    idEvento,
    idRecordatorio,
    titulo,
    estatus,
    comentario,
    descripcion,
    color,
    fechaInicio,
    horaInicio,
    fechaFinal,
    horaFinal,
    recordar,
    usuario,
    usuariosSeleccionados,
    clientesSeleccionados,
    invitadosSeleccionados, // Añadido
  } = req.body;

  try {
    const findEvento = await ExpedientesAgenda.findById(idEvento);

    if (!findEvento) {
      return res.status(404).json({ message: "Evento no encontrado" });
    }

    const fecha = moment.utc(fechaInicio).toDate();

    if (horaInicio) {
      fecha.setHours(moment(horaInicio).hours());
      fecha.setMinutes(moment(horaInicio).minutes());
    }

    const fechaFin = fechaFinal ? moment.utc(fechaFinal).toDate() : null;

    if (horaFinal && fechaFin) {
      fechaFin.setHours(moment(horaFinal).hours());
      fechaFin.setMinutes(moment(horaFinal).minutes());
    }

    const obj = {
      title: titulo,
      descripcion,
      fecha,
      fechaFin,
    };

    const updatedEvent = await ExpedientesAgenda.findByIdAndUpdate(
      idEvento,
      obj,
      { new: true }
    );

    const fechaRecordatorioMx =
      recordar !== null
        ? moment(fecha).subtract(parseInt(recordar), "minutes").toDate()
        : null;

    const objRecordatorio = {
      estatus,
      comentario,
      color,
      fechaRecordatorio: fechaRecordatorioMx,
      recordar,
    };

    // Manejo de usuarios
    const usuariosPrevios = await UsuariosAgenda.find({
      agenda: idEvento,
    }).select("usuario");
    const usuariosPreviosIds = usuariosPrevios.map((ua) =>
      ua?.usuario?.toString()
    );
    const usuariosAEliminar = usuariosPreviosIds.filter(
      (id) => !usuariosSeleccionados?.includes(id)
    );

    if (usuariosAEliminar.length > 0) {
      await UsuariosAgenda.deleteMany({
        agenda: idEvento,
        usuario: { $in: usuariosAEliminar },
      });
    }

    if (usuariosSeleccionados?.length > 0) {
      for (const _id of usuariosSeleccionados) {
        const findRecordatorio = await UsuariosAgenda.findOne({
          agenda: idEvento,
          usuario: _id,
        });

        if (!findRecordatorio) {
          await UsuariosAgenda.create({
            despacho: updatedEvent.despacho,
            expediente: updatedEvent.expediente,
            usuario: _id,
            fechaRecordatorio: fechaRecordatorioMx,
            agenda: updatedEvent._id,
            recordar,
            color,
          });
        } else {
          await UsuariosAgenda.findByIdAndUpdate(
            findRecordatorio._id,
            objRecordatorio
          );
        }
      }
    }

    // Manejo de clientes
    const clientesPrevios = await UsuariosAgenda.find({
      agenda: idEvento,
    }).select("cliente");
    const clientesPreviosIds = clientesPrevios
      .map((c) => c.cliente?.toString())
      .filter((id) => id !== undefined);
    const clientesAEliminar = clientesPreviosIds.filter(
      (id) => !clientesSeleccionados.includes(id)
    );

    if (clientesAEliminar.length > 0) {
      await UsuariosAgenda.deleteMany({
        agenda: idEvento,
        cliente: { $in: clientesAEliminar },
      });
    }

    if (clientesSeleccionados.length > 0) {
      for (const _id of clientesSeleccionados) {
        const findCliente = await UsuariosAgenda.findOne({
          agenda: idEvento,
          cliente: _id,
        });

        if (!findCliente) {
          await UsuariosAgenda.create({
            despacho: updatedEvent.despacho,
            expediente: updatedEvent.expediente,
            cliente: _id,
            fechaRecordatorio: fechaRecordatorioMx,
            agenda: updatedEvent._id,
            recordar,
            color,
          });
        } else {
          await UsuariosAgenda.findByIdAndUpdate(
            findCliente._id,
            objRecordatorio
          );
        }
      }
    }

    // manejo de invitados prospectos
    const prospectosPrevios = await UsuariosAgenda.find({
      agenda: idEvento,
    }).select("prospecto");
    const prospectosPreviosIds = prospectosPrevios
      .map((p) => p.prospecto?.toString())
      .filter((id) => id !== undefined);
    const prospectosAEliminar = prospectosPreviosIds.filter(
      (id) => !invitadosSeleccionados.includes(id)
    );

    if (prospectosAEliminar.length > 0) {
      await UsuariosAgenda.deleteMany({
        agenda: idEvento,
        prospecto: { $in: prospectosAEliminar },
      });
    }

    if (invitadosSeleccionados.length > 0) {
      for (const _id of invitadosSeleccionados) {
        const findProspecto = await UsuariosAgenda.findOne({
          agenda: idEvento,
          prospecto: _id,
        });

        if (!findProspecto) {
          await UsuariosAgenda.create({
            despacho: updatedEvent.despacho,
            expediente: updatedEvent.expediente,
            prospecto: _id,
            fechaRecordatorio: fechaRecordatorioMx,
            agenda: updatedEvent._id,
            recordar,
            color,
          });
        } else {
          await UsuariosAgenda.findByIdAndUpdate(
            findProspecto._id,
            objRecordatorio
          );
        }
      }
    }

    // Actualizar el recordatorio
    await UsuariosAgenda.findByIdAndUpdate(idRecordatorio, objRecordatorio);

    if (updatedEvent.despacho && updatedEvent.expediente) {
      // const tituloMovimiento = "Evento actualizado";
      const tituloMovimiento = "Agenda";
      const descripcionMovimiento = `Se ha actualizado el evento con el título: ${titulo}. Fecha: ${moment(
        fecha
      ).format("DD/MM/YYYY")}. Hora: ${moment(fecha).format("hh:mm A")}.`;
      notificarExpedienteUsuario({
        despacho: updatedEvent.despacho,
        expediente: updatedEvent.expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario,
      });
    }

    const numerosTelefonicos = [];

    // agrega de usuarios usuariosSeleccionados
    const findUsuarios = await UsuarioModel.find({
      _id: { $in: usuariosSeleccionados },
    })
      .select("telefono")
      .exec();
    const telefonosUsuarios = findUsuarios.map((usuario) => usuario.telefono);
    numerosTelefonicos.push(...telefonosUsuarios);

    // agrega de clientes clientesSeleccionados
    const findClientes = await clientesModel
      .find({ _id: { $in: clientesSeleccionados } })
      .select("telefono")
      .exec();
    const telefonosClientes = findClientes.map((cliente) => cliente.telefono);
    numerosTelefonicos.push(...telefonosClientes);

    // agrega de invitados invitadosSeleccionados

    const telefonosInvitados = await prospectosModel
      .find({ _id: { $in: invitadosSeleccionados } })
      .select("telefono")
      .exec();
    const telefonosInvitadosArray = telefonosInvitados.map(
      (invitado) => invitado.telefono
    );
    numerosTelefonicos.push(...telefonosInvitadosArray);

    // limpiar iguales y nulos de la lista
    const numerosTelefonicosUnicos = Array.from(
      new Set(numerosTelefonicos.filter((num) => num != null))
    );

    console.log(numerosTelefonicosUnicos, "numerosTelefonicosUnicos");

    const findUsuarioResponsable = await UsuarioModel.findById(usuario)
      .select("nombre")
      .exec();

    // enviar mensajes de whatsapp
    if (numerosTelefonicosUnicos.length > 0) {
      const fechaInicioStr = moment(fecha)?.format("DD/MM/YYYY") || " ";
      const horaInicioStr = moment(fecha)?.format("hh:mm A") || " ";
      const fechaFinalStr = fechaFin
        ? moment(fechaFin)?.format("DD/MM/YYYY")
        : " ";
      const horaFinalStr = fechaFin ? moment(fechaFin)?.format("hh:mm A") : " ";

      await Promise.all(
        numerosTelefonicosUnicos.map(async (telefono) => {
          await sendWhatsappEvent({
            responsable: findUsuarioResponsable.nombre,
            to: telefono,
            evento: titulo || " ",
            descripcion: descripcion || " ",
            fechaInicio: fechaInicioStr,
            horaInicio: horaInicioStr,
            fechaFinal: fechaFinalStr,
            horaFinal: horaFinalStr,
          });
        })
      );
    }

    return res
      .status(200)
      .json({ evento: updatedEvent, message: "Evento actualizado" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ message: error.message });
  }
};

const getEventosByExpediente = async (req, res) => {
  const { expediente } = req.params;

  try {
    if (!expediente) {
      return res.status(400).json({ message: "Expediente no especificado" });
    }

    const query = { expediente };

    const events = await UsuariosAgenda.find(query).populate("agenda").exec();
    // ordenar por fecha ASC
    const orderedEvents = events.sort((a, b) => {
      const fechaHoraInicioA = moment(a.agenda.fecha);
      const fechaHoraInicioB = moment(b.agenda.fecha);
      const horaInicioA = a.agenda.horaInicio;
      const horaInicioB = b.agenda.horaInicio;

      if (fechaHoraInicioA.isBefore(fechaHoraInicioB)) {
        return -1;
      } else if (fechaHoraInicioA.isAfter(fechaHoraInicioB)) {
        return 1;
      } else {
        if (horaInicioA < horaInicioB) {
          return -1;
        } else if (horaInicioA > horaInicioB) {
          return 1;
        } else {
          return 0;
        }
      }
    });

    const formattedEvents = formattedEventsFun(orderedEvents);

    return res.status(200).json({ eventos: formattedEvents });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const formattedEventsFun = (events) =>
  events.map((event) => {
    const { agenda, estatus, fechaRecordatorio, comentario } = event;

    let horaInicio = null;

    if (agenda.fecha) {
      horaInicio = moment(agenda.fecha).format("hh:mm A");
      // horaInicio.setHours(moment(agenda.horaInicio).hours());
    }

    let horaFin = null;

    if (agenda.fechaFin) {
      horaFin = moment(agenda.fechaFin).format("hh:mm A");
      // horaFin.setHours(moment(agenda.horaFin).hours());
    }

    return {
      id: agenda._id,
      _id: event._id,
      title: agenda.title,
      descripcion: agenda.descripcion,
      fecha: agenda.fecha,
      horaInicio: agenda.horaInicio || horaInicio,
      horaFin: agenda.horaFin || horaFin,
      creadoPor: agenda.creadoPor,
      fechaCreacion: agenda.fechaCreacion,
      estatus,
      color: event.color,
      fechaRecordatorio,
      comentario,
      recordar: event.recordar,
      fechaFin: agenda.fechaFin,
    };
  });

const getEventosClientes = async (req, res) => {
  const { cliente } = req.params;

  try {
    const findExpediente = await Expedientes.find({ cliente }).select("_id");

    const expedientes = findExpediente.map((expediente) => expediente._id);

    const query = { expediente: { $in: expedientes } };

    const events = await UsuariosAgenda.find(query).populate("agenda").exec();

    const formattedEvents = formattedEventsFun(events);

    return res.status(200).json({ eventos: formattedEvents });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const deleteEvento = async (req, res) => {
  const { idEvento, idRecordatorio } = req.params;

  try {
    const findEventoUsuario = await UsuariosAgenda.findById(idRecordatorio);

    if (!findEventoUsuario) {
      return res.status(404).json({ message: "Evento no encontrado" });
    }

    // si solo hay 1 usuario en el evento, se elimina el evento completo
    const findEventos = await UsuariosAgenda.find({ agenda: idEvento });

    if (findEventos.length === 1) {
      await ExpedientesAgenda.findByIdAndDelete(idEvento);
    }

    await UsuariosAgenda.findByIdAndDelete(idRecordatorio);

    if (findEventoUsuario.despacho && findEventoUsuario.expediente) {
      const tituloMovimiento = "Agenda";
      const descripcionMovimiento = `Se ha eliminado el evento con el título: ${
        findEventoUsuario.agenda.title
      }. Fecha: ${moment(findEventoUsuario.agenda.fecha).format(
        "DD/MM/YYYY"
      )}. Hora: ${moment(findEventoUsuario.agenda.fecha).format("hh:mm A")}.`;
      notificarExpedienteUsuario({
        despacho: findEventoUsuario.despacho,
        expediente: findEventoUsuario.expediente,
        descripcion: descripcionMovimiento,
        accionRealizada: tituloMovimiento,
        usuario: findEventoUsuario.usuario,
      });
    }

    return res.status(200).json({ message: "Evento eliminado" });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const getEventByID = async (req, res) => {
  const { id } = req.params;

  try {
    const event = await ExpedientesAgenda.findById(id)
      .populate("expediente")
      .lean();

    if (!event) {
      return res.status(404).json({ message: "Evento no encontrado" });
    }

    // Obtener los usuarios asociados al evento, incluyendo invitados
    const usuariosEvento = await UsuariosAgenda.find({ agenda: event._id })
      .select("usuario cliente invitado prospecto")
      .exec();

    // Filtrar los usuarios para eliminar los nulos o vacíos y mapear los invitados correctamente
    event.usuariosEvento = usuariosEvento
      .map((usuario) => usuario.usuario)
      .filter((usuario) => usuario !== null && usuario !== undefined);

    event.clientesEvento = usuariosEvento
      .map((cliente) => cliente.cliente)
      .filter((cliente) => cliente !== null && cliente !== undefined);

    event.prospectoEvento = usuariosEvento
      ?.map((prospecto) => prospecto?.prospecto)
      ?.filter((prospecto) => prospecto !== null && prospecto !== undefined);

    return res.status(200).json({ evento: event });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

module.exports = {
  getEvents,
  createEvent,
  updateEvent,
  getEventosByExpediente,
  getEventosClientes,
  deleteEvento,
  getEventByID,
};
