const etapasProcesalesModel = require('../models/EtapasProcesales.js');
const materiasModel = require('../models/Materias.js');

const getEtapasProcesales = async (req, res) => {
  const { estatus, page = 1, materia, search } = req.query;

  const options = {
    page: parseInt(page, 10),
    limit: 50,
    sort: { materia: 1, nombre: 1 },
    populate: {
      path: 'materia',
      select: 'nombre'
    }
  };

  try {
    const query = {};

    if (estatus) query.estatus = estatus;
    if (materia) query.materia = materia;
    if (search) query.nombre = { $regex: search, $options: 'i' };

    const etapasProcesales = await etapasProcesalesModel.paginate(query, options);

    // // Manual sort
    // etapasProcesales.docs.sort((a, b) => {
    //   if (a.materia.nombre < b.materia.nombre) return -1;
    //   if (a.materia.nombre > b.materia.nombre) return 1;
    //   if (a.nombre < b.nombre) return -1;
    //   if (a.nombre > b.nombre) return 1;
    //   return 0;
    // });

    res.status(200).json({ etapasProcesales });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getEtapasProcesalesSinPagina = async (req, res) => {
  const { materia } = req.query;

  try {
    const query = {
      estatus: 'Activo'
    };

    // if (estatus) query.estatus = estatus;

    if (materia) query.materia = materia;

    const etapasProcesales = await etapasProcesalesModel.find(query).sort({ nombre: 1 });

    const etapasProcesalesSinPagina = etapasProcesales.map((etapaProcesal) => {
      return {
        _id: etapaProcesal._id,
        nombre: etapaProcesal.nombre,
        label: etapaProcesal.nombre,
        value: etapaProcesal._id
      };
    });
    res.status(200).json({ etapasProcesales: etapasProcesalesSinPagina });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getMateriasEtapas = async (req, res) => {
  const { estatus, page = 1, materia } = req.query;

  const options = {
    page,
    limit: 5,
    sort: { nombre: 1 }
  };

  try {
    const query = {};
    if (materia) query.materia = materia;

    const materiasFind = await materiasModel.paginate(query, options);

    // Create an array of promises to fetch etapasProcesales for each materia
    const materiasWithEtapasProcesales = await Promise.all(
      materiasFind.docs.map(async (materia) => {
        const queryEtapasProcesales = { materia: materia._id };
        if (estatus) queryEtapasProcesales.estatus = estatus;

        const etapasProcesales = await etapasProcesalesModel.find(query).sort({ nombre: 1 }).select('nombre estatus');
        return { ...materia.toObject(), etapasProcesales }; // Use .toObject() to convert Mongoose document to plain object
      })
    );

    // Return the modified result with etapasProcesales included
    res.status(200).json({ materias: { ...materiasFind, docs: materiasWithEtapasProcesales } });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getEtapaProcesalById = async (req, res) => {
  const { id } = req.params;

  try {
    const materia = await etapasProcesalesModel.findById(id);
    if (!materia) {
      return res.status(404).json({ message: 'No se encontró la materia' });
    }

    res.status(200).json(materia);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createEtapaProcesal = async (req, res) => {
  const { nombre, estatus, materia, tipo } = req.body;

  console.log(req.body);

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  const etapaProcesal = {
    nombre,
    materia,
    tipo,
    estatus: estatus || 'Activo'
  };

  try {
    const newEtapa = await etapasProcesalesModel.create(etapaProcesal);
    res.status(201).json(newEtapa);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const updateEtapaProcesal = async (req, res) => {
  const { nombre, estatus, materia, tipo } = req.body;
  const { id } = req.params;

  if (!id) {
    return res.status(404).json({ message: 'El id es requerido' });
  }

  if (!nombre) {
    return res.status(404).json({ message: 'El nombre es requerido' });
  }

  try {
    const etapaProcesal = await etapasProcesalesModel.findById(id);

    if (!etapaProcesal) {
      return res.status(404).json({ message: 'Etapa no encontrada' });
    }

    etapaProcesal.nombre = nombre;
    etapaProcesal.materia = materia || etapaProcesal.materia;
    etapaProcesal.tipo = tipo || etapaProcesal.tipo;
    etapaProcesal.estatus = estatus || 'Activo';

    const updatedEtapa = await etapaProcesal.save();
    res.status(200).json(updatedEtapa);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

const deleteEtapaProcesal = async (req, res) => {
  const { id } = req.params;

  try {
    await etapasProcesalesModel.findByIdAndRemove(id);
    res.status(200).json({ message: 'Etapa eliminada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};
module.exports = {
  getEtapasProcesales,
  getEtapasProcesalesSinPagina,
  getMateriasEtapas,
  createEtapaProcesal,
  updateEtapaProcesal,
  getEtapaProcesalById,
  deleteEtapaProcesal
};
