const etapasProcesalesModel = require('../models/EtapasProcesalesDespachos.js');
const ExpedientesModel = require('../models/Expedientes.js');

const getEtapasProcesales = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, materia, search } = req.query;

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  const options = {
    page,
    limit: 15,
    sort: {
      materia: 1,
      nombre: 1
    },
    populate: {
      path: 'materia',
      select: 'nombre',
      sort: { nombre: 1 }
    }
  };

  try {
    const query = {
      despacho
    };

    if (materia) {
      query.materia = materia;
    }

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      query.nombre = { $regex: search, $options: 'i' };
    }

    const etapasProcesales = await etapasProcesalesModel.paginate(query, options);

    res.status(200).json({ etapasProcesales });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getEtapasProcesalesSinPagina = async (req, res) => {
  const { despacho } = req.params;
  const { materia } = req.query;

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  if (!materia) {
    return res.status(400).json({ message: 'La materia es requerida' });
  }

  try {
    const query = {
      estatus: 'Activo',
      despacho
    };

    // if (estatus) query.estatus = estatus;

    if (materia) query.materia = materia;

    const etapasProcesales = await etapasProcesalesModel.find(query).sort({ nombre: 1 });

    const etapasProcesalesSinPagina = etapasProcesales.map((etapaProcesal) => {
      return {
        key: etapaProcesal._id,
        nombre: etapaProcesal.nombre,
        label: etapaProcesal.nombre,
        value: etapaProcesal._id
      };
    });

    res.status(200).json({ etapasProcesales: etapasProcesalesSinPagina });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getEtapasProcesal = async (req, res) => {
  const { despacho, id } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  if (!id) {
    return res.status(400).json({ message: 'El id es requerido' });
  }

  try {
    const etapaProcesal = await etapasProcesalesModel.findOne({ _id: id, despacho });

    if (!etapaProcesal) {
      return res.status(404).json({ message: 'La etapa no existe' });
    }

    res.status(200).json({ etapaProcesal });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createEtapasProcesales = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, estatus, materia } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!materia) {
    return res.status(400).json({ message: 'La materia es requerida' });
  }

  try {
    const etapaProcesal = await etapasProcesalesModel.create({
      nombre,
      estatus: estatus || 'Activo',
      materia,
      despacho
    });

    res.status(200).json({ etapaProcesal, message: 'La etapa ha sido creada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteEtapaProcesal = async (req, res) => {
  const { id, despacho } = req.params;
  try {
    const findExpedientes = await ExpedientesModel.countDocuments({ etapaProcesal: id, despacho });

    if (findExpedientes > 0) {
      return res.status(400).json({ message: 'No se puede eliminar la etapa porque tiene expedientes asociados' });
    }

    await etapasProcesalesModel.findOneAndDelete({ _id: id, despacho });

    return res.status(200).json({ message: 'La etapa ha sido eliminada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateEtapaProcesal = async (req, res) => {
  const { id, despacho } = req.params;
  const { nombre, estatus, materia } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!materia) {
    return res.status(400).json({ message: 'La materia es requerida' });
  }

  try {
    const etapaProcesal = await etapasProcesalesModel.findOneAndUpdate(
      { _id: id, despacho },
      { nombre, estatus, materia },
      { new: true }
    );

    res.status(200).json({ etapaProcesal, message: 'La etapa ha sido actualizada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getEtapasProcesalesSinPagina,
  createEtapasProcesales,
  deleteEtapaProcesal,
  getEtapasProcesales,
  getEtapasProcesal,
  updateEtapaProcesal
};
