const { default: mongoose } = require('mongoose');
const Ventas = require('../models/Ventas');
const Despachos = require('../models/Despachos');
const Usuarios = require('../models/Usuarios');

const getEstadisticas = async (req, res) => {
  try {
    // Estados con mas despachos
    const resultEstados = Despachos.aggregate([
      {
        $match: { estatus: 'Activo' }
      },
      {
        $group: {
          _id: '$estado',
          totalDespachos: { $sum: 1 }
        }
      },
      {
        $sort: { totalDespachos: -1 }
      },
      {
        $limit: 5
      },
      {
        $lookup: {
          from: 'estados',
          localField: '_id',
          foreignField: '_id',
          as: 'estadoDetalles'
        }
      },
      {
        $unwind: '$estadoDetalles'
      },
      {
        $project: {
          _id: 0,
          nombre: '$estadoDetalles.nombre',
          cantidad: '$totalDespachos'
        }
      }
    ]);

    // Meses con mas ventas

    // Contadores

    const despachosActivosCount = Usuarios.countDocuments({
      'tipoUsuarios.despacho': { $exists: true },
      estatus: 'Activo'
    });

    const distribuidoresActivosCount = Usuarios.countDocuments({
      'tipoUsuarios.distribuidor': { $exists: true },
      estatus: 'Activo'
    });

    const despachosCount = Despachos.countDocuments({
      estatus: 'Activo'
    });

    const [estadosDespacho, abogados, distribuidoresCount, despachos] =
      await Promise.all([
        resultEstados,
        despachosActivosCount,
        distribuidoresActivosCount,
        despachosCount
      ]);

    const data = {
      estadosDespacho,
      abogados,
      distribuidoresCount,
      despachos
    };
    return res.status(200).json(data);
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al obtener las estadísticas',
      error
    });
  }
};

const getPaquetesVendidos = async (req, res) => {
  const { fechaInicioPaquetes, fechaFinalPaquetes, filtroPaquetes } = req.query;

  try {
    let fechaInicio = null;
    let fechaFinal = null;

    if (filtroPaquetes) {
      const hoy = new Date();
      if (filtroPaquetes === 'ultimos10dias') {
        fechaInicio = new Date(hoy.setDate(hoy.getDate() - 10));
        fechaFinal = new Date(); // Hoy
      } else if (filtroPaquetes === 'mesActual') {
        fechaInicio = new Date(hoy.getFullYear(), hoy.getMonth(), 1);
        fechaFinal = new Date(); // Hoy
      } else if (filtroPaquetes === 'mesAnterior') {
        fechaInicio = new Date(hoy.getFullYear(), hoy.getMonth() - 1, 1);
        fechaFinal = new Date(hoy.getFullYear(), hoy.getMonth(), 0);
      }
    } else {
      fechaInicio = fechaInicioPaquetes ? new Date(fechaInicioPaquetes) : null;
      fechaFinal = fechaFinalPaquetes ? new Date(fechaFinalPaquetes) : null;
    }

    const filtrosFecha = {};

    if (fechaInicio) {
      filtrosFecha.$gte = fechaInicio;
    }
    if (fechaFinal) {
      filtrosFecha.$lte = fechaFinal;
    }

    const resultPaquetes = await Ventas.aggregate([
      {
        $match: {
          producto: {
            $ne: mongoose.Types.ObjectId('66da43e02412e32ff68279d9')
          },
          ...((fechaInicio || fechaFinal) && { fechaVenta: filtrosFecha })
        }
      },
      {
        $group: {
          _id: '$producto',
          totalVentas: { $sum: 1 }
        }
      },
      {
        $sort: { totalVentas: -1 }
      },
      {
        $limit: 5
      },
      {
        $lookup: {
          from: 'productos',
          localField: '_id',
          foreignField: '_id',
          as: 'productoDetalles'
        }
      },
      {
        $unwind: '$productoDetalles'
      },
      {
        $project: {
          _id: 0,
          cantidad: '$totalVentas',
          tipo_producto: '$productoDetalles.tipo_producto',
          nombre: '$productoDetalles.nombre'
        }
      }
    ]);

    return res.status(200).json(resultPaquetes);
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al obtener las estadísticas',
      error
    });
  }
};

const getDistribuidoresVentas = async (req, res) => {
  const { fechaInicioPaquetes, fechaFinalPaquetes, filtroPaquetes } = req.query;
  try {
    let fechaInicio = null;
    let fechaFinal = null;

    if (filtroPaquetes) {
      const hoy = new Date();
      if (filtroPaquetes === 'ultimos10dias') {
        fechaInicio = new Date(hoy.setDate(hoy.getDate() - 10));
        fechaFinal = new Date(); // Hoy
      } else if (filtroPaquetes === 'mesActual') {
        fechaInicio = new Date(hoy.getFullYear(), hoy.getMonth(), 1);
        fechaFinal = new Date(); // Hoy
      } else if (filtroPaquetes === 'mesAnterior') {
        fechaInicio = new Date(hoy.getFullYear(), hoy.getMonth() - 1, 1);
        fechaFinal = new Date(hoy.getFullYear(), hoy.getMonth(), 0);
      }
    } else {
      fechaInicio = fechaInicioPaquetes ? new Date(fechaInicioPaquetes) : null;
      fechaFinal = fechaFinalPaquetes ? new Date(fechaFinalPaquetes) : null;
    }

    const filtrosFecha = {};

    if (fechaInicio) {
      filtrosFecha.$gte = fechaInicio;
    }
    if (fechaFinal) {
      filtrosFecha.$lte = fechaFinal;
    }

    const resultDistribuidoresVentas = await Ventas.aggregate([
      {
        $match: {
          'distribuidor.distribuidor': { $ne: null },
          ...(fechaInicio || fechaFinal ? { fechaVenta: filtrosFecha } : {})
        }
      },
      {
        $group: {
          _id: '$distribuidor.distribuidor',
          totalVentas: { $sum: 1 },
          totalImporte: { $sum: '$importe' }
        }
      },
      {
        $sort: { totalVentas: -1 }
      },
      {
        $lookup: {
          from: 'usuarios',
          localField: '_id',
          foreignField: '_id',
          as: 'distribuidorDetalles'
        }
      },
      {
        $unwind: '$distribuidorDetalles'
      },
      {
        $project: {
          _id: 0,
          nombre: '$distribuidorDetalles.nombre',
          cantidad: '$totalVentas'
          //   totalImporte: "$totalImporte",
        }
      },
      {
        $limit: 5
      }
    ]);

    return res.status(200).json(resultDistribuidoresVentas);
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al obtener las estadísticas',
      error
    });
  }
};

const getMesesVentasYear = async (req, res) => {
  const { year } = req.query;
  try {
    const months = [
      'Enero',
      'Febrero',
      'Marzo',
      'Abril',
      'Mayo',
      'Junio',
      'Julio',
      'Agosto',
      'Septiembre',
      'Octubre',
      'Noviembre',
      'Diciembre'
    ];

    const anioActual = year
      ? new Date(year, 0, 1).getFullYear()
      : new Date().getFullYear();
    const mesesConMasVentas = await Ventas.aggregate([
      {
        $match: {
          $expr: {
            $eq: [{ $year: '$fechaVenta' }, anioActual]
          }
        }
      },
      {
        $group: {
          _id: {
            mes: { $month: '$fechaVenta' },
            anio: { $year: '$fechaVenta' }
          },
          totalVentas: { $sum: 1 },
          totalImporte: { $sum: '$importe' }
        }
      },
      {
        $sort: { totalVentas: -1 }
      },
      {
        $project: {
          _id: 0,
          mes: '$_id.mes',
          anio: '$_id.anio',
          cantidad: '$totalVentas',
          totalImporte: '$totalImporte'
        }
      },
      {
        $limit: 5
      }
    ]);

    const mesesVentas = mesesConMasVentas.map((venta) => ({
      nombre: months[venta.mes - 1],
      anio: venta.anio,
      total: venta.cantidad,
      cantidad: venta.totalImporte
    }));

    return res.status(200).json(mesesVentas);
  } catch (error) {
    console.log(error);
    return res.status(500).json({
      mensaje: 'Error al obtener las estadísticas',
      error
    });
  }
};
module.exports = {
  getEstadisticas,
  getPaquetesVendidos,
  getDistribuidoresVentas,
  getMesesVentasYear
};
