/* eslint-disable indent */
const { default: mongoose } = require('mongoose');
const { generatePassword, encriptar, formatDateTime, formatMoney } = require('../config/functionsGlobal.js');
const { buildFileUri } = require('../config/s3.js');
const TipoUsuarios = require('../models/TipoUsuarios.js');
const UsuariosModel = require('../models/Usuarios.js');
const moment = require('moment-timezone');
const VentaModel = require('../models/Ventas.js');
const paqueteModel = require('../models/Productos.js');

const getDistribuidoresAdmin = async (req, res) => {
  const { estatus, page = 1, search, limit } = req.query;

  const options = {
    page,
    limit: limit || 10,
    sort: {
      'licencia.vigencia': -1,
      nombre: 1
    },
    populate: [
      {
        path: 'licencia.producto',
        select: 'nombre imagen'
      },
      {
        path: 'licencia.referidoPor',
        select: 'nombre foto'
      }
    ]
  };

  const query = {
    'tipoUsuarios.distribuidor': { $exists: true },
    estatus: { $in: ['Activo', 'Inactivo'] }
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [
      { nombre: { $regex: search, $options: 'i' } },
      { email: { $regex: search, $options: 'i' } }
    ];
  }

  try {
    const data = await UsuariosModel.paginate(query, options);

    data.docs = data.docs.map((distribuidor) => {
      let diasRestantes = 0;

      if (distribuidor.licencia) {
        const licencia = distribuidor.licencia;
        const vigencia = licencia.vigencia;
        const fechaMexico = moment()
          .tz('America/Mexico_City')
          .format('YYYY-MM-DD HH:mm:ss');

        diasRestantes = moment(vigencia).diff(fechaMexico, 'days');
      }

      let referidoPor = null;

      if (distribuidor.licencia?.referidoPor) {
        referidoPor = {
          nombre: distribuidor.licencia.referidoPor.nombre,
          foto: distribuidor.licencia.referidoPor.foto
            ? buildFileUri(`usuarios/${distribuidor.licencia.referidoPor.foto}`)
            : buildFileUri('default/icono_usuario_100x100_04.jpg')
        };
      }

      return {
        ...distribuidor._doc,
        ultimoAcceso: formatDateTime(distribuidor.ultimoAcceso),
        diasRestantes: diasRestantes > 0 ? diasRestantes : 0,
        logo: distribuidor.foto
          ? buildFileUri(`usuarios/${distribuidor?.foto}`)
          : buildFileUri('default/icono_usuario_100x100_04.jpg'),
        imagenProducto: distribuidor.licencia?.producto?.imagen
          ? buildFileUri(`productos/${distribuidor.licencia.producto.imagen}`)
          : '',
        referidoPor
      };
    });

    return res.status(200).json(data);
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los distribuidores', error });
  }
};

const updateDistribuidor = async (req, res) => {
  const { _id } = req.params;
  const {
    nombre,
    apellidoP,
    apellidoM,
    correo,
    telefono,
    estatus,
    licencia = null
  } = req.body;

  try {
    let datosLicencia;
    let vigencia;
    if (licencia) {
      datosLicencia = await paqueteModel.findById(licencia);
      vigencia = moment()
        .tz('America/Mexico_City')
        .add(datosLicencia.cantidad, 'years');
    }

    await UsuariosModel.findByIdAndUpdate(
      _id,
      {
        nombre,
        apellidoPaterno: apellidoP,
        apellidoMaterno: apellidoM,
        email: correo,
        telefono,
        estatus,
        licencia: {
          producto: licencia || undefined,
          comision: licencia ? datosLicencia.utilidad : undefined,
          vigencia: licencia ? vigencia : undefined
        },
        fechaCreacion: new Date()
      },
      { new: true }
    );

    return res.status(200).json({ message: 'Usuario actualizado' });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al actualizar el usuario', error });
  }
};

const createDistribuidor = async (req, res) => {
  const { nombre, apellidoP, apellidoM, correo, telefono, estatus, licencia = null } = req.body;

  if (!nombre || !apellidoP || !correo || !telefono) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }

  try {
    const existe = await UsuariosModel.findOne({ email: correo });
    if (existe) {
      return res.status(400).json({ message: 'El correo ya está en uso' });
    }

    const password = generatePassword();
    const passwordEncriptada = encriptar(password);

    const tipoUsuario = await TipoUsuarios.findOne({
      tipo: 'distribuidor'
    }).select('_id');

    let datosLicencia;
    let vigencia;
    if (licencia) {
      datosLicencia = await paqueteModel.findById(licencia);
      vigencia = moment()
        .tz('America/Mexico_City')
        .add(datosLicencia.cantidad, 'years');
    }

    const usuario = UsuariosModel({
      nombre,
      apellidoPaterno: apellidoP,
      apellidoMaterno: apellidoM,
      email: correo,
      telefono,
      password: passwordEncriptada,
      tipoUsuarios: {
        distribuidor: tipoUsuario._id
      },
      // tipoUsuario: tipoUsuario._id,
      // tipo: 'distribuidor',
      estatus,
      licencia: {
        licenciaFree: !!licencia,
        referidoPor: '671812af8794f718b55d4964',
        producto: licencia || undefined,
        comision: licencia ? datosLicencia.utilidad : undefined,
        vigencia: licencia ? vigencia : undefined
      },
      fechaCreacion: new Date()
    });

    const result = await usuario.save();

    if (!result) {
      return res.status(400).json({ message: 'Error al crear el usuario' });
    }

    // sendWhatsappRecovery({
    //   to: telefono,
    //   correo,
    //   password,
    //   url: DISTRIBUIDOR_APP
    // });

    // const htmlRegistro = RegistroUsuarioHTML({
    //   nombre,
    //   usuario: correo,
    //   password,
    //   url: DISTRIBUIDOR_APP
    // });

    // sendMail(htmlRegistro, 'Recordatorio de contraseña', correo);

    return res.status(200).json({ message: 'Usuario creado' });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al crear el usuario', error });
  }
};

const getDistribuidorById = async (req, res) => {
  const { id } = req.params;

  try {
    const resultado = await UsuariosModel.findById(id).populate({
      path: 'licencia.producto',
      select: 'nombre imagen'
    });

    if (!resultado) {
      return res
        .status(400)
        .json({ message: 'Error al encontrar el distribuidor' });
    }
    return res.status(200).json(resultado);
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al crear el usuario', error });
  }
};

const getVentasByDistribuidores = async (req, res) => {
  const { usuario } = req.params;

  const {
    producto,
    page = 1,
    fechaInicio = null,
    fechaFinal = null
  } = req.query;

  const limit = 10;
  const options = {
    page,
    limit,
    sort: {
      fechaVenta: -1
    },
    populate: [
      {
        path: 'producto',
        select: 'nombre imagen precio'
      }
    ],
    lean: true
  };

  // Base de coincidencias para la agregación
  const query = {
    usuario: mongoose.Types.ObjectId(usuario),
    ...(producto ? { producto: mongoose.Types.ObjectId(producto) } : {}),
    ...(fechaInicio &&
      fechaFinal &&
      moment(fechaInicio).isValid() &&
      moment(fechaFinal).isValid()
      ? {
        fechaVenta: {
          $gte: moment(fechaInicio).startOf('day').toDate(),
          $lte: moment(fechaFinal).endOf('day').toDate()
        }
      }
      : {})
  };

  try {
    const [findVentas, totalComision, importeTotal] = await Promise.all([
      VentaModel.paginate(query, options),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            totalComision: { $sum: '$distribuidor.importe' }
          }
        }
      ]),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            importe: { $sum: '$importe' }
          }
        }
      ])
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const { fechaVenta, producto } = venta;
      let imagen = '';

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = '';

      if (tipoReferencia === 'Paypal') {
        tipoReferenciaLogo =
          'https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png';
      } else if (tipoReferencia === 'Stripe') {
        tipoReferenciaLogo =
          'https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png';
      }

      if (producto?.imagen) {
        imagen = buildFileUri(`productos/${producto.imagen}`);
      }

      return {
        ...venta,
        tipoReferenciaLogo,
        fechaVenta: formatDateTime(fechaVenta),
        importe: formatMoney(venta.importe),

        producto: {
          ...producto,
          imagen,
          precio: producto.precio ? formatMoney(producto.precio) : 'Gratis'
        },
        distribuidor: {
          comision: `${venta.distribuidor.comision}%`,
          importe: formatMoney(venta.distribuidor.importe)
        }
      };
    });

    return res.status(200).json({
      ventas: findVentas,
      totalComision: formatMoney(totalComision[0]?.totalComision || 0),
      importeTotal: formatMoney(importeTotal[0]?.importe || 0)
    });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

const getDistribuidoresSinPaginar = async (req, res) => {
  try {
    const results = await UsuariosModel.find({
      'tipoUsuarios.distribuidor': { $exists: true },
      estatus: { $in: ['Activo', 'Inactivo'] }
    }).select('_id nombre');

    const distribuidores = results.map((distribuidor) => {
      return {
        value: distribuidor._id,
        label: distribuidor.nombre
        // foto: cliente?.foto ? `${APP_URL}/uploads/clientes/${cliente.foto}` : buildFileUri('default/icono_usuario_100x100_04.jpg')
      };
    });

    return res.status(200).json(distribuidores);
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

const updateEstatus = async (req, res) => {
  const { estatus, _id } = req.body;

  try {
    const findDistribuidor = await UsuariosModel.findById(_id);

    if (!findDistribuidor) {
      return res.status(500).json({ message: 'No se encontró el distribuidor' });
    }

    findDistribuidor.estatus = estatus;
    await findDistribuidor.save();

    return res.status(200).json({ message: 'Estatus actualizado' });
  } catch (error) {
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};
module.exports = {
  getDistribuidoresAdmin,
  createDistribuidor,
  updateDistribuidor,
  getDistribuidorById,
  getVentasByDistribuidores,
  getDistribuidoresSinPaginar,
  updateEstatus
};
