const { generatePassword, encriptar, sendWhatsappRecovery2 } = require('../config/functionsGlobal.js');
const { sendMail } = require('../config/mail.js');
const { buildFileUri } = require('../config/s3.js');
const { RegistroUsuarioHTML } = require('../Mail/RegistroUsuarioHTML.js');
const TipoUsuarios = require('../models/TipoUsuarios.js');
const UsuariosModel = require('../models/Usuarios.js');
const moment = require('moment-timezone');
const DISTRIBUIDOR_APP = process.env.DISTRIBUIDOR_APP;
const mongoose = require('mongoose');
const VentaModel = require('../models/Ventas.js');
const { formatDateTime, formatMoney } = require('../config/functionsGlobal.js');

const getDistribuidores = async (req, res) => {
  const { estatus, page = 1, search } = req.query;
  const { distribuidor } = req.params;

  const options = {
    page,
    limit: 10,
    sort: {
      'licencia.vigencia': -1,
      nombre: 1
    },
    populate: [
      {
        path: 'licencia.producto',
        select: 'nombre imagen'
      }
    ]
  };

  const query = {
    'licencia.referidoPor': distribuidor,
    'tipoUsuarios.distribuidor': { $exists: true },
    estatus: { $in: ['Activo', 'Inactivo'] }
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
  }
  try {
    const data = await UsuariosModel.paginate(query, options);

    data.docs = data.docs.map((distribuidor) => {
      let diasRestantes = 0;

      if (distribuidor.licencia) {
        const licencia = distribuidor.licencia;
        const vigencia = licencia.vigencia;
        const fechaMexico = moment()
          .tz('America/Mexico_City')
          .format('YYYY-MM-DD HH:mm:ss');

        diasRestantes = moment(vigencia).diff(fechaMexico, 'days');
      }

      return {
        ...distribuidor._doc,
        diasRestantes: diasRestantes > 0 ? diasRestantes : 0,
        logo: distribuidor.foto
          ? buildFileUri(`usuarios/${distribuidor?.foto}`)
          : buildFileUri('default/icono_usuario_100x100_04.jpg'),
        imagenProducto: distribuidor.licencia?.producto?.imagen
          ? buildFileUri(`productos/${distribuidor.licencia.producto.imagen}`)
          : ''
      };
    });

    return res.status(200).json(data);
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los distribuidores', error });
  }
};

const createDistribuidor = async (req, res) => {
  const { distribuidor } = req.params;
  const { nombre, correo, telefono } = req.body;
  try {
    const existe = await UsuariosModel.findOne({ email: correo });
    if (existe) {
      return res.status(400).json({ message: 'El correo ya está en uso' });
    }

    const password = generatePassword();
    const passwordEncriptada = encriptar(password);

    const tipoUsuario = await TipoUsuarios.findOne({
      tipo: 'distribuidor'
    }).select('_id');

    const usuario = UsuariosModel({
      nombre,
      email: correo,
      telefono,
      password: passwordEncriptada,
      tipoUsuarios: {
        distribuidor: tipoUsuario._id
      },
      estatus: 'Activo',
      licencia: {
        referidoPor: distribuidor
      },
      fechaCreacion: new Date()
    });

    const result = await usuario.save();

    if (!result) {
      return res
        .status(400)
        .json({ message: 'Error al crear al distribuidor' });
    }

    sendWhatsappRecovery2({
      idDespacho: null,
      to: telefono,
      correo,
      password,
      url: DISTRIBUIDOR_APP + `/password/${result._id}`
    });

    const htmlRegistro = RegistroUsuarioHTML({
      nombre,
      usuario: correo,
      password,
      url: DISTRIBUIDOR_APP + `/password/${result._id}`
    });

    sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correo);

    return res.status(200).json({ message: 'Distribuidor creado' });
  } catch (error) {
    console.log(error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los distribuidores', error });
  }
};

const getVentasByDistribuidores = async (req, res) => {
  const { usuario, distribuidor } = req.params;
  const {
    producto,
    page = 1,
    fechaInicio = null,
    fechaFinal = null
  } = req.query;

  const limit = 10;
  const options = {
    page,
    limit,
    sort: {
      fechaVenta: -1
    },
    populate: [
      {
        path: 'producto',
        select: 'nombre imagen precio'
      }
    ],
    lean: true
  };

  // Base de coincidencias para la agregación
  const query = {
    usuario: mongoose.Types.ObjectId(usuario),
    'distribuidor.distribuidor': mongoose.Types.ObjectId(distribuidor),
    ...(producto ? { producto: mongoose.Types.ObjectId(producto) } : {}),
    ...(fechaInicio &&
      fechaFinal &&
      moment(fechaInicio).isValid() &&
      moment(fechaFinal).isValid()
      ? {
        fechaVenta: {
          $gte: moment(fechaInicio).startOf('day').toDate(),
          $lte: moment(fechaFinal).endOf('day').toDate()
        }
      }
      : {})
  };

  try {
    const [findVentas, totalComision] = await Promise.all([
      VentaModel.paginate(query, options),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            totalComision: { $sum: '$distribuidor.importe' }
          }
        }
      ])
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const { fechaVenta, producto } = venta;
      let imagen = '';

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = '';

      if (tipoReferencia === 'Paypal') {
        tipoReferenciaLogo =
          'https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png';
      } else if (tipoReferencia === 'Stripe') {
        tipoReferenciaLogo =
          'https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png';
      }

      if (producto?.imagen) {
        imagen = buildFileUri(`productos/${producto.imagen}`);
      }

      return {
        ...venta,
        tipoReferenciaLogo,
        fechaVenta: formatDateTime(fechaVenta),
        importe: formatMoney(venta.importe),
        producto: {
          ...producto,
          imagen,
          precio: producto.precio ? formatMoney(producto.precio) : 'Gratis'
        },
        distribuidor: {
          comision: `${venta.distribuidor.comision}%`,
          importe: formatMoney(venta.distribuidor.importe)
        }
      };
    });

    return res.status(200).json({
      ventas: findVentas,
      totalComision: formatMoney(totalComision[0]?.totalComision || 0)
    });

    // return res.status(200).json({
    //   ventas: {
    //     docs: ventasModificadas,
    //     totalDocs,
    //     limit,
    //     page
    //   },
    //   // totalImporte: formatMoney(totalImporte),
    //   totalComision: formatMoney(totalComision)
    // });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

module.exports = {
  getDistribuidores,
  createDistribuidor,
  getVentasByDistribuidores
};
