const DiarioOficialModel = require('../models/DiarioOficial');

const getDiariosOficiales = async (req, res) => {
  const { estado, page = 1, search, limit, estatus } = req.query;

  try {
    const options = {
      page,
      limit,
      sort: { estado: 1 },
      populate: [
        {
          path: 'estado'
        }
      ]
    };

    const query = {};

    if (search) {
      query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
    }

    if (estado) {
      query.estado = estado;
    }

    if (estatus) {
      query.estatus = estatus;
    }

    const results = await DiarioOficialModel.paginate(query, options);

    return res.status(200).json(results);
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

const getDiarioOficialById = async (req, res) => {
  const { id } = req.params;
  if (!id) {
    return res.status(400).json({ message: 'Es necesario el id' });
  }
  try {
    const result = await DiarioOficialModel.findById(id);

    if (!result) {
      return res.status(400).json({ message: 'No se encontró el resultado' });
    }
    return res.status(200).json(result);
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

const newDiarioOficialEstatal = async (req, res) => {
  const { estado = null, tipo, nombre, url } = req.body;

  try {
    if (estado) {
      const results = await DiarioOficialModel.find({ estado });
      if (results.length > 0) {
        return res.status(400).json({ message: 'El estado ya fue guardado' });
      }
    }

    const newDiario = new DiarioOficialModel({
      estado,
      tipo,
      nombre,
      url
    });

    const result = await newDiario.save();

    if (!result) {
      return res.status(400).json({ message: 'Error al guardar' });
    }

    return res.status(200).json({ message: 'Diario oficial guardado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

const updateDiarioOficial = async (req, res) => {
  const { id } = req.params;
  const { estado, tipo, nombre, url, estatus } = req.body;

  try {
    const existingDiario = await DiarioOficialModel.findOne({
      estado,
      _id: { $ne: id }
    });

    if (existingDiario) {
      return res
        .status(400)
        .json({ message: 'El estado ya está en uso por otro Diario Oficial' });
    }

    const updatedDiario = await DiarioOficialModel.findByIdAndUpdate(
      id,
      {
        estado,
        tipo,
        nombre,
        url,
        estatus
      },
      { new: true, runValidators: true }
    );

    if (!updatedDiario) {
      return res.status(404).json({ message: 'Diario oficial no encontrado' });
    }

    return res.status(200).json({ message: 'Diario oficial actualizado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

const deleteDiarioOficial = async (req, res) => {
  const { id } = req.params;
  try {
    await DiarioOficialModel.findByIdAndRemove(id);
    return res.status(200).json({ message: 'Diario oficial eliminado' });
  } catch (error) {
    console.log(error);
    return res.status(500).json({ message: error.message });
  }
};

module.exports = {
  newDiarioOficialEstatal,
  getDiarioOficialById,
  getDiariosOficiales,
  updateDiarioOficial,
  deleteDiarioOficial
};
