const DespachoModel = require("../models/Despachos.js");
const Modulos = require("../models/Modulos.js");
const Usuarios = require("../models/Usuarios.js");
const TipoUsuarioModel = require("../models/TipoUsuarios.js");
const PaqueteModel = require("../models/Productos.js");
const Ventas = require("../models/Ventas.js");
const { sendMail } = require("../config/mail.js");
const { RegistroUsuarioHTML } = require("../Mail/RegistroUsuarioHTML.js");
const {
  generatePassword,
  encriptar,
  sendWhatsappRecovery,
} = require("../config/functionsGlobal.js");
const moment = require("moment-timezone");
const { generalDefaults } = require("../controllers/Despacho.Controller.js");
const { buildFileUri } = require("../config/s3.js");

const getDespachos = async (req, res) => {
  const { estatus, page = 1, search, distribuidor } = req.query;

  const options = {
    page,
    limit: 10,
    sort: {
      estatus: 1,
      // _id: -1,
      nombre: 1,
    },
    populate: [
      {
        path: "estado",
      },
      {
        path: "distribuidor",
        select: "nombre apellidoPaterno apellidoMaterno foto",
      },
    ],
    lean: true,
  };

  const query = {};

  if (estatus) {
    query.estatus = estatus;
  }

  if (distribuidor) {
    query.$and = [...(query.$and || []), { distribuidor }];
  }

  if (search) {
    query.$and = [
      ...(query.$and || []),
      { nombre: { $regex: search, $options: "i" } },
    ];
  }

  try {
    const findDespachos = await DespachoModel.paginate(query, options);

    findDespachos.docs = findDespachos.docs.map((despacho) => {
      const { contadorExp } = despacho;
      const { vigencia } = contadorExp;

      const fechaMexico = moment()
        .tz("America/Mexico_City")
        .format("YYYY-MM-DD HH:mm:ss");

      const diasRestantes = moment(vigencia).diff(fechaMexico, "days");

      const { distribuidor } = despacho;

      if (distribuidor) {
        despacho.distribuidor = {
          ...distribuidor,
          foto: distribuidor.foto
            ? buildFileUri(`usuarios/${distribuidor.foto}`)
            : buildFileUri("default/icono_usuario_100x100_04.jpg"),
        };
      }

      return {
        ...despacho,
        logo: despacho.logo
          ? buildFileUri(`despachos/${despacho?._id}/logo/${despacho.logo}`)
          : buildFileUri("default/icono_usuario_100x100_04.jpg"),
        diasRestantes,
      };
    });

    return res.status(200).json({
      despachos: findDespachos,
    });
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al obtener los despachos", error });
  }
};

const createDespachoRegistro = async (req, res) => {
  const {
    nombreCompleto,
    nombreDespacho,
    correoElectronico,
    telefono,
    distribuidor,
  } = req.body;

  if (!nombreCompleto) {
    return res.status(400).json({ message: "El nombre completo es requerido" });
  }

  if (!nombreDespacho) {
    return res
      .status(400)
      .json({ message: "El nombre del despacho es requerido" });
  }

  if (!correoElectronico) {
    return res
      .status(400)
      .json({ message: "El correo electrónico es requerido" });
  }

  if (!telefono) {
    return res.status(400).json({ message: "El teléfono es requerido" });
  }

  // if (!estado) return res.status(400).json({ message: 'El estado es requerido' });

  // if (!municipio) return res.status(400).json({ message: 'El municipio es requerido' });

  const findUsuario = await Usuarios.findOne({ email: correoElectronico });

  if (findUsuario) {
    return res
      .status(400)
      .json({ message: "Ya existe un usuario con ese correo electrónico" });
  }

  try {
    const despachoCreate = await DespachoModel.create({
      nombre: nombreDespacho,
      razonSocial: nombreCompleto,
      correo: correoElectronico,
      telefono,
      distribuidor: distribuidor || null,
    });

    const findModulos = await Modulos.find({
      tipo: "despacho",
      estatus: "Activo",
    }).select("_id");
    const paqueteFree = await PaqueteModel.findOne({ estatus: "Exclusivo" });

    const tipoUsuarioCreate = await TipoUsuarioModel.create({
      nombre: "ADMINISTRADOR",
      tipo: "despacho",
      despacho: despachoCreate._id,
      modulos: findModulos.map((modulo) => ({
        modulo: modulo._id,
        permisos: {
          create: true,
          read: true,
          update: true,
          delete: true,
          download: true,
        },
      })),
    });

    const password = generatePassword();
    const passwordEncripted = encriptar(password);

    const usuarioCreate = await Usuarios.create({
      nombre: nombreCompleto,
      email: correoElectronico,
      telefono,
      tipoUsuarios: {
        despacho: tipoUsuarioCreate._id,
      },
      despacho: despachoCreate._id,
      password: passwordEncripted,
    });

    // creadoPor
    await DespachoModel.findByIdAndUpdate(despachoCreate._id, {
      creadoPor: usuarioCreate._id,
    });

    const url = process.env.DESPACHO_APP || "https://despacho.nildda.com/login";

    sendWhatsappRecovery({
      to: telefono,
      correo: correoElectronico,
      password,
      url,
    });

    // sendWhatsappRecovery({
    //   to: '2213425514',
    //   correo: correoElectronico,
    //   password,
    //   url
    // });

    sendMail(
      RegistroUsuarioHTML(nombreCompleto, correoElectronico, password),
      "Recordatorio de contraseña",
      correoElectronico
    );

    // agregar sus materias

    const despachoId = despachoCreate._id;

    const fechaMexico = moment()
      .tz("America/Mexico_City")
      .format("YYYY-MM-DD HH:mm:ss");

    const cantidad = paqueteFree.cantidad || 10;
    const tiempo = paqueteFree.tiempo || 30;

    const fechaVigencia = moment(fechaMexico)
      .add(tiempo, "days")
      .format("YYYY-MM-DD HH:mm:ss");

    // generar la venta gratis

    Ventas.create({
      despacho: despachoCreate._id,
      usuario: usuarioCreate._id,
      producto: paqueteFree._id,
      importe: paqueteFree.precio,
      referencia: "",
      tipoReferencia: "Gratis",
      estatus: "Pagado",
    });

    await DespachoModel.findByIdAndUpdate(despachoId, {
      contadorExp: {
        contador: 0,
        limite: cantidad,
        vigencia: fechaVigencia,
      },
    });

    generalDefaults({ idDespacho: despachoId });

    const htmlRegistro = RegistroUsuarioHTML(
      nombreCompleto,
      correoElectronico,
      password,
      url
    );
    sendMail(htmlRegistro, "Bienvenido a NILDDA", correoElectronico);

    res.status(200).json({
      despacho: despachoCreate,
      usuario: usuarioCreate,
      message:
        "Gracias por registrarte en NILDDA, se ha enviado un correo con tus credenciales de acceso",
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getDespachoById = async (req, res) => {
  const { id } = req.params;
  try {
    const despacho = await DespachoModel.findById(id).populate("estado").lean();

    if (!despacho) {
      return res.status(404).json({ message: "Despacho no encontrado" });
    }

    const { contadorExp } = despacho;
    const { vigencia } = contadorExp;

    const diasRestantes = vigencia
      ? Math.ceil((vigencia - new Date()) / (1000 * 60 * 60 * 24))
      : 0;

    const despachoData = {
      ...despacho,
      logo: despacho.logo
        ? buildFileUri(`despachos/${despacho?._id}/logo/${despacho.logo}`)
        : "",
      diasRestantes: diasRestantes > 0 ? diasRestantes : 0,
    };

    return res.status(200).json(despachoData);
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al obtener el despacho", error });
  }
};

const updateDespacho = async (req, res) => {
  const { id } = req.params;
  const { estatus, distribuidor, vigencia } = req.body;

  if (!estatus) {
    return res.status(400).json({ message: "Es obligatorio el estatus" });
  }
  try {
    const nuevoDespacho = {
      estatus,
      distribuidor: distribuidor || null,
      ...(vigencia && { "contadorExp.vigencia": vigencia }),
    };

    await DespachoModel.findByIdAndUpdate(id, nuevoDespacho);

    return res.status(200).json({ message: "Despacho actualizado" });
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al obtener el despacho", error });
  }
};
const deleteDespacho = async (req, res) => {
  const { id } = req.params;
  try {
    await DespachoModel.findByIdAndDelete(id);

    // elimianr usuarios del despacho
    await Usuarios.deleteMany({ despacho: id });

    // eliminar tipo de usuario
    await TipoUsuarioModel.deleteMany({ despacho: id });

    // eliminar ventas

    await Ventas.deleteMany({ despacho: id });

    return res.status(200).json({ message: "Despacho eliminado" });
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al obtener el despacho", error });
  }
};

const updateEstatus = async (req, res) => {
  const { id, estatus } = req.body;
  try {
    const findDespachos = await DespachoModel.findById(id);

    if (!findDespachos) {
      return res.status(500).json({ message: "Despacho no encontrado" });
    }

    findDespachos.estatus = estatus;

    await findDespachos.save();
    return res.status(200).json({ message: "Estatus actualizado" });
  } catch (error) {
    console.error("Error:", error);
    return res
      .status(500)
      .json({ message: "Error al actualizar el estatus", error });
  }
};

module.exports = {
  getDespachos,
  createDespachoRegistro,
  getDespachoById,
  deleteDespacho,
  updateDespacho,
  updateEstatus,
};
