/* eslint-disable indent */
const DespachoModel = require('../models/Despachos.js');
const { buildFileUri } = require('../config/s3.js');
const { formatDateTime, formatMoney, sendWhatsappRecovery2 } = require('../config/functionsGlobal.js');
const moment = require('moment-timezone');
const { default: mongoose } = require('mongoose');
const PaqueteModel = require('../models/Productos.js');
const { generalDefaults } = require('../controllers/Despacho.Controller.js');
const UsuarioModel = require('../models/Usuarios.js');
const { sendMail } = require('../config/mail.js');
const { RegistroUsuarioHTML } = require('../Mail/RegistroUsuarioHTML.js');
const VentaModel = require('../models/Ventas.js');
const { DESPACHO_APP } = require('../config/index.js');

const getDespachos = async (req, res) => {
  const { estatus, page = 1, search } = req.query;
  const { distribuidor } = req.params;

  const options = {
    page,
    limit: 10,
    sort: {
      'contadorExp.vigencia': -1,
      nombre: 1
    },
    populate: [
      {
        path: 'estado',
        select: 'nombre'
      },
      {
        path: 'contadorExp.paquete',
        select: 'nombre imagen'
      }
    ]
  };

  const query = {
    distribuidor
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    query.$or = [{ nombre: { $regex: search, $options: 'i' } }];
  }

  try {
    const findDespachos = await DespachoModel.paginate(query, options);

    findDespachos.docs = findDespachos.docs.map((despacho) => {
      const { contadorExp } = despacho;
      const { vigencia } = contadorExp;
      const fechaMexico = moment()
        .tz('America/Mexico_City')
        .format('YYYY-MM-DD HH:mm:ss');

      const diasRestantes = moment(vigencia).diff(fechaMexico, 'days');

      return {
        ...despacho._doc,
        diasRestantes,
        logo: despacho.logo
          ? buildFileUri(`despachos/${despacho?._id}/logo/${despacho.logo}`)
          : buildFileUri('default/icono_usuario_100x100_04.jpg'),
        planActivo: despacho?.contadorExp?.paquete?.nombre || '',
        planActivoImagen: despacho?.contadorExp?.paquete?.imagen
          ? buildFileUri(`productos/${despacho.contadorExp.paquete.imagen}`)
          : ''
      };
    });

    return res.status(200).json({
      despachos: findDespachos
    });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los despachos', error });
  }
};

const getDespachosSinPaginar = async (req, res) => {
  const { distribuidor } = req.params;

  try {
    const despachos = await DespachoModel.find({ distribuidor });

    const despachosSinPaginar = despachos.map((despacho) => {
      return {
        value: despacho._id,
        label: despacho.nombre
      };
    });

    return res.status(200).json(despachosSinPaginar);
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los despachos', error });
  }
};

const getVentasByDespacho = async (req, res) => {
  const { despacho, distribuidor } = req.params;
  const {
    producto,
    page = 1,
    fechaInicio = null,
    fechaFinal = null
  } = req.query;

  const limit = 10;
  const options = {
    page,
    limit,
    sort: {
      fechaVenta: -1
    },
    populate: [
      {
        path: 'producto',
        select: 'nombre imagen precio'
      }
    ],
    lean: true
  };

  // Base de coincidencias para la agregación
  const query = {
    despacho: mongoose.Types.ObjectId(despacho),
    'distribuidor.distribuidor': mongoose.Types.ObjectId(distribuidor),
    ...(producto ? { producto: mongoose.Types.ObjectId(producto) } : {}),
    ...(fechaInicio &&
      fechaFinal &&
      moment(fechaInicio).isValid() &&
      moment(fechaFinal).isValid()
      ? {
        fechaVenta: {
          $gte: moment(fechaInicio).startOf('day').toDate(),
          $lte: moment(fechaFinal).endOf('day').toDate()
        }
      }
      : {})
  };

  try {
    const [findVentas, totalComision] = await Promise.all([
      VentaModel.paginate(query, options),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            totalComision: { $sum: '$distribuidor.importe' }
          }
        }
      ])
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const { fechaVenta, producto } = venta;
      let imagen = '';

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = '';

      if (tipoReferencia === 'Paypal') {
        tipoReferenciaLogo =
          'https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png';
      } else if (tipoReferencia === 'Stripe') {
        tipoReferenciaLogo =
          'https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png';
      }

      if (producto?.imagen) {
        imagen = buildFileUri(`productos/${producto.imagen}`);
      }

      return {
        ...venta,
        tipoReferenciaLogo,
        fechaVenta: formatDateTime(fechaVenta),
        importe: formatMoney(venta.importe),
        producto: {
          ...producto,
          imagen,
          precio: producto.precio ? formatMoney(producto.precio) : 'Gratis'
        },
        distribuidor: {
          comision: `${venta.distribuidor.comision}%`,
          importe: formatMoney(venta.distribuidor.importe)
        }
      };
    });

    return res.status(200).json({
      ventas: findVentas,
      totalComision: formatMoney(totalComision[0]?.totalComision || 0)
    });

    // return res.status(200).json({
    //   ventas: {
    //     docs: ventasModificadas,
    //     totalDocs,
    //     limit,
    //     page
    //   },
    //   // totalImporte: formatMoney(totalImporte),
    //   totalComision: formatMoney(totalComision)
    // });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

const getDespachosSinPaginarVencidos = async (req, res) => {
  const { distribuidor } = req.params;

  try {
    const despachos = await DespachoModel.find({ distribuidor }).select(
      'contadorExp nombre'
    );

    const despachosSinPaginar = despachos.filter((despacho) => {
      const contadorExp = despacho?.contadorExp || null;
      const vigencia = contadorExp?.vigencia || null;

      const fechaMexico = moment()
        .tz('America/Mexico_City')
        .format('YYYY-MM-DD HH:mm:ss');

      const diasRestantes = moment(vigencia).diff(fechaMexico, 'days');

      return diasRestantes <= 0;
    });

    const despachosSinPaginarModificados = despachosSinPaginar.map(
      (despacho) => {
        return {
          value: despacho._id,
          label: despacho.nombre
        };
      }
    );

    return res.status(200).json(despachosSinPaginarModificados);
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener los despachos', error });
  }
};

const createDespachoDistribuidor = async (req, res) => {
  const { distribuidor } = req.params;
  const { nombreDespacho, nombre, correo, telefono } = req.body;

  if (!nombreDespacho) {
    return res
      .status(400)
      .json({ message: 'El nombre del despacho es requerido' });
  }

  if (!nombre) {
    return res
      .status(400)
      .json({ message: 'El nombre del usuario es requerido' });
  }

  if (!correo) {
    return res
      .status(400)
      .json({ message: 'El correo del usuario es requerido' });
  }

  if (!telefono) {
    return res
      .status(400)
      .json({ message: 'El teléfono del usuario es requerido' });
  }

  try {
    const findUsuarioCorreo = await UsuarioModel.find({ email: correo });

    if (findUsuarioCorreo.length > 0) {
      return res.status(400).json({ message: 'El correo ya está registrado' });
    }

    const despachoCreate = new DespachoModel({
      distribuidor,
      nombre: nombreDespacho,
      telefono,
      correo
    });

    await despachoCreate.save();

    const paqueteFree = await PaqueteModel.findOne({ estatus: 'Exclusivo' });

    const usuarioCreate = await UsuarioModel.create({
      nombre,
      email: correo,
      telefono,
      despacho: despachoCreate._id
    });

    // creadoPor
    await DespachoModel.findByIdAndUpdate(despachoCreate._id, {
      creadoPor: usuarioCreate._id
    });

    sendWhatsappRecovery2({
      idDespacho: usuarioCreate.despacho,
      to: telefono,
      nombreUsuario: nombre,
      correo,
      url: DESPACHO_APP + `/password/${usuarioCreate._id}`
    });

    sendMail(
      RegistroUsuarioHTML(nombre, correo, '', DESPACHO_APP + `/password/${usuarioCreate._id}`),
      'NILDDA: Envió Accesos',
      correo
    );

    // agregar sus materias

    const despachoId = despachoCreate._id;

    const fechaMexico = moment()
      .tz('America/Mexico_City')
      .format('YYYY-MM-DD HH:mm:ss');

    const cantidad = paqueteFree?.cantidad || 0;
    const tiempo = paqueteFree?.tiempo || 0;

    // generar la venta gratis

    await VentaModel.create({
      despacho: despachoId,
      usuario: usuarioCreate._id,
      producto: paqueteFree._id,
      importe: paqueteFree.precio,
      referencia: '',
      tipoReferencia: 'Gratis',
      estatus: 'Pagado'
    });

    const fechaVigencia = moment(fechaMexico)
      .add(tiempo, 'days')
      .format('YYYY-MM-DD HH:mm:ss');

    await DespachoModel.findByIdAndUpdate(despachoId, {
      contadorExp: {
        contador: 0,
        limite: cantidad,
        vigencia: fechaVigencia,
        paquete: paqueteFree._id
      }
    });

    generalDefaults({ idDespacho: despachoId });

    return res.status(200).json({ message: 'Despacho creado correctamente' });

    // copy
  } catch (error) {
    console.log('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al crear el despacho', error });
  }
};

module.exports = {
  getDespachos,
  getVentasByDespacho,
  getDespachosSinPaginar,
  getDespachosSinPaginarVencidos,
  createDespachoDistribuidor
};
