const Despacho = require('../models/Despachos.js');
const UsuarioModel = require('../models/Usuarios.js');
const TipoUsuarioModel = require('../models/TipoUsuarios.js');
const ModulosModel = require('../models/Modulos.js');
const { generatePassword, encriptar, desencriptar, sendWhatsappRecovery } = require('../config/functionsGlobal.js');
const { sendMail } = require('../config/mail.js');
const { RegistroUsuarioHTML } = require('../Mail/RegistroUsuarioHTML.js');
const MateriasModel = require('../models/Materias.js');
const MateriasDespachoModel = require('../models/MateriasDespacho.js');
const ProductoModel = require('../models/Productos.js');
const moment = require('moment-timezone');
const EtapasProcesalesModel = require('../models/EtapasProcesales.js');
const EtapasProcesalesDespachoModel = require('../models/EtapasProcesalesDespachos.js');
const { CompraHTML } = require('../Mail/CompraHTML.js');
const { buildFileUri, deleteFileS3, uploadFile } = require('../config/s3.js');
const VentasModel = require('../models/Ventas.js');
const PaqueteModel = require('../models/Productos.js');
const RecursosIncidenciasModel = require('../models/RecursosIncidencias.js');
const RecursosDespachoModel = require('../models/RecursosDespacho.js');
const IncidenciasDespachoModel = require('../models/IncidenciasDespacho.js');
const DESPACHO_APP = process.env.DESPACHO_APP;

const ConfiguracionesDespacho = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) return res.status(404).json({ message: 'Despacho no encontrado' });

  try {
    const findDespacho = await Despacho.findById(despacho);

    if (!findDespacho) return res.status(404).json({ message: 'Despacho no encontrado' });

    if (findDespacho.logo) {
      findDespacho.logo = buildFileUri(`despachos/${despacho}/logo/${findDespacho.logo}`);
    }

    res.status(200).json({ despacho: findDespacho });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const actualizarDespacho = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, correo, direccion, telefono, razonSocial, rfc, cRegimenFiscal, lugarExpedicion, serie, numeroCertificado, clavePrivada, estado, municipio } = req.body;

  if (!despacho) return res.status(404).json({ message: 'Despacho no encontrado' });

  const logo = req.files?.logo ?? null;
  const certificado = req.files?.certificado ?? null;
  const llave = req.files?.llave ?? null;

  try {
    if (!nombre) {
      return res.status(400).json({ message: 'El nombre del despacho es requerido' });
    }

    if (!correo) {
      return res.status(400).json({ message: 'El correo electrónico es requerido' });
    }

    if (!telefono) {
      return res.status(400).json({ message: 'El teléfono es requerido' });
    }
    const findDespacho = await Despacho.findById(despacho);

    if (!findDespacho) return res.status(404).json({ message: 'Despacho no encontrado' });

    const updateDespacho = await Despacho.findByIdAndUpdate(despacho, {
      nombre,
      correo,
      direccion: direccion ?? '',
      telefono,
      razonSocial: razonSocial ?? '',
      rfc: rfc ?? '',
      cRegimenFiscal: cRegimenFiscal ?? '',
      lugarExpedicion: lugarExpedicion ?? '',
      serie: serie ?? '',
      numeroCertificado: numeroCertificado ?? '',
      clavePrivada: clavePrivada ?? '',
      estado,
      municipio
    }, { new: true });

    if (logo !== null || certificado !== null || llave !== null) {
      if (logo !== null) {
        // Delete previous logo
        if (findDespacho.logo) {
          deleteFileS3(`despachos/${despacho}/logo/${findDespacho.logo}`);
        }
        // updateDespacho.logo = logo[0].filename;
        console.log(logo);
        updateDespacho.logo = await uploadFile(logo, `despachos/${despacho}/logo`).then(data => data.fileName);
      }

      if (certificado !== null) {
        updateDespacho.certificado = certificado[0].filename;
      }

      if (llave !== null) {
        updateDespacho.llave = llave[0].filename;
      }

      await updateDespacho.save();
    }

    if (updateDespacho.logo) {
      updateDespacho.logo = buildFileUri(`despachos/${despacho}/logo/${updateDespacho.logo}`);
    }

    res.status(200).json({
      despacho: updateDespacho,
      message: 'Despacho actualizado con éxito'
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

// const planActual = async (req, res) => {
//   const { despacho } = req.params;

//   if (!despacho) return res.status(404).json({ message: 'Despacho no encontrado' });

//   try {
//     const findDespacho = await Despacho.findById(despacho);

//     if (!findDespacho) return res.status(404).json({ message: 'Despacho no encontrado' });

//     const { vigencia } = findDespacho;

//     const fechaActual = new Date();

//     const alerta = {
//       planActual: '',
//       diasRestantes: 0,
//       fechaVigencia: ''
//     };
//   } catch (error) {
//     res.status(404).json({ message: error.message });
//   }
// };

const createDespacho = async (req, res) => {
  const {
    nombreCompleto, // razon social
    nombreDespacho,
    correoElectronico,
    telefono,
    producto, // plan
    tipoReferencia,
    referencia,
    distribuidor = null
  } = req.body;

  if (!nombreCompleto || !nombreDespacho || !correoElectronico || !telefono) {
    return res.status(400).json({ message: 'Todos los campos son requeridos' });
  }

  const findUsuario = await UsuarioModel.findOne({ email: correoElectronico });

  if (findUsuario) {
    return res.status(400).json({ message: 'Ya existe un usuario con ese correo electrónico' });
  }

  try {
    const despachoCreate = await Despacho.create({
      nombre: nombreDespacho,
      razonSocial: nombreCompleto,
      rfc: '',
      correo: correoElectronico,
      telefono,
      distribuidor: distribuidor || null
    });

    // const findModulos = await ModulosModel.find({ tipo: 'despacho', estatus: 'Activo' }).select('_id');

    // const tipoUsuarioCreate = await TipoUsuarioModel.create({
    //   nombre: 'ADMINISTRADOR',
    //   tipo: 'despacho',
    //   despacho: despachoCreate._id,
    //   modulos: findModulos.map(modulo => ({
    //     modulo: modulo._id,
    //     permisos: {
    //       create: true,
    //       read: true,
    //       update: true,
    //       delete: true,
    //       download: true
    //     }
    //   }))
    // });

    const password = generatePassword();
    const passwordEncripted = encriptar(password);

    const usuarioCreate = await UsuarioModel.create({
      nombre: nombreCompleto,
      email: correoElectronico,
      telefono,
      despacho: despachoCreate._id,
      password: passwordEncripted
    });

    // creadoPor
    await Despacho.findByIdAndUpdate(despachoCreate._id, { creadoPor: usuarioCreate._id });


    sendWhatsappRecovery({
      to: telefono,
      correo: correoElectronico,
      password,
      url: DESPACHO_APP
    });

    sendMail(RegistroUsuarioHTML(nombreCompleto, correoElectronico, password), 'NILDDA: Envió Accesos', correoElectronico);

    // agregar sus materias

    const despachoId = despachoCreate._id;

    if (producto) {
      const findProducto = await ProductoModel.findById(producto);

      if (findProducto) {
        sendMail(CompraHTML({
          nombre: nombreCompleto,
          nombreProducto: findProducto.nombre,
          descripcionProducto: findProducto.descripcion,
          precioProducto: findProducto.precio
        }), 'Compra de producto', correoElectronico);

        const fechaMexico = moment().tz('America/Mexico_City').format('YYYY-MM-DD HH:mm:ss');

        const { cantidad, precio, tiempo } = findProducto;

        const fechaVigencia = moment(fechaMexico).add(tiempo, 'days').format('YYYY-MM-DD HH:mm:ss');

        await Despacho.findByIdAndUpdate(despachoId,
          {
            contadorExp: {
              contador: 0,
              limite: cantidad,
              vigencia: fechaVigencia,
              paquete: producto
            }
          });

        let distribuidorVenta = null;

        if (distribuidor) {
          const fechaMexico = moment().tz('America/Mexico_City');

          const findDistribuidor = await UsuarioModel.findOne({
            _id: distribuidor,
            'licencia.producto': { $ne: null },
            estatus: 'Activo'
          });

          if (findDistribuidor && findDistribuidor.licencia.vigencia >= fechaMexico.toDate()) {
            distribuidorVenta = {
              comision: findDistribuidor.licencia.comision,
              distribuidor,
              importe: precio * (findDistribuidor.licencia.comision / 100)
            };
          }
        }

        await VentasModel.create({
          despacho: despachoId,
          usuario: usuarioCreate._id,
          producto,
          importe: precio,
          estatus: 'Pagado',
          tipoReferencia,
          referencia,
          distribuidor: distribuidorVenta
        });
      }
    }
    generalDefaults({ idDespacho: despachoId });

    const htmlRegistro = RegistroUsuarioHTML(nombreCompleto, correoElectronico, password, DESPACHO_APP);
    sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correoElectronico);

    res.status(200).json({
      despacho: despachoCreate,
      usuario: usuarioCreate,
      message: 'Gracias por registrarte en NILDDA, se ha enviado un correo con tus credenciales de acceso'
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createDespachoRegistro = async (req, res) => {
  const {
    nombreCompleto, // razon social
    nombreDespacho,
    correoElectronico,
    telefono,
    claveDistribuidor = null,
    googleId = null
  } = req.body;

  if (!nombreCompleto) return res.status(400).json({ message: 'El nombre completo es requerido' });

  if (!nombreDespacho) return res.status(400).json({ message: 'El nombre del despacho es requerido' });

  if (!correoElectronico) return res.status(400).json({ message: 'El correo electrónico es requerido' });

  // if (!telefono) return res.status(400).json({ message: 'El teléfono es requerido' });

  const findUsuario = await UsuarioModel.findOne({ email: correoElectronico });

  if (findUsuario) {
    return res.status(400).json({ message: 'Ya existe un usuario con ese correo electrónico' });
  }

  let findDistribuidor = null;

  if (claveDistribuidor) {
    findDistribuidor = await UsuarioModel.findOne({ clave: claveDistribuidor, estatus: 'Activo' }).select('_id');
  }
  try {
    const despachoCreate = await Despacho.create({
      nombre: nombreDespacho,
      razonSocial: nombreCompleto,
      correo: correoElectronico,
      telefono,
      distribuidor: findDistribuidor ? findDistribuidor._id : null
    });

    const findModulos = await ModulosModel.find({ tipo: 'despacho', estatus: 'Activo' }).select('_id');
    const paqueteFree = await PaqueteModel.findOne({ estatus: 'Exclusivo' });

    const tipoUsuarioCreate = await TipoUsuarioModel.create({
      nombre: 'ADMINISTRADOR',
      tipo: 'despacho',
      despacho: despachoCreate._id,
      modulos: findModulos.map(modulo => ({
        modulo: modulo._id,
        permisos: {
          create: true,
          read: true,
          update: true,
          delete: true,
          download: true
        }
      }))
    });

    const password = generatePassword();
    const passwordEncripted = encriptar(password);

    const usuarioCreate = await UsuarioModel.create({
      nombre: nombreCompleto,
      email: correoElectronico,
      telefono,
      tipoUsuarios: {
        despacho: tipoUsuarioCreate._id
      },
      despacho: despachoCreate._id,
      password: passwordEncripted,
      googleId
    });

    // creadoPor
    await Despacho.findByIdAndUpdate(despachoCreate._id, { creadoPor: usuarioCreate._id });

    if (telefono) {
      sendWhatsappRecovery({
        to: telefono,
        correo: correoElectronico,
        password,
        url: DESPACHO_APP
      });
    }

    sendMail(RegistroUsuarioHTML(nombreCompleto, correoElectronico, password), 'NILDDA: Envió Accesos', correoElectronico);

    // agregar sus materias

    const despachoId = despachoCreate._id;

    const fechaMexico = moment().tz('America/Mexico_City').format('YYYY-MM-DD HH:mm:ss');

    const cantidad = paqueteFree.cantidad || 10;
    const tiempo = paqueteFree.tiempo || 30;

    const fechaVigencia = moment(fechaMexico).add(tiempo, 'days').format('YYYY-MM-DD HH:mm:ss');

    // generar la venta gratis

    VentasModel.create({
      despacho: despachoCreate._id,
      usuario: usuarioCreate._id,
      producto: paqueteFree._id,
      importe: paqueteFree.precio,
      referencia: '',
      tipoReferencia: 'Gratis',
      estatus: 'Pagado'
    });

    await Despacho.findByIdAndUpdate(despachoId,
      {
        contadorExp: {
          contador: 0,
          limite: cantidad,
          vigencia: fechaVigencia
        }
      });

    generalDefaults({ idDespacho: despachoId });

    const htmlRegistro = RegistroUsuarioHTML(nombreCompleto, correoElectronico, password, DESPACHO_APP);
    sendMail(htmlRegistro, 'NILDDA: Envió Accesos', correoElectronico);

    res.status(200).json({
      despacho: despachoCreate,
      usuario: usuarioCreate,
      message: 'Gracias por registrarte en NILDDA, se ha enviado un correo con tus credenciales de acceso'
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const generalDefaults = async ({ idDespacho }) => {
  try {
    const [materias, etapasProcesales, recursos, incidencia] = await Promise.all([
      MateriasModel.find({ estatus: 'Activo' }),
      EtapasProcesalesModel.find({ estatus: 'Activo' }),
      RecursosIncidenciasModel.find({ estatus: 'Activo', tipo: 'Recurso' }),
      RecursosIncidenciasModel.find({ estatus: 'Activo', tipo: 'Incidencia' })
    ]);

    const materiasDespachosArray = await Promise.all(materias.map(async materia => {
      const materiaDespacho = await MateriasDespachoModel.create({
        despacho: idDespacho,
        materia: materia._id,
        nombre: materia.nombre
      });
      return materiaDespacho;
    }));

    etapasProcesales.forEach(async etapaProcesal => {
      const findMateria = await MateriasDespachoModel.findOne({ despacho: idDespacho, materia: etapaProcesal.materia }).select('_id');

      if (!findMateria) {
        return;
      }

      const newObjEtapa = {
        despacho: idDespacho,
        nombre: etapaProcesal.nombre,
        tipo: etapaProcesal.tipo,
        materia: findMateria._id
      };
      await EtapasProcesalesDespachoModel.create(newObjEtapa);
    });

    recursos.forEach(async recurso => {
      const findMateria = await MateriasDespachoModel.findOne({ despacho: idDespacho, materia: recurso.materia }).select('_id');

      if (!findMateria) {
        return;
      }

      const newObjRecurso = {
        despacho: idDespacho,
        nombre: recurso.nombre,
        tipo: recurso.tipo,
        materia: findMateria._id
      };
      await RecursosDespachoModel.create(newObjRecurso);
    });

    incidencia.forEach(async incidencia => {
      const findMateria = await MateriasDespachoModel.findOne({ despacho: idDespacho, materia: incidencia.materia }).select('_id');

      if (!findMateria) {
        return;
      }

      const newObjIncidencia = {
        despacho: idDespacho,
        nombre: incidencia.nombre,
        tipo: incidencia.tipo,
        materia: findMateria._id
      };

      await IncidenciasDespachoModel.create(newObjIncidencia);
    });

    // generar venta

    return {
      materiasDespachosArray,
      etapasProcesales
    };
  } catch (error) {
    console.log(error);
    return { error: error.message };
  }
};

const createDespachoDistribuidor = async (req, res) => {
  const { nombreDespacho, distribuidor } = req.body;

  if (!nombreDespacho) {
    return res.status(400).json({ message: 'El nombre del despacho es requerido' });
  }

  try {
    const findDistribuidor = await UsuarioModel.findById(distribuidor);

    if (!findDistribuidor) {
      return res.status(404).json({ message: 'Distribuidor no encontrado' });
    }

    const fechaMexico = moment().tz('America/Mexico_City');
    const findPaquete = await PaqueteModel.findOne({ estatus: 'Exclusivo' });
    const fechaVigencia = fechaMexico.add(findPaquete.tiempo, 'days').format('YYYY-MM-DD HH:mm:ss');

    const despachoCreate = await Despacho.create({
      nombre: nombreDespacho,
      razonSocial: findDistribuidor.nombre,
      rfc: '',
      correo: findDistribuidor.email,
      telefono: findDistribuidor.telefono,
      distribuidor,
      creadoPor: findDistribuidor._id,
      contadorExp: {
        contador: 0,
        limite: findPaquete.cantidad,
        vigencia: fechaVigencia,
        paquete: findPaquete._id
      }
    });

    findDistribuidor.despacho = despachoCreate._id;
    findDistribuidor.tipoUsuarios.despacho = despachoCreate._id;

    await findDistribuidor.save();

    generalDefaults({ idDespacho: despachoCreate._id });

    // Enviar acceso al distribuidor
    const password = desencriptar(findDistribuidor.password);

    sendWhatsappRecovery({
      to: findDistribuidor.telefono,
      correo: findDistribuidor.email,
      password,
      url: DESPACHO_APP
    });

    sendMail(RegistroUsuarioHTML(findDistribuidor.nombre, findDistribuidor.email, password, DESPACHO_APP), 'NILDDA: Envió Accesos', findDistribuidor.email);

    res.status(200).json({
      despacho: despachoCreate,
      message: 'Despacho creado con éxito'
    });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  ConfiguracionesDespacho,
  actualizarDespacho,
  createDespacho,
  generalDefaults,
  createDespachoDistribuidor,
  createDespachoRegistro
};
