const DependenciasModel = require('../models/Dependencias.js');

const getDependencias = async (req, res) => {
  const { page = 1, search, estado, estatus } = req.query;
  const options = {
    page,
    limit: 20,
    sort: { estado: 1, nombre: 1 },
    populate: 'estado'
  };

  const query = {};

  if (search) {
    query.$or = [
      { nombre: { $regex: search, $options: 'i' } }
    ];
  }

  if (estado) {
    query.estado = estado;
  }

  if (estado) {
    query.estado = estado;
  }

  if (estatus) {
    query.estatus = estatus;
  }

  try {
    const findDependencias = await DependenciasModel.paginate(query, options);
    res.status(200).json({ dependencias: findDependencias });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createDependencia = async (req, res) => {
  const { nombre, direccion, telefonos, estado, paginaWeb } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }

  try {
    const newTelefonos = telefonos.length > 0
      ? telefonos.split(',').map(tel => tel.trim())
      : [];

    const newLegislacion = new DependenciasModel({
      nombre,
      direccion,
      telefonos: newTelefonos,
      estado,
      liga: paginaWeb
    });

    await newLegislacion.save();
    res.status(201).json({ message: 'Dependencia creada' });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const updateDependencia = async (req, res) => {
  const { id } = req.params;
  const { nombre, direccion, telefonos, estado, paginaWeb, estatus } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!estado) {
    return res.status(400).json({ message: 'El estado es requerido' });
  }

  const newTelefonos = telefonos.length > 0
    ? telefonos.split(',').map(tel => tel.trim())
    : [];

  try {
    const updatedFiscalia = await DependenciasModel.findByIdAndUpdate(id, {
      nombre,
      estado,
      liga: paginaWeb,
      estatus: estatus || 'Activo',
      direccion,
      telefonos: newTelefonos

    }, { new: true });

    res.status(200).json({ message: 'Dependencia actualizada correctamente', fiscalia: updatedFiscalia });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getDependenciaById = async (req, res) => {
  const { id } = req.params;
  try {
    const fiscalia = await DependenciasModel.findById(id);

    return res.status(200).json(fiscalia);
  } catch (error) {
    return res.status(404).json({ message: 'Dependencia no encontrada' });
  }
};

const deleteDependencia = async (req, res) => {
  const { id } = req.params;

  try {
    await DependenciasModel.findByIdAndDelete(id);
    res.status(200).json({ message: 'Dependencia eliminada' });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getDependencias,
  createDependencia,
  updateDependencia,
  getDependenciaById,
  deleteDependencia
};
