const DeclaracionesFiscales = require("../models/DeclaracionesFiscales.js");
const { buildFileUri, deleteFileS3, uploadFile } = require("../config/s3.js");

const getDeclaracionFiscal = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search, year } = req.query;

  const options = {
    page,
    limit: 20,
    sort: { fecha: -1 },
    populate: [
      {
        path: "creadoPor",
        select: "nombre apellidoPaterno apellidoMaterno",
      },
    ],
  };

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  const query = {
    despacho,
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    // query.nombre = { $regex: search, $options: 'i' };
    query.$or = [{ nombre: { $regex: search, $options: "i" } }];
  }

  if (year) {
    const start = new Date(`${year}-01-01T00:00:00.000Z`);
    const end = new Date(`${year}-12-31T23:59:59.999Z`);
    query.fecha = { $gte: start, $lte: end };
  }

  try {
    const findDeclaraciones = await DeclaracionesFiscales.paginate(
      query,
      options
    );

    findDeclaraciones.docs.forEach((declaracion) => {
      if (declaracion.adjunto.archivo) {
        declaracion.adjunto.archivo = buildFileUri(
          `despachos/${declaracion.despacho}/declaraciones-fiscales/${declaracion.adjunto.archivo}`
        );
      }
    });
    res.status(200).json({ declaraciones: findDeclaraciones });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const creteDeclaracionFiscal = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, tipo, creadoPor, estatus = "Pendiente" } = req.body;

  const { archivo } = req?.files || null;

  const tipos = ["Anual", "Mensual"];

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }

  if (!tipo) {
    return res.status(400).json({ message: "El tipo es requerido" });
  }

  if (!tipos.includes(tipo)) {
    return res.status(400).json({ message: "El tipo es inválido" });
  }

  try {
    const adjunto = {
      nombre: "",
      archivo: "",
    };

    console.log(archivo);
    if (archivo) {
      const fileUp = await uploadFile(
        archivo,
        `despachos/${despacho}/declaraciones-fiscales`
      );
      adjunto.nombre = fileUp.originalName;
      adjunto.archivo = fileUp.fileName;
    }

    const newDeclaraciones = {
      despacho,
      nombre,
      tipo,
      creadoPor,
      adjunto,
      estatus,
    };

    const declaraciones = await DeclaracionesFiscales.create(newDeclaraciones);

    res.status(201).json({ declaraciones });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const deleteDeclaracionFiscal = async (req, res) => {
  const { id } = req.params;
  try {
    const declaracion = await DeclaracionesFiscales.findByIdAndDelete(id);
    if (declaracion.adjunto.archivo) {
      deleteFileS3(
        `despachos/${declaracion.despacho}/declaraciones-fiscales/${declaracion.adjunto.archivo}`
      );
    }
    res.status(200).json({ message: "Declaración eliminada" });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getDeclaracionFiscalById = async (req, res) => {
  const { id } = req.params;
  try {
    const declaracion = await DeclaracionesFiscales.findById(id);
    if (declaracion.adjunto.archivo) {
      declaracion.adjunto.archivo = buildFileUri(
        `despachos/${declaracion.despacho}/declaraciones-fiscales/${declaracion.adjunto.archivo}`
      );
    }
    res.status(200).json({ declaracion });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateDeclaracionFiscal = async (req, res) => {
  const { id } = req.params;
  const { nombre, tipo, creadoPor, estatus = "Pendiente" } = req.body;
  const { archivo } = req?.files ?? null;
  const tipos = ["Anual", "Mensual"];
  if (!nombre) {
    return res.status(400).json({ message: "El nombre es requerido" });
  }
  if (!tipo) {
    return res.status(400).json({ message: "El tipo es requerido" });
  }
  if (!tipos.includes(tipo)) {
    return res.status(400).json({ message: "El tipo es inválido" });
  }
  try {
    if (archivo) {
      const declaracion = await DeclaracionesFiscales.findById(id);
      if (declaracion.adjunto.archivo) {
        deleteFileS3(
          `despachos/${declaracion.despacho}/declaraciones-fiscales/${declaracion.adjunto.archivo}`
        );
      }

      const adjunto = {
        nombre: "",
        archivo: "",
      };

      const fileUp = await uploadFile(
        archivo,
        `despachos/${declaracion.despacho}/declaraciones-fiscales`
      );
      adjunto.nombre = fileUp.originalName;
      adjunto.archivo = fileUp.fileName;

      declaracion.adjunto = adjunto;
      await declaracion.save();
    }

    const newDeclaraciones = {
      nombre,
      tipo,
      creadoPor,
      estatus,
    };
    const declaraciones = await DeclaracionesFiscales.findByIdAndUpdate(
      id,
      newDeclaraciones,
      { new: true }
    );
    res.status(201).json({ declaraciones });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const declaracionFiscalPDF = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search, year } = req.query;

  try {
    const options = {
      page,
      limit: 20,
      sort: { fecha: -1 },
      populate: [
        {
          path: "creadoPor",
          select: "nombre apellidoPaterno apellidoMaterno",
        },
      ],
    };

    if (!despacho) {
      return res.status(400).json({ message: "El despacho es requerido" });
    }

    const query = {
      despacho,
    };

    if (estatus) {
      query.estatus = estatus;
    }

    if (search) {
      // query.nombre = { $regex: search, $options: 'i' };
      query.$or = [{ nombre: { $regex: search, $options: "i" } }];
    }

    if (year) {
      const start = new Date(`${year}-01-01T00:00:00.000Z`);
      const end = new Date(`${year}-12-31T23:59:59.999Z`);
      query.fecha = { $gte: start, $lte: end };
    }

    const findDeclaraciones = await DeclaracionesFiscales.paginate(
      query,
      options
    );

    findDeclaraciones.docs.forEach((declaracion) => {
      if (declaracion.adjunto.archivo) {
        declaracion.adjunto.archivo = buildFileUri(
          `despachos/${declaracion.despacho}/declaraciones-fiscales/${declaracion.adjunto.archivo}`
        );
      }
    });

    res.status(200).json({ declaraciones: findDeclaraciones });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

module.exports = {
  getDeclaracionFiscal,
  creteDeclaracionFiscal,
  deleteDeclaracionFiscal,
  getDeclaracionFiscalById,
  declaracionFiscalPDF,
  updateDeclaracionFiscal,
};
