const moment = require('moment-timezone');
const ExpedientesAgendaUsuarios = require('../models/ExpedientesAgendaUsuarios');
const DespachoModel = require('../models/Despachos');
const { EventoHTML } = require('../Mail/EventoHTML.js');
const { sendMail } = require('../config/mail.js');
const RenovarHTMl = require('../Mail/RenovacionHTML');
const DESPACHO_APP = process.env.DESPACHO_APP;
const ConfiguracionesModel = require('../models/Configuraciones.js');
const UsuarioModel = require('../models/Usuarios.js');
const { getVentasXFechas } = require('./Cortes.Distribuidores.Controller.js');
const CorteModel = require('../models/CortesDistribuidores.js');
const VentaModel = require('../models/Ventas.js');
const FirmaDocumentosModel = require('../models/firma_documento.js');
const FirmaDocumentoMail = require('../Mail/FirmaDocumentoHTML.js');
const AvisosAgenda = async (req, res) => {
  try {
    const fechaActualMexico = moment();

    // Cambiar para agregar 5 minutos en lugar de 1
    const fechaCincoMinutosDespues = moment().add(5, 'minutes');

    const resultados = await ExpedientesAgendaUsuarios.find({
      estatus: 'Pendiente',
      fechaRecordatorio: {
        $gte: fechaActualMexico,
        $lt: fechaCincoMinutosDespues
      }
    }).populate('usuario', 'email').populate('agenda').populate('prospecto', 'correo').populate('cliente', 'correo');

    for (const expediente of resultados) {
      const { usuario, agenda, prospecto, cliente } = expediente;

      let email = null;

      if (usuario) {
        email = usuario.email;
      }

      if (prospecto && prospecto.correo) {
        email = prospecto.correo;
      }

      if (cliente && cliente.correo) {
        email = cliente.correo;
      }

      if (!email) {
        continue;
      }

      const { title, descripcion, fecha } = agenda;

      // Crear el HTML del evento
      const html = EventoHTML({
        title,
        description: descripcion,
        fecha: moment(fecha).format('DD/MM/YYYY'),
        hora: moment(fecha).format('HH:mm'),
        url: `${DESPACHO_APP}/agenda`
      });

      // Enviar el correo electrónico
      sendMail(html, 'Recordatorio de evento', email);

      // Actualizar el estatus del expediente (descomentarlo si necesario)
      // await ExpedientesAgendaUsuarios.findByIdAndUpdate(expediente._id, { estatus: 'Enviado' });
    }

    // Convertir 'ahora' y 'fechaLimite' a String para una visualización correcta en la respuesta
    res.status(200).json({
      data: resultados,
      ahora: fechaActualMexico.format(),
      fechaLimite: fechaCincoMinutosDespues.format()
    });
  } catch (error) {
    console.error('Error al realizar la búsqueda:', error);
    res.status(500).json({ message: 'Error al realizar la búsqueda' });
  }
};

const AvisoSuscripcion = async (req, res) => {
  try {
    const tomorrow = moment().add(1, 'day').startOf('day').toDate();

    const despachos = await DespachoModel.find({
      'contadorExp.vigencia': {
        $gte: tomorrow,
        $lt: moment(tomorrow).endOf('day').toDate()
      }
    });

    const html = RenovarHTMl();

    despachos.forEach(despacho => {
      sendMail(html, 'Renueva tu suscripción', despacho.correo);
    });

    return res.status(200).json({
      data: despachos,
      fecha: tomorrow
    });
  } catch (error) {
    console.error('Error al realizar la búsqueda:', error);
    res.status(500).json({ message: 'Error al realizar la búsqueda' });
  }
};

const ReinicioContadorDiarioChatGpt3 = async (req, res) => {
  try {
    const now = new Date();

    const findConfiguraciones = await ConfiguracionesModel.findOne({});

    if (!findConfiguraciones) {
      return res.status(404).json({ message: 'No se encontraron las configuraciones' });
    }

    const { limiteDiarioGPT } = findConfiguraciones;

    if (!limiteDiarioGPT) {
      return res.status(404).json({ message: 'No se encontró el límite diario de GPT' });
    }

    const result = await DespachoModel.updateMany(
      { estatus: 'Activo' },
      {
        $set: {
          'contadorChatGPT.diario.limite': limiteDiarioGPT,
          'contadorChatGPT.diario.contador': 0,
          'contadorChatGPT.diario.fechaUltimoReinicio': now
        }
      }
    );

    return res.status(200).json({
      data: result,
      fecha: now
    });
  } catch (error) {
    console.error('Error al actualizar los despachos:', error);
    res.status(500).json({ message: 'Error al actualizar los despachos' });
  }
};

const DespachoSuscripcionReinicio = async (req, res) => {
  try {
    const today = moment().startOf('day').toDate();

    const despachos = await DespachoModel.find({
      'contadorExp.vigencia': {
        $lt: today
      },
      'contadorExp.paquete': { $ne: null }
    });

    despachos.forEach(despacho => {
      despacho.contadorExp.paquete = null;
      despacho.contadorExp.limite = 0;
      despacho.save();
    });

    return res.status(200).json({
      data: despachos,
      fecha: today
    });
  } catch (error) {
    console.error('Error al realizar la búsqueda:', error);
    res.status(500).json({ message: 'Error al realizar la búsqueda' });
  }
};

const createCorteDistribuidor = async (req, res) => {
  try {
    const findDistribuidores = await UsuarioModel.find({ estatus: 'Activo', 'tipoUsuarios.distribuidor': { $ne: null, $exists: true } }).select('_id');

    const resultados = await Promise.all(
      findDistribuidores.map(async findDistribuidor => {
        const distribuidor = findDistribuidor._id;
        const ventas = await getVentasXFechas({ distribuidor });

        if (ventas === null) {
          return null;
        }

        const { ventasTotales, totalImporte, findVentas, importeSinComision } = ventas;

        const now = moment().tz('America/Mexico_City');

        const ventaIds = findVentas.map(venta => venta._id);

        // Fecha de inicio: mes anterior, día 26 a las 00:00 horas
        const inicio = moment().tz('America/Mexico_City').subtract(1, 'month').date(26).startOf('day');

        // Fecha de fin: día 25 del mes actual a las 23:59 horas
        const fin = moment().tz('America/Mexico_City').date(25).endOf('day');

        const objCorte = {
          distribuidor,
          fecha: {
            inicio,
            fin,
            corte: now
          },
          totalVentas: ventasTotales,
          importePago: totalImporte,
          importeSinComision,
          ventas: ventaIds,
          comprobante: {
            factura: null,
            xml: null,
            pago: null,
            fechaPago: null
          },
          estatus: 'Pendiente de revisión'
        };

        const corte = new CorteModel(objCorte);

        const newCorte = await corte.save();

        await VentaModel.updateMany({ _id: { $in: ventaIds } }, { 'distribuidor.corte': newCorte._id });

        return newCorte;
      })
    );

    return res.status(201).json({ corte: resultados, message: 'Corte creado exitosamente' });
  } catch (error) {
    console.log('error:', error); // Depuración
    res.status(400).json({ message: error.message });
  }
};

const RecordatorioFirmaDocumento = async (req, res) => {
  try {
    // Obtén los documentos que requieren un recordatorio según la fecha de expiración y el último envío
    const find = await FirmaDocumentosModel.find({
      estatus: 'En proceso',
      recordatorio: { $ne: null }, // Excluye documentos con recordatorio null
      $and: [
        {
          $or: [
            { fechaExpiracion: null },
            { fechaExpiracion: { $gt: new Date() } }
          ]
        },
        {
          $or: [
            { 'usuarios.estatus': 'Sin firmar' },
            { 'clientes.estatus': 'Sin firmar' },
            { 'prospectos.estatus': 'Sin firmar' }
          ]
        }
      ]
    })
      .populate({
        path: 'usuarios.usuario',
        select: 'nombre apellidoPaterno apellidoMaterno email foto'
      })
      .populate({
        path: 'clientes.cliente',
        select: 'nombre correo foto'
      })
      .populate({
        path: 'prospectos.prospecto',
        select: 'nombre correo'
      })
      .populate({
        path: 'creadoPor',
        select: 'nombre apellidoPaterno apellidoMaterno email'
      });

    const today = new Date();

    for (const documento of find) {
      const nombreEnviado = `${documento.creadoPor?.nombre || ''} ${documento.creadoPor?.apellidoPaterno || ''} ${documento.creadoPor?.apellidoMaterno || ''}`;
      const diasRestantes = calcularDiasRestantes(documento.fechaExpiracion);
      const nombreDocumento = `${documento.nombreDocumento || ''}.pdf`;
      const diasDesdeUltimoEnvio = documento.ultimaFechaEnvio
        ? Math.ceil((today - new Date(documento.ultimaFechaEnvio)) / (1000 * 60 * 60 * 24))
        : null;

      // Verificar si es el momento adecuado para enviar un recordatorio
      let enviarRecordatorio = false;
      switch (documento.recordatorio) {
        case 'Día':
          enviarRecordatorio = diasDesdeUltimoEnvio === null || diasDesdeUltimoEnvio >= 1;
          break;
        case 'Tres días':
          enviarRecordatorio = diasDesdeUltimoEnvio === null || diasDesdeUltimoEnvio >= 3;
          break;
        case 'Semana':
          enviarRecordatorio = diasDesdeUltimoEnvio === null || diasDesdeUltimoEnvio >= 7;
          break;
      }

      if (enviarRecordatorio) {
        // Enviar correos a usuarios, clientes y prospectos asociados
        for (const usuario of documento.usuarios) {
          if (debeEnviarRecordatorio(diasRestantes, documento.recordatorio)) {
            const nombreRecibido = `${usuario.usuario?.nombre || ''} ${usuario.usuario?.apellidoPaterno || ''} ${usuario.usuario?.apellidoMaterno || ''}`;
            const rutaFirma = `${DESPACHO_APP}/firma/${documento?.id}/${usuario.usuario?._id}/u`;
            const html = FirmaDocumentoMail({ nombreEnviado, nombreRecibido, nombreDocumento, rutaFirma });
            sendMail(html, `${documento.creadoPor?.nombre || ''} te ha invitado a firmar un documento`, usuario.usuario?.email);
          }
        }

        for (const cliente of documento.clientes) {
          if (debeEnviarRecordatorio(diasRestantes, documento.recordatorio)) {
            const nombreRecibido = `${cliente?.cliente?.nombre || ''}`;
            const rutaFirma = `${DESPACHO_APP}/firma/${documento?.id}/${cliente?.cliente?._id}/c`;
            const html = FirmaDocumentoMail({ nombreEnviado, nombreRecibido, nombreDocumento, rutaFirma });
            sendMail(html, `${documento.creadoPor?.nombre || ''} te ha invitado a firmar un documento`, cliente?.cliente?.correo);
          }
        }

        for (const prospecto of documento.prospectos) {
          if (debeEnviarRecordatorio(diasRestantes, documento.recordatorio)) {
            const nombreRecibido = `${prospecto?.prospecto?.nombre || ''}`;
            const rutaFirma = `${DESPACHO_APP}/firma/${documento?.id}/${prospecto?.prospecto?._id}/p`;
            const html = FirmaDocumentoMail({ nombreEnviado, nombreRecibido, nombreDocumento, rutaFirma });
            sendMail(html, `${documento.creadoPor?.nombre || ''} te ha invitado a firmar un documento`, prospecto?.prospecto?.correo);
          }
        }

        // Actualizar última fecha de envío del documento
        documento.ultimaFechaEnvio = today;
        await documento.save();
      }
    }

    return res.status(200).json({ message: 'Correos enviados de firma de documentos', find });
  } catch (error) {
    console.log('error:', error); // Depuración
    res.status(400).json({ message: error.message });
  }
};

// Helper para verificar días restantes y tipo de recordatorio
const debeEnviarRecordatorio = (diasRestantes, recordatorio) => {
  return (
    (recordatorio === 'Día' && diasRestantes <= 1) ||
    (recordatorio === 'Tres días' && diasRestantes <= 3) ||
    (recordatorio === 'Semana' && diasRestantes <= 7)
  );
};

const calcularDiasRestantes = (fechaExpiracion) => {
  const hoy = new Date();
  const diferenciaTiempo = fechaExpiracion - hoy;
  return Math.ceil(diferenciaTiempo / (1000 * 60 * 60 * 24));
};

module.exports = {
  AvisosAgenda,
  AvisoSuscripcion,
  ReinicioContadorDiarioChatGpt3,
  DespachoSuscripcionReinicio,
  createCorteDistribuidor,
  RecordatorioFirmaDocumento
};
