/* eslint-disable array-callback-return */
const { default: mongoose } = require('mongoose');
const CorteModel = require('../models/CortesDistribuidores.js');
const VentaModel = require('../models/Ventas.js');
const moment = require('moment-timezone');
const { formatMoney, formatDateTime, formatDate } = require('../config/functionsGlobal.js');
const { buildFileUri, uploadFile } = require('../config/s3.js');
const UsuarioModel = require('../models/Usuarios.js');
const cuentaBancariaDistribuidorModel = require('../models/CuentaBancaria.js');

const getCortesDistribuidores = async (req, res) => {
  const { distribuidor } = req.params;
  const { page = 1, fechaInicio, fechaFinal } = req.query;

  const options = {
    page,
    limit: 10,
    sort: { 'fecha.corte': -1 }
    // populate: {
    //   path: 'ventas',
    //   populate: {
    //     path: 'producto'
    //   }
    // }
  };

  if (!distribuidor) {
    return res.status(400).json({ message: 'El distribuidor es requerido' });
  }

  const query = {
    distribuidor
  };

  if (fechaInicio && fechaFinal) {
    query['fecha.corte'] = {
      $gte: moment(fechaInicio).startOf('day').toDate(),
      $lte: moment(fechaFinal).endOf('day').toDate()
    };
  }

  try {
    const [findCortes, findDistri] = await Promise.all([
      CorteModel.paginate(query, options),
      UsuarioModel.findById(distribuidor).select('licencia')
    ]);

    findCortes.docs = findCortes.docs.map(corte => {
      return {
        ...corte._doc,
        comision: findDistri?.licencia?.comision + '%',

        fecha: {
          corte: formatDateTime(corte.fecha.corte),
          inicio: formatDate(corte.fecha.inicio),
          fin: formatDate(corte.fecha.fin)
        },
        importePago: formatMoney(corte.importePago),
        importeSinComision: formatMoney(corte.importeSinComision),

        comprobante: {
          xml: corte.comprobante.xml ? buildFileUri(`cortes/${corte.comprobante.xml}`) : null,
          factura: corte.comprobante.factura ? buildFileUri(`cortes/${corte.comprobante.factura}`) : null,
          pago: corte.comprobante.pago ? buildFileUri(`cortes/${corte.comprobante.pago}`) : null,
          fechaPago: corte.comprobante.fechaPago ? formatDateTime(corte.comprobante.fechaPago) : null
        },
        ventas: corte.ventas.map(venta => {
          return {
            ...venta._doc,
            fechaVenta: formatDateTime(venta.fechaVenta),
            fecha_pago: venta.fecha_pago ? formatDateTime(venta.fecha_pago) : null,
            importe: formatMoney(venta.importe)

          };
        })
      };
    });

    res.status(200).json({ cortes: findCortes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createCorteDistribuidor = async (req, res) => {
  const { distribuidor, idCorte } = req.params;
  const factura = req?.files?.factura || null;
  const xml = req?.files?.xml || null;

  if (!distribuidor) {
    return res.status(400).json({ message: 'El distribuidor es requerido' });
  }

  if (!factura) {
    return res.status(400).json({ message: 'La factura es requerida' });
  }

  if (!xml) {
    return res.status(400).json({ message: 'El XML es requerido' });
  }

  if (!idCorte) {
    return res.status(400).json({ message: 'El ID del corte es requerido' });
  }

  try {
    // --- Validar que el distribuidor tenga todos los documentos necesarios para realizar el corte ---
    const [findUsuario, findCuentabancariaDis] = await Promise.all([
      UsuarioModel.findById(distribuidor),
      cuentaBancariaDistribuidorModel.findOne({ usuario: distribuidor })
    ]);

    if (!findUsuario) {
      return res.status(404).json({ message: 'Distribuidor no encontrado' });
    }

    if (!findUsuario?.rfc) {
      return res.status(400).json({ message: 'Debes registrar tu RFC para poder realizar el corte', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    if (!findUsuario?.comprobantes?.constancia) {
      return res.status(400).json({ message: 'Debes registrar tu constancia de situación fiscal para poder realizar el corte', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    if (!findUsuario?.comprobantes?.ine?.frontal) {
      return res.status(400).json({ message: 'Debes registrar tu INE frontal para poder realizar el corte', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    if (!findUsuario?.comprobantes?.ine?.trasera) {
      return res.status(400).json({ message: 'Debes registrar tu INE trasera para poder realizar el corte', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    if (!findUsuario?.comprobantes?.domicilio) {
      return res.status(400).json({ message: 'Debes registrar tu comprobante de domicilio para poder realizar el corte', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    if (!findCuentabancariaDis) {
      return res.status(404).json({ message: 'Debes registrar donde se depositará el pago de tus ventas', redirectTo: '/perfil/editar?tipo=retirar' });
    }

    // --- Fin de la validación ---
    const findCorte = await CorteModel.findById(idCorte).lean();

    if (!findCorte) {
      return res.status(404).json({ message: 'Corte no encontrado' });
    }
    // const ventas = await getVentasXFechas({ distribuidor });

    // if (ventas === null) {
    //   return res.status(404).json({ message: 'No se encontraron ventas' });
    // }

    // const { ventasTotales, totalImporte, findVentas, desde, hasta } = ventas;

    // const now = moment().tz('America/Mexico_City');

    const [facturaCreated, xmlCreated] = await Promise.all([
      uploadFile(factura, 'cortes'),
      uploadFile(xml, 'cortes')
    ]);

    // const ventaIds = findCorte.ventas.map(venta => venta._id);

    findCorte.comprobante = {
      factura: facturaCreated.fileName,
      xml: xmlCreated.fileName,
      pago: null,
      fechaPago: null
    };

    findCorte.estatus = 'Pendiente';

    await CorteModel.findByIdAndUpdate(idCorte, findCorte);

    return res.status(201).json({ message: 'Corte creado exitosamente' });
  } catch (error) {
    console.log('error:', error); // Depuración
    res.status(400).json({ message: error.message });
  }
};

const previewDelCorte = async (req, res) => {
  const { distribuidor } = req.params;

  if (!distribuidor) {
    return res.status(400).json({ message: 'El distribuidor es requerido' });
  }

  try {
    const ventas = await getVentasXFechas({ distribuidor });

    if (!ventas) {
      return res.status(404).json({ message: 'No se encontraron ventas' });
    }

    let { ventasTotales, totalImporte, findVentas, desde, hasta } = ventas;
    if (!findVentas) {
      return res.status(404).json({ message: 'No se encontraron ventas' });
    }
    findVentas = findVentas.map(venta => {
      return {
        ...venta,
        fechaVenta: formatDateTime(venta.fechaVenta),
        distribuidor: {
          comision: (venta?.distribuidor?.comision || '0') + '%',
          importe: formatMoney(venta.distribuidor.importe)
        }

      };
    });

    res.status(200).json({ ventasTotales, totalImporte: formatMoney(totalImporte), ventas: findVentas, desde: formatDate(desde), hasta: formatDate(hasta) });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const getVentasXFechas = async ({ distribuidor }) => {
  const query = {
    'distribuidor.corte': null,
    'distribuidor.distribuidor': mongoose.Types.ObjectId(distribuidor),
    estatus: 'Pagado'
  };

  try {
    let total = 0;
    let findVentas = await VentaModel.find(query)
      .populate({ path: 'producto', select: 'nombre imagen' })
      .sort({ fechaVenta: -1 }).lean();

    if (!findVentas?.length || findVentas?.length === 0) {
      return null;
    }

    findVentas = findVentas.map(venta => {
      total += venta.importe;
      return {
        ...venta,
        fechaVenta: venta.fechaVenta,
        // fechaVenta: formatDateTime(venta.fechaVenta),
        producto: {
          ...venta.producto,
          imagen: venta?.producto?.imagen?.includes('http')
            ? venta.producto.imagen
            : buildFileUri(`productos/${venta.producto.imagen}`)
          // precio: formatMoney(venta?.producto?.precio || 0)
        },
        importe: formatMoney(venta.importe)
      };
    });

    const ventasTotales = findVentas?.length || 0;
    const totalImporte = findVentas?.reduce((acc, venta) => acc + venta?.distribuidor?.importe, 0) || 0;
    const desde = findVentas[findVentas.length - 1].fechaVenta || null;
    const hasta = findVentas[0].fechaVenta || null;

    return {
      ventasTotales,
      totalImporte,
      findVentas,
      desde,
      hasta,
      importeSinComision: total
    };
  } catch (error) {
    console.error(error); // Log the error for debugging purposes
    return null;
  }
};

const getCorteById = async (req, res) => {
  const { id } = req.params;

  if (!id) {
    return res.status(400).json({ message: 'El ID es requerido' });
  }

  try {
    const findCorte = await CorteModel.findById(id).populate([{
      path: 'ventas',
      select: 'producto importe distribuidor.comision  distribuidor.importe fechaVenta tipoReferencia',
      populate: {
        path: 'producto',
        select: 'nombre imagen'
      }
    }]).lean();

    if (!findCorte) {
      return res.status(404).json({ message: 'Corte no encontrado' });
    }

    findCorte.ventas = findCorte.ventas.map(venta => {
      return {
        ...venta,
        producto: {
          ...venta.producto,
          imagen: venta?.producto?.imagen?.includes('http')
            ? venta.producto.imagen
            : buildFileUri(`productos/${venta.producto.imagen}`)
        },
        importe: formatMoney(venta.importe),
        distribuidor: {
          comision: (venta?.distribuidor?.comision || '0') + '%',
          importe: formatMoney(venta.distribuidor.importe)
        },
        fechaVenta: formatDateTime(venta.fechaVenta)
      };
    });

    findCorte.fecha = {
      corte: formatDateTime(findCorte.fecha.corte),
      inicio: formatDate(findCorte.fecha.inicio),
      fin: formatDate(findCorte.fecha.fin)
    };

    findCorte.importePago = formatMoney(findCorte.importePago);

    res.status(200).json({ corte: findCorte });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getCortesDistribuidores,
  createCorteDistribuidor,
  previewDelCorte,
  getCorteById,
  getVentasXFechas
};
